<?php

namespace Drupal\cl_preview\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\cl_preview\Service\ComponentDiscoveryService;
use Drupal\cl_preview\Service\ComponentHelperService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Controller for component preview pages.
 */
class ComponentPreviewController extends ControllerBase {

  /**
   * The component discovery service.
   *
   * @var \Drupal\cl_preview\Service\ComponentDiscoveryService
   */
  protected $componentDiscovery;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The component helper service.
   *
   * @var \Drupal\cl_preview\Service\ComponentHelperService
   */
  protected $componentHelper;

  /**
   * Constructs a ComponentPreviewController object.
   */
  public function __construct(
    ComponentDiscoveryService $component_discovery,
    ConfigFactoryInterface $config_factory,
    ComponentHelperService $component_helper
  ) {
    $this->componentDiscovery = $component_discovery;
    $this->configFactory = $config_factory;
    $this->componentHelper = $component_helper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('cl_preview.component_discovery'),
      $container->get('config.factory'),
      $container->get('cl_preview.component_helper')
    );
  }

  /**
   * Get title for component preview page.
   */
  public function getTitle($plugin_id) {
    $definition = $this->componentDiscovery->getComponent($plugin_id);
    return $definition['name'] ?? $plugin_id;
  }

  /**
   * View a single component (admin page with iframe).
   */
  public function view($plugin_id) {
    $definition = $this->componentDiscovery->getComponent($plugin_id);

    if (!$definition) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Build metadata.
    $props = $definition['props']['properties'] ?? [];
    $slots = $definition['slots'] ?? [];

    // Check if preview is an array of variations or a single object.
    $preview_data = $definition['preview'] ?? [];
    $has_multiple_previews = $this->isMultiplePreviewVariations($preview_data);
    $preview_variations = $has_multiple_previews ? $preview_data : [$preview_data];

    $default_props = $this->generateDefaultProps($definition);

    // Check for custom examples in Twig comments first.
    $custom_examples = $this->componentHelper->hasExamplesTag($definition) ? $this->extractExamplesFromTwigForDisplay($definition) : NULL;
    $code_example = $custom_examples ?: $this->generateCodeExample($plugin_id, $default_props, $slots);
    $is_custom_example = !empty($custom_examples);

    // Build iframe URL.
    $iframe_url = \Drupal\Core\Url::fromRoute('cl_preview.iframe', ['plugin_id' => $plugin_id])->toString();

    $config = $this->configFactory->get('cl_preview.settings');
    $hot_reload_enabled = (bool) ($config->get('hot_reload_enabled') ?? TRUE);

    $build = [
      '#attached' => [
        'library' => ['cl_preview/library'],
      ],
    ];

    // Component description.
    if (!empty($definition['description'])) {
      $build['description'] = [
        '#type' => 'item',
        '#markup' => $definition['description'],
      ];
    }

    // Metadata section.
    $status = $definition['status'] ?? 'stable';

    $build['meta'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Provider'),
        $this->t('Status'),
        $this->t('ID'),
      ],
      '#rows' => [
        [
          $definition['provider'] ?? '',
          $status,
          ['data' => ['#markup' => '<code>' . $plugin_id . '</code>']],
        ],
      ],
    ];

    // Preview section.
    $build['preview'] = [
      '#type' => 'details',
      '#title' => $this->t('Preview'),
      '#open' => TRUE,
    ];

    $build['preview']['hot_reload'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Hot Reload'),
      '#return_value' => 1,
      '#id' => 'edit-hot-reload',
      '#name' => 'hot_reload',
      '#attributes' => $hot_reload_enabled ? ['checked' => 'checked'] : [],
    ];

    $build['preview']['refresh'] = [
      '#type' => 'button',
      '#value' => $this->t('Refresh Preview'),
      '#button_type' => 'primary',
      '#attributes' => [
        'onclick' => 'document.querySelector(".cl-preview-iframe").contentWindow.location.reload(); return false;',
      ],
      '#states' => [
        'visible' => [
          ':input[name="hot_reload"]' => ['checked' => FALSE],
        ],
      ],
    ];

    // Add preview variation selector if multiple previews exist.
    if ($has_multiple_previews && count($preview_variations) > 1) {
      $options = [];
      foreach ($preview_variations as $index => $variation) {
        $label = $this->getPreviewVariationLabel($variation, $index);
        $options[$index] = $label;
      }

      $build['preview']['variation_selector'] = [
        '#type' => 'select',
        '#title' => $this->t('Preview Variation'),
        '#options' => $options,
        '#default_value' => 0,
        '#id' => 'edit-preview-variation',
        '#name' => 'preview_variation',
        '#attributes' => [
          'class' => ['preview-variation-selector'],
        ],
        '#description' => $this->t('Select a preview variation to display.'),
      ];
    }

    $build['preview']['iframe_wrapper'] = [
      '#type' => 'fieldset',
    ];

    $build['preview']['iframe_wrapper']['iframe'] = [
      '#type' => 'inline_template',
      '#template' => '<iframe src="{{ url }}" data-component-id="{{ plugin_id }}" class="cl-preview-iframe" frameborder="0"></iframe>',
      '#context' => [
        'url' => $iframe_url,
        'plugin_id' => $plugin_id,
      ],
    ];

    // Props section (read-only display).
    if (!empty($props)) {
      // Check for empty preview values
      $empty_props = [];
      foreach ($props as $prop_name => $prop_definition) {
        $type = $prop_definition['type'] ?? 'string';
        $type_normalized = is_array($type) ? $type[0] : $type;

        if (is_string($type_normalized) && !str_starts_with($type_normalized, 'Drupal\\')) {
          $preview_value = $this->getPreviewValue($prop_name, $prop_definition, $type_normalized, $definition);
          if ($this->isPreviewValueEmpty($preview_value, $prop_definition, $type_normalized)) {
            $empty_props[] = $prop_name;
          }
        }
      }

      // Add warning message if there are empty preview values
      if (!empty($empty_props)) {
        $this->messenger()->addWarning($this->t('Some props have empty preview values: @props. Consider adding preview values to the <em>*.component.yml</em> file for a better preview experience.', [
          '@props' => implode(', ', $empty_props),
        ]));
      }

      $build['props'] = [
        '#type' => 'details',
        '#title' => $this->t('Props (@count)', ['@count' => count($props)]),
        '#open' => TRUE,
        '#attributes' => [
          'class' => ['cl-preview-props-section'],
        ],
      ];

      $rows = [];
      foreach ($props as $prop_name => $prop_definition) {
        $type = $prop_definition['type'] ?? 'string';
        $type_string = is_array($type) ? implode(' | ', $type) : $type;
        $type_normalized = is_array($type) ? $type[0] : $type;

        // Get the preview/default value
        $preview_value = $this->getPreviewValue($prop_name, $prop_definition, $type_normalized, $definition);

        // Format value for display
        if (is_bool($preview_value)) {
          $display_value = $preview_value ? 'true' : 'false';
        }
        elseif (is_array($preview_value) || is_object($preview_value)) {
          $display_value = json_encode($preview_value, JSON_PRETTY_PRINT);
        }
        elseif ($preview_value === '' || $preview_value === NULL) {
          $display_value = $this->t('(empty)');
        }
        else {
          $display_value = $preview_value;
        }

        $rows[] = [
          ['data' => ['#markup' => '<code>' . $prop_name . '</code>']],
          $type_string,
          ['data' => ['#markup' => '<code>' . htmlspecialchars($display_value) . '</code>']],
          $prop_definition['description'] ?? '',
        ];
      }

      $build['props']['table'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('Name'),
          $this->t('Type'),
          $this->t('Preview Value'),
          $this->t('Description'),
        ],
        '#rows' => $rows,
        '#attributes' => [
          'class' => ['cl-preview-props-table'],
        ],
      ];
    }

    // Slots section.
    if (!empty($slots)) {
      $build['slots'] = [
        '#type' => 'details',
        '#title' => $this->t('Slots (@count)', ['@count' => count($slots)]),
        '#open' => TRUE,
      ];

      $rows = [];
      foreach ($slots as $slot_name => $slot_definition) {
        $rows[] = [
          ['data' => ['#markup' => '<code>' . $slot_name . '</code>']],
          $slot_definition['title'] ?? $slot_name,
          $slot_definition['description'] ?? '',
        ];
      }

      $build['slots']['table'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('Name'),
          $this->t('Title'),
          $this->t('Description'),
        ],
        '#rows' => $rows,
      ];
    }

    // Code example section.
    $build['code'] = [
      '#type' => 'details',
      '#title' => $this->t('Code Example'),
      '#open' => TRUE,
    ];

    // Add informational message about code example source.
    if ($is_custom_example) {
      $build['code']['note'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('These examples are defined using the <code>@examples</code> tag in the component\'s Twig template. <a href="https://git.drupalcode.org/project/cl_preview/-/blob/1.x/README.md#custom-code-examples-with-examples-tag" target="_blank">Learn more</a>') . '</p>',
        '#weight' => -1,
      ];
      $build['code']['examples_fieldset'] = [
        '#type' => 'fieldset',
        '#weight' => 0,
      ];
      $build['code']['examples_fieldset']['content'] = [
        '#type' => 'inline_template',
        '#template' => '{{ code|raw }}',
        '#context' => [
          'code' => $code_example,
        ],
      ];
    }
    else {
      $build['code']['note'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('Auto-generated from the component\'s props and preview values. To add custom examples, use the <code>@examples</code> tag in your Twig template. <a href="https://git.drupalcode.org/project/cl_preview/-/blob/1.x/README.md#custom-code-examples-with-examples-tag" target="_blank">Learn more</a>') . '</p>',
        '#weight' => -1,
      ];
      $build['code']['example'] = [
        '#type' => 'inline_template',
        '#template' => '<pre><code>{{ code }}</code></pre>',
        '#context' => [
          'code' => $code_example,
        ],
      ];
    }

    return $build;
  }

  /**
   * Render component in iframe (uses preview theme with full assets).
   */
  public function iframe($plugin_id) {
    $definition = $this->componentDiscovery->getComponent($plugin_id);

    if (!$definition) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Get the preview rendering mode from settings.
    $config = $this->configFactory->get('cl_preview.settings');
    $render_mode = $config->get('preview_render_mode') ?: 'examples';

    // Check if we should try to use @examples snippets.
    if ($render_mode === 'examples') {
      // Try to get the first example from the Twig file's @examples tag.
      $example_snippet = $this->componentHelper->getFirstExampleSnippet($definition);

      if ($example_snippet) {
        // Render using the example snippet.
        return $this->renderSnippet($example_snippet);
      }
    }

    // Get preview variation index from query parameter.
    $variation_index = \Drupal::request()->query->get('variation', 0);

    // Fallback or 'preview_values' mode: Generate default props from schema.
    $default_props = $this->generateDefaultProps($definition, (int) $variation_index);

    // Generate default content for slots.
    $slots = $definition['slots'] ?? [];
    $slot_content = [];
    foreach ($slots as $slot_name => $slot_definition) {
      $slot_content[$slot_name] = $this->generateSlotDefaultContent($slot_name, $slot_definition);
    }

    // Build the component render array.
    $build = [
      '#type' => 'component',
      '#component' => $plugin_id,
      '#props' => $default_props,
      '#slots' => $slot_content,
      '#cache' => [
        'max-age' => 0,
      ],
    ];

    // Attach the CL Preview iframe CSS.
    $build['#attached']['library'][] = 'cl_preview/iframe';

    // Attach the preview theme's global libraries.
    $preview_theme = $this->configFactory->get('cl_preview.settings')->get('preview_theme');
    if ($preview_theme) {
      $theme_handler = \Drupal::service('theme_handler');
      $theme_info = $theme_handler->listInfo()[$preview_theme] ?? NULL;

      if ($theme_info && !empty($theme_info->info['libraries'])) {
        foreach ($theme_info->info['libraries'] as $library) {
          $build['#attached']['library'][] = $library;
        }
      }
    }

    // Return the component render array directly.
    // This preserves the automatic library attachments from SDC
    // (component CSS/JS files are auto-discovered and attached).
    return $build;
  }

  /**
   * Check for component file updates (for hot reload).
   */
  public function checkUpdates($plugin_id) {
    $definition = $this->componentDiscovery->getComponent($plugin_id);

    if (!$definition) {
      return new JsonResponse(['error' => 'Component not found'], 404);
    }

    // Get component directory path
    $path = $definition['path'] ?? '';
    if (empty($path)) {
      return new JsonResponse(['error' => 'Component path not found'], 404);
    }

    // Get the component machine name (without provider prefix)
    $machineName = $definition['machineName'] ?? '';
    if (empty($machineName)) {
      return new JsonResponse(['error' => 'Component machine name not found'], 404);
    }

    // Find all relevant files in component directory
    $files = [
      $path . '/' . $machineName . '.twig',
      $path . '/' . $machineName . '.css',
      $path . '/' . $machineName . '.js',
      $path . '/' . $machineName . '.component.yml',
    ];

    // Get the most recent modification time and check component.yml
    $lastModified = 0;
    $componentYmlMtime = 0;
    foreach ($files as $file) {
      if (file_exists($file)) {
        $mtime = filemtime($file);
        if ($mtime > $lastModified) {
          $lastModified = $mtime;
        }
        // Track component.yml modification time
        if (str_ends_with($file, '.component.yml')) {
          $componentYmlMtime = $mtime;
        }
      }
    }

    // Clear component plugin cache on every request when component.yml exists
    // This ensures changes to component.yml are reflected immediately
    if ($componentYmlMtime > 0) {
      \Drupal::service('plugin.manager.sdc')->clearCachedDefinitions();
    }

    return new JsonResponse([
      'last_modified' => $lastModified,
      'component_id' => $plugin_id,
    ]);
  }

  /**
   * Get the preview value for a prop (preview > default).
   *
   * @param string $prop_name
   *   The prop name.
   * @param array $prop
   *   The prop definition.
   * @param string $type
   *   The normalized prop type.
   * @param array $definition
   *   The full component definition.
   *
   * @return mixed
   *   The preview value.
   */
  protected function getPreviewValue(string $prop_name, array $prop, string $type, array $definition) {
    // Priority 1: Check top-level preview section for this prop
    if (isset($definition['preview'][$prop_name])) {
      return $definition['preview'][$prop_name];
    }

    // Priority 2: Use default value if defined
    if (isset($prop['default'])) {
      $default = $prop['default'];

      // For booleans, YAML parser converts false to empty string - treat as false
      if ($type === 'boolean' && $default === '') {
        return FALSE;
      }

      return $default;
    }

    // Priority 3: Fallback to type-based empty value
    return match($type) {
      'string' => '',
      'integer', 'number' => 0,
      'boolean' => FALSE,
      'array' => [],
      'object' => [],
      default => '',
    };
  }

  /**
   * Check if preview value is empty/missing and should be warned about.
   *
   * @param mixed $value
   *   The preview value to check.
   * @param array $prop
   *   The full prop definition.
   * @param string $type
   *   The normalized prop type.
   *
   * @return bool
   *   TRUE if value is considered empty and should be highlighted as warning.
   */
  protected function isPreviewValueEmpty($value, array $prop, string $type) {
    // Boolean false is always valid
    if ($type === 'boolean') {
      return FALSE;
    }

    // Check if prop type allows null
    $prop_type = $prop['type'] ?? 'string';
    $allows_null = is_array($prop_type) && in_array('null', $prop_type, TRUE);

    // If null is allowed, empty values are valid
    if ($allows_null && ($value === '' || $value === NULL)) {
      return FALSE;
    }

    // For arrays/objects, empty array [] or object {} are VALID values
    // Only warn if the value is truly unset (null/not provided)
    if ($type === 'array' || $type === 'object') {
      return $value === NULL;
    }

    // For other types, check if empty string or null
    return $value === '' || $value === NULL;
  }

  /**
   * Generate default prop values from component schema.
   *
   * @param array $definition
   *   The component definition.
   * @param int $variation_index
   *   The preview variation index (for multiple preview support).
   *
   * @return array
   *   The default props.
   */
  protected function generateDefaultProps(array $definition, int $variation_index = 0) {
    $props = $definition['props']['properties'] ?? [];
    $defaults = [];

    // Get the appropriate preview data based on variation index.
    $preview_data = $definition['preview'] ?? [];
    $is_multiple = $this->isMultiplePreviewVariations($preview_data);

    // If multiple previews, use the specific variation.
    if ($is_multiple && isset($preview_data[$variation_index])) {
      $active_preview = $preview_data[$variation_index];
    }
    elseif ($is_multiple && !empty($preview_data)) {
      // Fallback to first variation if index is out of bounds.
      $active_preview = $preview_data[0];
    }
    else {
      // Single preview or no preview.
      $active_preview = $preview_data;
    }

    // Create a modified definition with the active preview.
    $modified_definition = $definition;
    $modified_definition['preview'] = $active_preview;

    foreach ($props as $key => $prop) {
      $type = $prop['type'] ?? 'string';

      // Handle array types.
      if (is_array($type)) {
        $type = $type[0] ?? 'string';
      }

      // Skip Drupal-specific types.
      if (is_string($type) && str_starts_with($type, 'Drupal\\')) {
        continue;
      }

      // Get preview value (preview > default).
      $defaults[$key] = $this->getPreviewValue($key, $prop, $type, $modified_definition);
    }

    return $defaults;
  }

  /**
   * Generate default content for a slot.
   *
   * @param string $slot_name
   *   The slot name.
   * @param array $slot_definition
   *   The slot definition.
   *
   * @return string
   *   Default slot content.
   */
  protected function generateSlotDefaultContent(string $slot_name, array $slot_definition) {
    $slot_lower = strtolower($slot_name);

    // Content slot - use lorem ipsum.
    if (str_contains($slot_lower, 'content') || $slot_lower === 'default') {
      return 'Lorem ipsum dolor sit amet, consectetur adipiscing elit.';
    }

    // Title/heading slot.
    if (str_contains($slot_lower, 'title') || str_contains($slot_lower, 'heading')) {
      return 'Example Title';
    }

    // Description slot.
    if (str_contains($slot_lower, 'description')) {
      return 'This is an example description for the component.';
    }

    // Footer slot.
    if (str_contains($slot_lower, 'footer')) {
      return '<small>Example footer content</small>';
    }

    // Header slot.
    if (str_contains($slot_lower, 'header')) {
      return '<strong>Example Header</strong>';
    }

    // Generic slot - use slot title or name.
    return $slot_definition['title'] ?? ucwords(str_replace(['_', '-'], ' ', $slot_name));
  }

  /**
   * Generate Twig code example.
   */
  protected function generateCodeExample($plugin_id, array $props, array $slots) {
    $code = "{% include '$plugin_id' with {\n";
    $code .= $this->formatPropsForCode($props, 1);
    $code .= "} only %}";

    return $code;
  }

  /**
   * Format props recursively for code example.
   *
   * @param array $props
   *   Props to format.
   * @param int $indent_level
   *   Current indentation level.
   *
   * @return string
   *   Formatted props string.
   */
  protected function formatPropsForCode(array $props, int $indent_level = 0) {
    $code = '';
    $indent = str_repeat('  ', $indent_level);

    foreach ($props as $key => $value) {
      if (is_string($value)) {
        $code .= "{$indent}{$key}: '{$value}',\n";
      }
      elseif (is_bool($value)) {
        $code .= "{$indent}{$key}: " . ($value ? 'true' : 'false') . ",\n";
      }
      elseif (is_array($value)) {
        // Check if it's an associative array (object) or empty/numeric array.
        if (empty($value)) {
          $code .= "{$indent}{$key}: [],\n";
        }
        elseif (array_keys($value) !== range(0, count($value) - 1)) {
          // Associative array (object).
          $code .= "{$indent}{$key}: {\n";
          $code .= $this->formatPropsForCode($value, $indent_level + 1);
          $code .= "{$indent}},\n";
        }
        else {
          // Numeric array.
          $code .= "{$indent}{$key}: [],\n";
        }
      }
      elseif (is_numeric($value)) {
        $code .= "{$indent}{$key}: {$value},\n";
      }
      else {
        $code .= "{$indent}{$key}: {$value},\n";
      }
    }

    return $code;
  }


  /**
   * Render a Twig snippet as inline template.
   *
   * @param string $snippet
   *   The Twig snippet to render.
   *
   * @return array
   *   Render array with the snippet.
   */
  protected function renderSnippet(string $snippet): array {
    // Kill page cache to ensure fresh content.
    \Drupal::service('page_cache_kill_switch')->trigger();

    $build = [
      '#type' => 'inline_template',
      '#template' => $snippet,
      '#cache' => [
        'max-age' => 0,
        'contexts' => [],
        'tags' => [],
      ],
    ];

    // Attach the CL Preview iframe CSS.
    $build['#attached']['library'][] = 'cl_preview/iframe';

    // Attach the preview theme's global libraries.
    $preview_theme = $this->configFactory->get('cl_preview.settings')->get('preview_theme');
    if ($preview_theme) {
      $theme_handler = \Drupal::service('theme_handler');
      $theme_info = $theme_handler->listInfo()[$preview_theme] ?? NULL;

      if ($theme_info && !empty($theme_info->info['libraries'])) {
        foreach ($theme_info->info['libraries'] as $library) {
          $build['#attached']['library'][] = $library;
        }
      }
    }

    return $build;
  }

  /**
   * Extract examples content for display in the code example section.
   *
   * Looks for @examples tag in Twig comments and formats the content for display.
   *
   * @param array $definition
   *   The component definition.
   *
   * @return string|null
   *   The extracted examples HTML, or NULL if not found.
   */
  protected function extractExamplesFromTwigForDisplay(array $definition): ?string {
    // Get the path to the component directory and template filename.
    $component_path = $definition['path'] ?? NULL;
    $template_filename = $definition['template'] ?? NULL;

    if (!$component_path || !$template_filename) {
      return NULL;
    }

    // Construct the full path to the Twig template.
    $template_path = $component_path . '/' . $template_filename;

    if (!file_exists($template_path) || !is_file($template_path)) {
      return NULL;
    }

    // Read the Twig file.
    $twig_file = file_get_contents($template_path);
    if ($twig_file === FALSE) {
      return NULL;
    }

    // Pattern to match Twig comments containing @examples tag.
    // Matches: {# ... @examples ... content ... #}
    $pattern = '/\{#\s*@examples\s*(.*?)\s*#\}/s';

    if (preg_match($pattern, $twig_file, $matches)) {
      $examples_content = trim($matches[1]);

      if (!empty($examples_content)) {
        // Return raw content wrapped in pre/code tags.
        return '<pre><code>' . htmlspecialchars($examples_content) . '</code></pre>';
      }
    }

    return NULL;
  }

  /**
   * Check if preview data contains multiple variations (array of objects).
   *
   * @param mixed $preview_data
   *   The preview data from component definition.
   *
   * @return bool
   *   TRUE if preview contains multiple variations, FALSE otherwise.
   */
  protected function isMultiplePreviewVariations($preview_data): bool {
    if (!is_array($preview_data) || empty($preview_data)) {
      return FALSE;
    }

    // Check if this is a numeric array (list) of preview variations.
    // If all keys are numeric and sequential, it's a list of variations.
    $keys = array_keys($preview_data);
    $is_numeric_array = array_keys($keys) === $keys;

    // Also check if the first element is an array (object).
    $first_element = reset($preview_data);
    $first_is_array = is_array($first_element);

    return $is_numeric_array && $first_is_array;
  }

  /**
   * Get a human-readable label for a preview variation.
   *
   * @param array $variation
   *   The preview variation data.
   * @param int $index
   *   The index of the variation.
   *
   * @return string
   *   The label for the variation.
   */
  protected function getPreviewVariationLabel(array $variation, int $index): string {
    // Try to generate a meaningful label from variation properties.
    // Common properties that might indicate the variant type.
    $label_props = ['tag', 'variant', 'type', 'style', 'size', 'color'];

    foreach ($label_props as $prop) {
      if (isset($variation[$prop]) && !empty($variation[$prop])) {
        return ucfirst((string) $variation[$prop]);
      }
    }

    // If no meaningful property found, check for title/heading.
    if (isset($variation['title']) && !empty($variation['title'])) {
      $title = (string) $variation['title'];
      return strlen($title) > 30 ? substr($title, 0, 30) . '...' : $title;
    }

    // Fallback to "Variation N".
    return $this->t('Variation @num', ['@num' => $index + 1]);
  }

}
