<?php

namespace Drupal\cl_preview\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\cl_preview\Service\ComponentDiscoveryService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for selecting components to pin (Library page).
 */
class LibraryForm extends FormBase {

  /**
   * The component discovery service.
   *
   * @var \Drupal\cl_preview\Service\ComponentDiscoveryService
   */
  protected $componentDiscovery;

  /**
   * The theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a LibraryForm object.
   */
  public function __construct(
    ComponentDiscoveryService $component_discovery,
    ThemeHandlerInterface $theme_handler,
    ModuleHandlerInterface $module_handler,
    ConfigFactoryInterface $config_factory
  ) {
    $this->componentDiscovery = $component_discovery;
    $this->themeHandler = $theme_handler;
    $this->moduleHandler = $module_handler;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('cl_preview.component_discovery'),
      $container->get('theme_handler'),
      $container->get('module_handler'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cl_preview_library_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('cl_preview.settings');
    $scan_themes = $config->get('scan_themes') ?: [];
    $scan_modules = $config->get('scan_modules') ?: [];

    // Check if no themes or modules are configured.
    if (empty($scan_themes) && empty($scan_modules)) {
      $settings_url = Url::fromRoute('cl_preview.settings');

      $form['empty_message'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        'message' => [
          '#markup' => $this->t('No themes or modules are configured for scanning. Please <a href="@settings_url">configure the module settings</a> to select which themes and modules should be scanned for components.', [
            '@settings_url' => $settings_url->toString(),
          ]),
        ],
      ];

      return $form;
    }

    $grouped_components = $this->componentDiscovery->getGroupedComponents();

    // Check if no components were found.
    if (empty($grouped_components)) {
      $form['no_components'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        'message' => [
          '#markup' => $this->t('No components found in the configured themes and modules. Make sure the selected themes/modules contain a <code>components/</code> directory with <code>*.component.yml</code> files.'),
        ],
      ];

      return $form;
    }

    // Get currently pinned components.
    $pinned_components = $config->get('pinned_components') ?: [];

    $form['#attached']['library'][] = 'cl_preview/library';

    $form['intro'] = [
      '#type' => 'html_tag',
      '#tag' => 'p',
      '#value' => $this->t('Select components to display on the All-Components page. You can view individual component previews by clicking their names.'),
    ];

    // Group components by provider first.
    $by_provider = [];
    foreach ($grouped_components as $group => $components) {
      foreach ($components as $plugin_id => $definition) {
        $provider = $definition['provider'];
        $by_provider[$provider][$group][$plugin_id] = $definition;
      }
    }

    // Sort providers alphabetically.
    ksort($by_provider);

    // Build form with grouped checkboxes.
    $form['components'] = [
      '#type' => 'container',
      '#tree' => TRUE,
    ];

    foreach ($by_provider as $provider => $groups) {
      $provider_key = 'provider_' . preg_replace('/[^a-z0-9_]+/', '_', strtolower($provider));
      $provider_label = $this->getProviderLabel($provider);

      foreach ($groups as $group => $components) {
        $group_key = $provider_key . '_' . preg_replace('/[^a-z0-9_]+/', '_', strtolower($group));

        $form['components'][$group_key] = [
          '#type' => 'details',
          '#title' => $provider_label . ' (' . $group . ')',
          '#open' => TRUE,
        ];

        foreach ($components as $plugin_id => $definition) {
          $component_name = $definition['name'] ?? $plugin_id;

          $form['components'][$group_key][$plugin_id] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['cl-preview-library-checkbox-item']],
          ];

          $form['components'][$group_key][$plugin_id]['checkbox'] = [
            '#type' => 'checkbox',
            '#title' => '',
            '#default_value' => in_array($plugin_id, $pinned_components),
            '#attributes' => ['class' => ['cl-preview-library-checkbox']],
          ];

          $form['components'][$group_key][$plugin_id]['link'] = [
            '#type' => 'link',
            '#title' => $component_name,
            '#url' => Url::fromRoute('cl_preview.component', ['plugin_id' => $plugin_id]),
            '#attributes' => ['class' => ['cl-preview-library-component-link']],
          ];
        }
      }
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save Selection'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $components = $form_state->getValue('components', []);
    $pinned_components = [];

    // Collect all checked components.
    foreach ($components as $group_key => $group_items) {
      if (is_array($group_items)) {
        foreach ($group_items as $plugin_id => $item) {
          if (isset($item['checkbox']) && $item['checkbox']) {
            $pinned_components[] = $plugin_id;
          }
        }
      }
    }

    // Save to config.
    $this->configFactory->getEditable('cl_preview.settings')
      ->set('pinned_components', $pinned_components)
      ->save();

    $this->messenger()->addStatus($this->t('Component selection saved. @count components selected.', [
      '@count' => count($pinned_components),
    ]));
  }

  /**
   * Get the human-readable label for a provider (theme or module).
   *
   * @param string $provider
   *   The machine name of the provider.
   *
   * @return string
   *   The human-readable label.
   */
  protected function getProviderLabel($provider) {
    // Check if it's a theme.
    $themes = $this->themeHandler->listInfo();
    if (isset($themes[$provider])) {
      return $this->t('Theme: @name', ['@name' => $themes[$provider]->info['name']]);
    }

    // Check if it's a module.
    $module_list = $this->moduleHandler->getModuleList();
    if (isset($module_list[$provider])) {
      $module_info = \Drupal::service('extension.list.module')->getExtensionInfo($provider);
      return $this->t('Module: @name', ['@name' => $module_info['name'] ?? $provider]);
    }

    // Fallback to provider name.
    return $provider;
  }

}
