<?php

namespace Drupal\cl_preview\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for pinning/unpinning components.
 */
class PinComponentForm extends FormBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a PinComponentForm object.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cl_preview_pin_component_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $plugin_id = NULL, $is_pinned = FALSE) {
    $form_state->set('plugin_id', $plugin_id);

    $form['#prefix'] = '<div id="pin-form-wrapper-' . $plugin_id . '">';
    $form['#suffix'] = '</div>';

    $form['pinned'] = [
      '#type' => 'checkbox',
      '#title' => '',
      '#default_value' => $is_pinned,
      '#ajax' => [
        'callback' => '::ajaxTogglePinCallback',
        'wrapper' => 'pin-form-wrapper-' . $plugin_id,
        'effect' => 'fade',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Form submission handled via AJAX.
  }

  /**
   * AJAX callback for toggling pin status.
   */
  public function ajaxTogglePinCallback(array &$form, FormStateInterface $form_state) {
    $plugin_id = $form_state->get('plugin_id');
    $is_pinned = $form_state->getValue('pinned');

    // Get current pinned components.
    $config = $this->configFactory->getEditable('cl_preview.settings');
    $pinned_components = $config->get('pinned_components') ?: [];

    // Toggle pin status.
    if ($is_pinned) {
      // Add to pinned if not already there.
      if (!in_array($plugin_id, $pinned_components)) {
        $pinned_components[] = $plugin_id;
      }
    }
    else {
      // Remove from pinned.
      $pinned_components = array_values(array_filter($pinned_components, function($id) use ($plugin_id) {
        return $id !== $plugin_id;
      }));
    }

    // Save updated pinned components.
    $config->set('pinned_components', $pinned_components)->save();

    // Return the form with updated state.
    return $form;
  }

}
