<?php

namespace Drupal\cl_preview\Form;

use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure CL Preview settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a SettingsForm object.
   */
  public function __construct(ThemeHandlerInterface $theme_handler, ModuleHandlerInterface $module_handler) {
    $this->themeHandler = $theme_handler;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('theme_handler'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cl_preview_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['cl_preview.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('cl_preview.settings');

    // Get list of installed themes.
    $themes = $this->themeHandler->listInfo();
    $theme_options = [];
    foreach ($themes as $name => $theme) {
      $theme_options[$name] = $theme->info['name'];
    }

    $form['preview_theme'] = [
      '#type' => 'select',
      '#title' => $this->t('Preview theme'),
      '#description' => $this->t('Select the theme to use for rendering component previews. Leave empty to use the current default theme.'),
      '#options' => ['' => $this->t('- Default theme -')] + $theme_options,
      '#default_value' => $config->get('preview_theme'),
    ];

    $form['scan_themes_wrapper'] = [
      '#type' => 'details',
      '#title' => $this->t('Scan themes for components'),
      '#description' => $this->t('Select which themes to scan for SDC components.'),
    ];
    
    $form['scan_themes_wrapper']['scan_themes'] = [
      '#type' => 'checkboxes',
      '#options' => $theme_options,
      '#default_value' => $config->get('scan_themes') ?: [],
    ];
    
    // Get list of modules that might have components.
    $modules = $this->moduleHandler->getModuleList();
    $module_options = [];
    foreach ($modules as $name => $extension) {
      $module_info = \Drupal::service('extension.list.module')->getExtensionInfo($name);
      $module_options[$name] = $module_info['name'] ?? $name;
    }
    
    $form['scan_modules_wrapper'] = [
      '#type' => 'details',
      '#title' => $this->t('Scan modules for components'),
      '#description' => $this->t('Select which modules to scan for SDC components.'),
    ];

    $form['scan_modules_wrapper']['scan_modules'] = [
      '#type' => 'checkboxes',
      '#options' => $module_options,
      '#default_value' => $config->get('scan_modules') ?: [],
    ];

    $form['group_by'] = [
      '#type' => 'radios',
      '#title' => $this->t('Group components by'),
      '#options' => [
        'path' => $this->t('Directory structure (Atoms, Molecules, Organisms)'),
        'provider' => $this->t('Theme/Module name'),
        'status' => $this->t('Status (Stable, Experimental, etc.)'),
      ],
      '#default_value' => $config->get('group_by') ?: 'path',
    ];

    $form['preview_render_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Preview rendering mode'),
      '#description' => $this->t('Choose how component previews should be rendered in the iframe.'),
      '#options' => [
        'preview_values' => $this->t('<strong>Preview values</strong> - Use preview values defined in the component.yml file (or defaults if not specified)'),
        'examples' => $this->t('<strong>Example snippets</strong> - Use custom Twig examples defined with @examples tag in the component template (falls back to preview values if no @examples tag exists)'),
      ],
      '#default_value' => $config->get('preview_render_mode') ?: 'examples',
    ];

    $form['hot_reload_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable hot reload by default'),
      '#description' => $this->t('When enabled, component previews will automatically reload when files change (checks every second). This can be toggled per-component in the preview page.'),
      '#default_value' => $config->get('hot_reload_enabled') ?? TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Get the checkbox values (they return arrays with keys as values, 0 for unchecked).
    $scan_themes = $form_state->getValue('scan_themes');
    $scan_modules = $form_state->getValue('scan_modules');

    // Filter out unchecked items (value of 0) and reindex.
    $scan_themes = is_array($scan_themes) ? array_values(array_filter($scan_themes)) : [];
    $scan_modules = is_array($scan_modules) ? array_values(array_filter($scan_modules)) : [];

    $this->config('cl_preview.settings')
      ->set('preview_theme', $form_state->getValue('preview_theme'))
      ->set('scan_themes', $scan_themes)
      ->set('scan_modules', $scan_modules)
      ->set('group_by', $form_state->getValue('group_by'))
      ->set('preview_render_mode', $form_state->getValue('preview_render_mode'))
      ->set('hot_reload_enabled', $form_state->getValue('hot_reload_enabled'))
      ->save();

    parent::submitForm($form, $form_state);

    // Redirect to the component library.
    $form_state->setRedirect('cl_preview.library');
  }

}
