<?php

namespace Drupal\cl_preview\Service;

use Drupal\cl_editorial\NoThemeComponentManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\sdc\ComponentPluginManager;

/**
 * Service for discovering SDC components.
 */
class ComponentDiscoveryService {

  /**
   * The component manager.
   *
   * @var \Drupal\sdc\ComponentPluginManager
   */
  protected $componentManager;

  /**
   * The no-theme component manager.
   *
   * @var \Drupal\cl_editorial\NoThemeComponentManager
   */
  protected $noThemeComponentManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a ComponentDiscoveryService object.
   */
  public function __construct(
    ComponentPluginManager $component_manager,
    NoThemeComponentManager $no_theme_component_manager,
    ConfigFactoryInterface $config_factory
  ) {
    $this->componentManager = $component_manager;
    $this->noThemeComponentManager = $no_theme_component_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * Get all components based on configuration.
   *
   * @return array
   *   Array of component definitions grouped by category.
   */
  public function getGroupedComponents() {
    $config = $this->configFactory->get('cl_preview.settings');
    $scan_themes = $config->get('scan_themes') ?: [];
    $scan_modules = $config->get('scan_modules') ?: [];
    $group_by = $config->get('group_by') ?: 'path';

    // Get all components.
    $all_components = $this->componentManager->getDefinitions();

    // Filter by configured themes/modules.
    $filtered_components = [];
    foreach ($all_components as $plugin_id => $definition) {
      $provider = $definition['provider'];

      if (in_array($provider, $scan_themes) || in_array($provider, $scan_modules)) {
        $filtered_components[$plugin_id] = $definition;
      }
    }

    // Group components.
    $grouped = [];
    foreach ($filtered_components as $plugin_id => $definition) {
      $group = $this->getGroupKey($definition, $group_by);
      $grouped[$group][$plugin_id] = $definition;
    }

    // Sort groups alphabetically.
    ksort($grouped);

    return $grouped;
  }

  /**
   * Get the group key for a component based on grouping strategy.
   *
   * @param array $definition
   *   The component definition.
   * @param string $group_by
   *   The grouping strategy.
   *
   * @return string
   *   The group key.
   */
  protected function getGroupKey(array $definition, $group_by) {
    switch ($group_by) {
      case 'provider':
        return $definition['provider'];

      case 'status':
        return $definition['status'] ?? 'stable';

      case 'path':
      default:
        // Extract category from path (atoms, molecules, organisms, etc.).
        $path = $definition['path'] ?? '';

        // Try to find category in path.
        if (preg_match('#/(atoms?|molecules?|organisms?|templates?|pages?)/#i', $path, $matches)) {
          return ucfirst(rtrim($matches[1], 's')) . 's';
        }

        // Try to extract from component ID.
        $id = $definition['id'] ?? '';
        if (preg_match('/^([amo])-/', $id, $matches)) {
          $prefix_map = [
            'a' => 'Atoms',
            'm' => 'Molecules',
            'o' => 'Organisms',
          ];
          return $prefix_map[$matches[1]] ?? 'Components';
        }

        return 'Components';
    }
  }

  /**
   * Get a single component definition.
   *
   * @param string $plugin_id
   *   The component plugin ID.
   *
   * @return array|null
   *   The component definition or NULL if not found.
   */
  public function getComponent($plugin_id) {
    $definitions = $this->componentManager->getDefinitions();
    return $definitions[$plugin_id] ?? NULL;
  }

}
