<?php

namespace Drupal\cl_preview\Service;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for shared component helper methods.
 */
class ComponentHelperService {

  use StringTranslationTrait;

  /**
   * The theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a ComponentHelperService object.
   */
  public function __construct(
    ThemeHandlerInterface $theme_handler,
    ModuleHandlerInterface $module_handler
  ) {
    $this->themeHandler = $theme_handler;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Get the human-readable label for a provider (theme or module).
   *
   * @param string $provider
   *   The machine name of the provider.
   *
   * @return string
   *   The human-readable label.
   */
  public function getProviderLabel($provider) {
    // Check if it's a theme.
    $themes = $this->themeHandler->listInfo();
    if (isset($themes[$provider])) {
      return $this->t('Theme: @name', ['@name' => $themes[$provider]->info['name']]);
    }

    // Check if it's a module.
    $module_list = $this->moduleHandler->getModuleList();
    if (isset($module_list[$provider])) {
      $module_info = \Drupal::service('extension.list.module')->getExtensionInfo($provider);
      return $this->t('Module: @name', ['@name' => $module_info['name'] ?? $provider]);
    }

    // Fallback to provider name.
    return $provider;
  }

  /**
   * Get first example snippet from @examples tag in Twig template.
   *
   * @param array $definition
   *   The component definition.
   *
   * @return string|null
   *   The snippet or NULL if not found.
   */
  public function getFirstExampleSnippet(array $definition): ?string {
    $component_path = $definition['path'] ?? NULL;
    $template_filename = $definition['template'] ?? NULL;

    if (!$component_path || !$template_filename) {
      return NULL;
    }

    $template_path = $component_path . '/' . $template_filename;

    if (!file_exists($template_path) || !is_file($template_path)) {
      return NULL;
    }

    $twig_file = file_get_contents($template_path);
    if ($twig_file === FALSE) {
      return NULL;
    }

    $pattern = '/\{#\s*@examples\s*(.*?)\s*#\}/s';

    if (preg_match($pattern, $twig_file, $matches)) {
      $examples_content = trim($matches[1]);

      if (!empty($examples_content)) {
        // Extract the first complete Twig snippet.
        if (preg_match('/\{%\s*include\s+.*?%\}/s', $examples_content, $snippet_match)) {
          return trim($snippet_match[0]);
        }

        if (preg_match('/\{%.*?%\}/s', $examples_content, $snippet_match)) {
          return trim($snippet_match[0]);
        }

        if (preg_match('/\{\{.*?\}\}/s', $examples_content, $snippet_match)) {
          return trim($snippet_match[0]);
        }
      }
    }

    return NULL;
  }

  /**
   * Generate snippet from component props.
   *
   * @param string $plugin_id
   *   The component plugin ID.
   * @param array $definition
   *   The component definition.
   *
   * @return string
   *   The generated snippet.
   */
  public function generateSnippetFromProps($plugin_id, array $definition) {
    $props = $definition['props']['properties'] ?? [];
    $default_props = [];

    foreach ($props as $key => $prop) {
      $type = $prop['type'] ?? 'string';

      if (is_array($type)) {
        $type = $type[0] ?? 'string';
      }

      if (is_string($type) && str_starts_with($type, 'Drupal\\')) {
        continue;
      }

      // Get preview or default value.
      if (isset($definition['preview'][$key])) {
        $default_props[$key] = $definition['preview'][$key];
      }
      elseif (isset($prop['default'])) {
        $default_props[$key] = $prop['default'];
      }
    }

    $code = "{% include '$plugin_id' with {\n";
    foreach ($default_props as $key => $value) {
      if (is_string($value)) {
        $code .= "  {$key}: '{$value}',\n";
      }
      elseif (is_bool($value)) {
        $code .= "  {$key}: " . ($value ? 'true' : 'false') . ",\n";
      }
      elseif (is_numeric($value)) {
        $code .= "  {$key}: {$value},\n";
      }
    }
    $code .= "} only %}";

    return $code;
  }

  /**
   * Check if there's an @examples tag in the Twig template.
   *
   * @param array $definition
   *   The component definition.
   *
   * @return bool
   *   TRUE if @examples tag exists, FALSE otherwise.
   */
  public function hasExamplesTag(array $definition): bool {
    $component_path = $definition['path'] ?? NULL;
    $template_filename = $definition['template'] ?? NULL;

    if (!$component_path || !$template_filename) {
      return FALSE;
    }

    $template_path = $component_path . '/' . $template_filename;

    if (!file_exists($template_path) || !is_file($template_path)) {
      return FALSE;
    }

    $twig_file = file_get_contents($template_path);
    if ($twig_file === FALSE) {
      return FALSE;
    }

    $pattern = '/\{#\s*@examples\s*(.*?)\s*#\}/s';
    return preg_match($pattern, $twig_file) === 1;
  }

  /**
   * Group components by provider.
   *
   * @param array $grouped_components
   *   Components already grouped by another criterion.
   *
   * @return array
   *   Components grouped by provider, then by original group.
   */
  public function groupComponentsByProvider(array $grouped_components): array {
    $by_provider = [];
    foreach ($grouped_components as $group => $components) {
      foreach ($components as $plugin_id => $definition) {
        $provider = $definition['provider'];
        $by_provider[$provider][$group][$plugin_id] = $definition;
      }
    }

    // Sort providers alphabetically.
    ksort($by_provider);

    return $by_provider;
  }

  /**
   * Generate a sanitized key from a string.
   *
   * @param string $string
   *   The string to convert.
   *
   * @return string
   *   Sanitized key containing only lowercase alphanumeric and underscores.
   */
  public function generateKey($string): string {
    return preg_replace('/[^a-z0-9_]+/', '_', strtolower($string));
  }

}
