<?php

namespace Drupal\clean_filename\Plugin\Filter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;

/**
 * Provides a filter to enable clean filename behavior for CKEditor uploads.
 *
 * @Filter(
 *   id = "filter_clean_filename",
 *   title = @Translation("Clean Filename for CKEditor"),
 *   description = @Translation("Enables clean filename behavior for images uploaded through CKEditor in this text format."),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_TRANSFORM_IRREVERSIBLE,
 *   settings = {
 *     "clean_filename_enabled" = FALSE,
 *   },
 *   weight = -10
 * )
 */
class FilterCleanFilename extends FilterBase {

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['clean_filename_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Clean Filename for CKEditor uploads'),
      '#description' => $this->t('When enabled, images uploaded through CKEditor in this text format will use clean filenames. This setting controls CKEditor uploads independently of field configurations.'),
      '#default_value' => $this->settings['clean_filename_enabled'] ?? FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode) {
    // This filter doesn't actually modify text content.
    // Its purpose is to provide configuration for clean filename behavior
    // during CKEditor uploads. The actual file processing happens in
    // the CleanFilenameManager service when files are uploaded.
    return new FilterProcessResult($text);
  }

  /**
   * {@inheritdoc}
   */
  public function tips($long = FALSE) {
    if ($long) {
      return $this->t('This filter enables clean filename behavior for images uploaded through CKEditor. It does not modify text content, but affects how uploaded files are named.');
    }
    else {
      return $this->t('Enables clean filename behavior for CKEditor uploads.');
    }
  }

  /**
   * Checks if clean filename is enabled for this filter.
   *
   * @return bool
   *   TRUE if clean filename is enabled, FALSE otherwise.
   */
  public function isCleanFilenameEnabled() {
    return !empty($this->settings['clean_filename_enabled']);
  }

}
