<?php

namespace Drupal\clean_filename\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Clean Filename settings.
 */
class CleanFilenameSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a CleanFilenameSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typedConfigManager, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($config_factory, $typedConfigManager);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['clean_filename.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'clean_filename_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('clean_filename.settings');

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Configure clean filename settings. When enabled for specific fields, new files will keep clean filenames while existing ones get suffixes.') . '</p>',
    ];

    $form['global_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Global Settings'),
    ];

    $form['global_settings']['enable_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable detailed logging'),
      '#description' => $this->t('Log file renaming operations for debugging purposes.'),
      '#default_value' => $config->get('enable_logging') ?? FALSE,
    ];

    $form['global_settings']['max_rename_attempts'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum rename attempts'),
      '#description' => $this->t('Maximum number of attempts to find an available filename suffix.'),
      '#default_value' => $config->get('max_rename_attempts') ?? 100,
      '#min' => 1,
      '#max' => 1000,
    ];

    $form['field_configurations'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Field Configurations'),
    ];

    $form['field_configurations']['info'] = [
      '#markup' => '<p>' . $this->t('To enable clean filename for specific fields, edit the field configuration in the field settings.') . '</p>',
    ];

    // Get all field configurations with clean filename enabled.
    $enabled_fields = $this->getEnabledFields();

    if (!empty($enabled_fields)) {
      $items = [];
      foreach ($enabled_fields as $field_config) {
        $items[] = $this->t('@entity_type > @bundle > @field', [
          '@entity_type' => $field_config->getTargetEntityTypeId(),
          '@bundle' => $field_config->getTargetBundle(),
          '@field' => $field_config->getName(),
        ]);
      }

      $form['field_configurations']['enabled_fields'] = [
        '#theme' => 'item_list',
        '#items' => $items,
        '#title' => $this->t('Fields with clean filename enabled:'),
        '#empty' => $this->t('No fields have clean filename enabled.'),
      ];
    }
    else {
      $form['field_configurations']['no_fields'] = [
        '#markup' => '<p><em>' . $this->t('No fields currently have clean filename enabled.') . '</em></p>',
      ];
    }

    // Text format configurations.
    $form['text_format_configurations'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Text Format Configurations'),
    ];

    $form['text_format_configurations']['info'] = [
      '#markup' => '<p>' . $this->t('To enable clean filename for CKEditor uploads in specific text formats, go to <strong>Configuration > Content authoring > Text formats and editors</strong>, configure a text format, and enable the "Clean Filename for CKEditor" filter. CKEditor uploads are controlled independently by the text format filter configuration.') . '</p>',
    ];

    // Get all text format configurations with clean filename filter enabled.
    $enabled_text_formats = $this->getEnabledTextFormats();

    if (!empty($enabled_text_formats)) {
      $items = [];
      foreach ($enabled_text_formats as $text_format) {
        $items[] = $this->t('@text_format (Clean Filename filter enabled)', [
          '@text_format' => $text_format->label(),
        ]);
      }

      $form['text_format_configurations']['enabled_text_formats'] = [
        '#theme' => 'item_list',
        '#items' => $items,
        '#title' => $this->t('Text formats with Clean Filename filter enabled:'),
        '#empty' => $this->t('No text formats have the Clean Filename filter enabled.'),
      ];
    }
    else {
      $form['text_format_configurations']['no_text_formats'] = [
        '#markup' => '<p><em>' . $this->t('No text formats currently have the Clean Filename filter enabled for CKEditor uploads.') . '</em></p>',
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('clean_filename.settings');

    $config->set('enable_logging', $form_state->getValue('enable_logging'));
    $config->set('max_rename_attempts', $form_state->getValue('max_rename_attempts'));

    $config->save();

    $this->messenger()->addStatus($this->t('Clean Filename settings have been saved.'));

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets all field configurations with clean filename enabled.
   *
   * @return \Drupal\field\Entity\FieldConfig[]
   *   Array of field configurations.
   */
  protected function getEnabledFields() {
    $field_config_storage = $this->entityTypeManager->getStorage('field_config');
    $field_configs = $field_config_storage->loadByProperties([]);

    $enabled_fields = [];

    foreach ($field_configs as $field_config) {
      $field_type = $field_config->getType();
      if (in_array($field_type, ['file', 'image', 'media'])) {
        if ($field_config->getThirdPartySetting('clean_filename', 'clean_filename_enabled', FALSE)) {
          $enabled_fields[] = $field_config;
        }
      }
    }

    return $enabled_fields;
  }

  /**
   * Gets all text format configurations with clean filename filter enabled.
   *
   * @return \Drupal\filter\Entity\FilterFormat[]
   *   Array of text format configurations.
   */
  protected function getEnabledTextFormats() {
    $text_format_storage = $this->entityTypeManager->getStorage('filter_format');
    $text_formats = $text_format_storage->loadMultiple();

    $enabled_text_formats = [];

    foreach ($text_formats as $text_format) {
      // Check if the clean filename filter is enabled and configured.
      $filters = $text_format->filters();
      if ($filters->has('filter_clean_filename')) {
        $filter = $filters->get('filter_clean_filename');
        if ($filter->status) {
          // Filter is enabled, check its settings.
          $settings = $filter->getConfiguration();
          if (!empty($settings['settings']['clean_filename_enabled'])) {
            $enabled_text_formats[] = $text_format;
          }
        }
      }
    }

    return $enabled_text_formats;
  }

}
