<?php

namespace Drupal\clean_filename\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\file\Entity\File;
use Drupal\field\Entity\FieldConfig;
use Drupal\filter\Entity\FilterFormat;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service for managing clean filename behavior.
 */
class CleanFilenameManager {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The file system service.
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The logger factory.
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * The route match service.
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * The request stack.
   */
  protected RequestStack $requestStack;

  /**
   * The private temp store factory.
   */
  protected PrivateTempStoreFactory $tempStoreFactory;

  /**
   * Constructs a CleanFilenameManager object.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    FileSystemInterface $file_system,
    LoggerChannelFactoryInterface $logger_factory,
    Connection $database,
    RouteMatchInterface $route_match,
    RequestStack $request_stack,
    PrivateTempStoreFactory $temp_store_factory,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->fileSystem = $file_system;
    $this->loggerFactory = $logger_factory;
    $this->database = $database;
    $this->routeMatch = $route_match;
    $this->requestStack = $request_stack;
    $this->tempStoreFactory = $temp_store_factory;
  }

  /**
   * Handles file upload with clean filename behavior.
   *
   * @param \Drupal\file\Entity\File $new_file
   *   The new file entity being uploaded.
   */
  public function handleFileUpload(File $new_file): void {
    if (!$new_file || !$new_file->getFileUri()) {
      $this->loggerFactory->get('clean_filename')->warning('Invalid file entity passed to handleFileUpload');
      return;
    }

    $uri = $new_file->getFileUri();
    $filename = $this->fileSystem->basename($uri);
    $directory = $this->fileSystem->dirname($uri);

    $this->loggerFactory->get('clean_filename')->info('Processing file: @filename in @directory', [
      '@filename' => $filename,
      '@directory' => $directory,
    ]);

    // Detect if Drupal has already renamed this file (e.g., "file_0.jpg").
    $original_filename = $this->getOriginalFilename($filename);

    // Only proceed if the filename was actually changed by Drupal.
    if ($filename === $original_filename) {
      $this->loggerFactory->get('clean_filename')->info('No conflict detected - filename unchanged');
      return;
    }

    $this->loggerFactory->get('clean_filename')->info('Original filename detected: @original (from @current)', [
      '@original' => $original_filename,
      '@current' => $filename,
    ]);

    // Find the file that conflicts with the original name.
    $conflicting_file = $this->getConflictingFile($original_filename, $directory, $new_file->id());

    if ($conflicting_file) {
      try {
        // Get the suffix Drupal used for the new file.
        $drupal_used_suffix = $this->extractSuffixFromFilename($filename);

        // Find next available suffix (excluding the one Drupal just used).
        $next_suffix = $this->getNextAvailableSuffix($original_filename, $directory, $drupal_used_suffix);

        // Rename the conflicting file.
        $this->renameSingleFile($conflicting_file, $original_filename, $next_suffix, $directory);

        // Give the new file the clean name.
        $this->updateNewFileToOriginalName($new_file, $original_filename, $directory);

        $this->loggerFactory->get('clean_filename')->info('Successfully processed clean filename for @original', [
          '@original' => $original_filename,
        ]);
      }
      catch (\Exception $e) {
        $this->loggerFactory->get('clean_filename')->error('Error during clean filename processing: @error', [
          '@error' => $e->getMessage(),
        ]);
      }
    }
  }

  /**
   * Checks if clean filename should be applied for the current context.
   */
  public function shouldApplyCleanFilenameForCurrentContext(): bool {
    // Debug mode override.
    if (Settings::get('clean_filename_debug', FALSE)) {
      return TRUE;
    }

    // Try to determine the specific context.
    $context = $this->determineUploadContext();

    if ($context['type'] === 'field' && $context['field_config']) {
      // Regular field upload - check field configuration.
      return $context['field_config']->getThirdPartySetting('clean_filename', 'clean_filename_enabled', FALSE);
    }

    if ($context['type'] === 'ckeditor' && $context['text_format']) {
      // CKEditor upload - check only text format filter configuration.
      return $this->isCleanFilenameFilterEnabled($context['text_format']);
    }

    // Fallback: check if any field has clean filename enabled (old behavior).
    return $this->hasAnyFieldWithCleanFilenameEnabled();
  }

  /**
   * Determines the upload context (field upload vs CKEditor upload).
   */
  protected function determineUploadContext(): array {
    $context = [
      'type' => 'unknown',
      'field_config' => NULL,
      'text_format' => NULL,
      'target_field' => NULL,
    ];

    // Check if we're in a CKEditor upload context.
    $route_name = $this->routeMatch->getRouteName();

    if ($route_name === 'ckeditor5.upload_image') {
      // This is a CKEditor image upload.
      $context['type'] = 'ckeditor';

      // Get text format from request.
      $request = $this->requestStack->getCurrentRequest();
      $text_format_id = $request->request->get('text_format');

      if ($text_format_id) {
        $context['text_format'] = FilterFormat::load($text_format_id);
      }

      return $context;
    }

    // Check for regular field upload context.
    $field_context = $this->getFieldContextFromTempStore();
    if ($field_context && isset($field_context['field_name'])) {
      $context['type'] = 'field';
      $context['field_config'] = FieldConfig::loadByName(
        $field_context['entity_type'],
        $field_context['bundle'],
        $field_context['field_name']
      );
    }

    return $context;
  }

  /**
   * Gets field context from temporary store.
   *
   * Used for regular field uploads (not CKEditor).
   */
  protected function getFieldContextFromTempStore(): ?array {
    $temp_store = $this->tempStoreFactory->get('clean_filename');

    // Try to get the most recent field context.
    $request = $this->requestStack->getCurrentRequest();
    $form_build_id = $request->request->get('form_build_id');

    if ($form_build_id && $temp_store->get($form_build_id)) {
      return $temp_store->get($form_build_id);
    }

    return NULL;
  }

  /**
   * Checks if the clean filename filter is enabled for a text format.
   */
  protected function isCleanFilenameFilterEnabled(FilterFormat $text_format): bool {
    // Get the filter collection for this text format.
    $filters = $text_format->filters();

    // Check if our clean filename filter is enabled.
    if ($filters->has('filter_clean_filename')) {
      $filter = $filters->get('filter_clean_filename');
      if ($filter->status) {
        // Filter is enabled, check its settings.
        $settings = $filter->getConfiguration();
        return !empty($settings['settings']['clean_filename_enabled']);
      }
    }

    return FALSE;
  }

  /**
   * Checks if any field in the system has clean filename enabled.
   *
   * This is the fallback method for backward compatibility.
   */
  protected function hasAnyFieldWithCleanFilenameEnabled(): bool {
    $field_config_storage = $this->entityTypeManager->getStorage('field_config');
    $field_configs = $field_config_storage->loadByProperties([]);

    foreach ($field_configs as $field_config) {
      if ($field_config instanceof FieldConfig) {
        $field_type = $field_config->getType();
        if (in_array($field_type, ['file', 'image', 'media'])) {
          if ($field_config->getThirdPartySetting('clean_filename', 'clean_filename_enabled', FALSE)) {
            return TRUE;
          }
        }
      }
    }

    return FALSE;
  }

  /**
   * Sets field context for clean filename decisions.
   */
  public function setFieldContext(string $form_build_id, array $field_info): void {
    $temp_store = $this->tempStoreFactory->get('clean_filename');
    $temp_store->set($form_build_id, $field_info);
  }

  /**
   * Finds the single file that conflicts with the new upload.
   */
  protected function getConflictingFile(string $original_filename, string $directory, ?int $exclude_file_id = NULL): ?File {
    $file_storage = $this->entityTypeManager->getStorage('file');

    $query = $file_storage->getQuery();
    $query->condition('uri', $directory . '/' . $original_filename);
    $query->accessCheck(FALSE);

    if ($exclude_file_id) {
      $query->condition('fid', $exclude_file_id, '<>');
    }

    $file_ids = $query->execute();

    if (!empty($file_ids)) {
      $files = $file_storage->loadMultiple($file_ids);
      return reset($files);
    }

    return NULL;
  }

  /**
   * Finds the next available suffix number.
   */
  protected function getNextAvailableSuffix(string $original_filename, string $directory, ?int $exclude_suffix = NULL): int {
    $used_suffixes = $this->getUsedSuffixes($original_filename, $directory);

    // Exclude the suffix Drupal just used.
    if ($exclude_suffix !== NULL) {
      $used_suffixes[] = $exclude_suffix;
    }

    // Find the next available number.
    $next_suffix = 0;
    while (in_array($next_suffix, $used_suffixes)) {
      $next_suffix++;
    }

    return $next_suffix;
  }

  /**
   * Gets all suffix numbers currently in use.
   */
  protected function getUsedSuffixes(string $original_filename, string $directory): array {
    $file_info = pathinfo($original_filename);
    $base_name = $file_info['filename'];
    $extension = isset($file_info['extension']) ? '.' . $file_info['extension'] : '';
    $used_suffixes = [];

    // Check database for managed files.
    $file_storage = $this->entityTypeManager->getStorage('file');
    $query = $file_storage->getQuery();
    $query->condition('uri', $directory . '/', 'STARTS_WITH');
    $query->accessCheck(FALSE);
    $file_ids = $query->execute();

    if ($file_ids) {
      $files = $file_storage->loadMultiple($file_ids);

      foreach ($files as $file) {
        $filename = $file->getFilename();
        $pattern = '/^' . preg_quote($base_name, '/') . '_(\d+)' . preg_quote($extension, '/') . '$/';
        if (preg_match($pattern, $filename, $matches)) {
          $used_suffixes[] = (int) $matches[1];
        }
      }
    }

    // Check physical files on disk (for orphaned files).
    $directory_path = $this->fileSystem->realpath($directory);
    if ($directory_path && is_dir($directory_path)) {
      $pattern = '/^' . preg_quote($base_name, '/') . '_(\d+)' . preg_quote($extension, '/') . '$/';

      $files = scandir($directory_path);
      foreach ($files as $filename) {
        if ($filename === '.' || $filename === '..') {
          continue;
        }

        if (preg_match($pattern, $filename, $matches)) {
          $used_suffixes[] = (int) $matches[1];
        }
      }
    }

    return array_unique($used_suffixes);
  }

  /**
   * Renames a single file with the given suffix.
   */
  protected function renameSingleFile(File $file, string $original_filename, int $suffix, string $directory): void {
    $file_info = pathinfo($original_filename);
    $new_filename = $file_info['filename'] . '_' . $suffix;
    if (isset($file_info['extension'])) {
      $new_filename .= '.' . $file_info['extension'];
    }

    $original_uri = $file->getFileUri();
    $new_uri = $directory . '/' . $new_filename;

    if (!file_exists($original_uri)) {
      $this->loggerFactory->get('clean_filename')->error('Source file does not exist: @uri', [
        '@uri' => $original_uri,
      ]);
      return;
    }

    if ($this->fileSystem->move($original_uri, $new_uri, FileExists::Replace)) {
      // Update database.
      $this->database->update('file_managed')
        ->fields([
          'uri' => $new_uri,
          'filename' => $new_filename,
        ])
        ->condition('fid', $file->id())
        ->execute();

      // Clear cache.
      $this->entityTypeManager->getStorage('file')->resetCache([$file->id()]);

      $this->loggerFactory->get('clean_filename')->info('Renamed conflicting file @old to @new', [
        '@old' => $original_uri,
        '@new' => $new_uri,
      ]);
    }
    else {
      $this->loggerFactory->get('clean_filename')->error('Failed to move file @old to @new', [
        '@old' => $original_uri,
        '@new' => $new_uri,
      ]);
    }

  }

  /**
   * Updates the new file to use the original filename.
   */
  protected function updateNewFileToOriginalName(File $new_file, string $original_filename, string $directory): void {
    $current_uri = $new_file->getFileUri();
    $new_uri = $directory . '/' . $original_filename;

    if (!file_exists($current_uri)) {
      $this->loggerFactory->get('clean_filename')->error('New file source does not exist: @uri', [
        '@uri' => $current_uri,
      ]);
      return;
    }

    if ($this->fileSystem->move($current_uri, $new_uri, FileExists::Replace)) {
      // Update the file entity.
      if ($new_file->isNew()) {
        $new_file->setFileUri($new_uri);
        $new_file->setFilename($original_filename);
      }
      else {
        $this->database->update('file_managed')
          ->fields([
            'uri' => $new_uri,
            'filename' => $original_filename,
          ])
          ->condition('fid', $new_file->id())
          ->execute();

        $this->entityTypeManager->getStorage('file')->resetCache([$new_file->id()]);
      }

      $this->loggerFactory->get('clean_filename')->info('Updated file to clean name: @uri', [
        '@uri' => $new_uri,
      ]);
    }
    else {
      $this->loggerFactory->get('clean_filename')->error('Failed to move new file @old to @new', [
        '@old' => $current_uri,
        '@new' => $new_uri,
      ]);
    }

  }

  /**
   * Detects the original filename from a potentially renamed file.
   */
  protected function getOriginalFilename(string $filename): string {
    if (preg_match('/^(.+)_(\d+)(\.[^.]+)?$/', $filename, $matches)) {
      return $matches[1] . ($matches[3] ?? '');
    }

    return $filename;
  }

  /**
   * Extracts the suffix number from a filename.
   */
  protected function extractSuffixFromFilename(string $filename): ?int {
    if (preg_match('/^(.+)_(\d+)(\.[^.]+)?$/', $filename, $matches)) {
      return (int) $matches[2];
    }

    return NULL;
  }

}
