# Clean Filename

The Clean Filename module ensures that new file uploads always get clean, original filenames by reversing Drupal's default conflict resolution. When you upload a file with the same name as an existing file, the existing file is renamed with a suffix and the new upload keeps the clean name.

## How It Works

**Default Drupal behavior:**
- Existing file: `document.pdf`
- New upload becomes: `document_0.pdf` ❌

**With Clean Filename module:**
- Existing file becomes: `document_1.pdf` (renamed)
- New upload stays: `document.pdf` ✅ (clean name)

### Configuration Logic

The module now uses **granular configuration** instead of global settings:

**For regular file field uploads:**
- Clean filename applies only if the specific field has "Enable Clean Filename" checked

**For CKEditor image uploads:**
- Clean filename applies if the text format has the "Clean Filename for CKEditor" filter enabled and configured
- CKEditor uploads work independently of field configurations

This gives you precise control over where clean filenames are used.

**The process:**
1. Drupal detects naming conflict and creates `filename_0.ext` for new upload
2. Module detects this rename and intervenes
3. Existing `filename.ext` is moved to `filename_1.ext` (next available suffix)
4. New file `filename_0.ext` is renamed to clean `filename.ext`

## Table of Contents

- [Key Benefits](#key-benefits)
- [Requirements](#requirements)
- [Installation](#installation)
- [Configuration](#configuration)
- [Technical Details](#technical-details)
- [API](#api)
- [Testing](#testing)
- [Troubleshooting](#troubleshooting)
- [Maintainers](#maintainers)

## Key Benefits

- **Clean URLs for new content**: Latest uploads always have clean, SEO-friendly filenames
- **Preserved references**: Existing files maintain their URLs and references
- **Per-field control**: Enable only for specific file, image, or media fields
- **Smart conflict resolution**: Handles complex naming scenarios automatically
- **Drupal integration**: Works seamlessly with Drupal's file system
- **Battle-tested**: Comprehensive test coverage and error handling

## Requirements

This module requires the following modules:

- File (Core)
- Field (Core)
- System (Core)

## Installation

Install as you would normally install a contributed Drupal module. For further information, see [Installing Drupal Modules](https://www.drupal.org/docs/extending-drupal/installing-drupal-modules).

### Using Composer (recommended)

```bash
composer require drupal/clean_filename
drush en clean_filename
```

### Manual Installation

1. Download the module from drupal.org
2. Extract to your `modules/` directory
3. Enable the module using Drush or the admin interface:
   ```bash
   drush en clean_filename
   ```

## Configuration

### Global Settings

1. Navigate to **Administration > Configuration > Media > Clean Filename Settings**
2. Configure global options:
   - **Enable detailed logging**: Log file renaming operations for debugging

### Field-Level Configuration

1. Navigate to the field configuration for any file, image, or media field
2. In the field settings, look for **"Clean Filename Settings"** section
3. Check **"Enable Clean Filename"**
4. Save the field configuration

Example path: **Administration > Structure > Content types > [Content Type] > Manage fields > [Field Name] > Edit**

### Text Format Configuration (for CKEditor uploads)

1. Navigate to **Administration > Configuration > Content authoring > Text formats and editors**
2. Click **Configure** for any text format that uses CKEditor 5
3. In the **"Enabled filters"** section, check **"Clean Filename for CKEditor"**
4. Configure the filter by checking **"Enable Clean Filename for CKEditor uploads"**
5. Save the text format configuration

**Important:** CKEditor uploads are controlled independently by the text format filter configuration.

## How It Works

### Default Drupal Behavior
- `document.pdf` (existing file)
- `document_0.pdf` (new upload with same name)

### With Clean Filename Module
- `document_0.pdf` (existing file, renamed)
- `document.pdf` (new upload keeps clean name)

### Step-by-Step Example

**Initial state:** `report.pdf` exists

**Second upload of `report.pdf`:**
1. Drupal creates: `report_0.pdf` (new upload)
2. Module renames: `report.pdf` → `report_1.pdf` (existing)
3. Module renames: `report_0.pdf` → `report.pdf` (new gets clean name)

**Third upload of `report.pdf`:**
1. Drupal creates: `report_0.pdf` (new upload) 
2. Module renames: `report.pdf` → `report_2.pdf` (existing)
3. Module renames: `report_0.pdf` → `report.pdf` (new gets clean name)

**Result:** Latest file is always `report.pdf`, older versions have suffixes (`report_1.pdf`, `report_2.pdf`)

## Technical Details

### Architecture

- **Service-based**: Core logic in `FilePriorityManager` service
- **Hook integration**: Uses `hook_entity_presave()` and `hook_form_FORM_ID_alter()`
- **Field-aware**: Respects individual field configurations
- **Smart sequencing**: Analyzes existing suffixes to prevent conflicts

### Key Components

- `CleanFilenameManager`: Main service handling file operations
- `CleanFilenameSettingsForm`: Administrative configuration form
- Field configuration integration for per-field settings

### Processing Logic

1. **Drupal Upload**: File uploaded as `q.jpeg` but conflict detected with existing `q.jpeg`
2. **Drupal Renames**: New file becomes `q_0.jpeg` (standard Drupal behavior)
3. **Module Detection**: Detects that `q_0.jpeg` was created from original `q.jpeg`
4. **Existing File Rename**: Original `q.jpeg` → `q_1.jpeg` (next available suffix)
5. **New File Cleanup**: `q_0.jpeg` → `q.jpeg` (gets clean name)

**Result**: New uploads always have clean filenames, existing files get suffixes but maintain their references.

## Permissions

- **Administer Clean Filename**: Required to access configuration forms

## Compatibility

- **Drupal Core**: 10.x, 11.x
- **Required modules**: file, field, system
- **Field types**: file, image, media

## Testing

The module includes comprehensive PHPUnit tests covering:

### Running Tests

```bash
# Run all tests
cd modules/custom/clean_filename
../../../vendor/bin/phpunit

# Run specific test types
../../../vendor/bin/phpunit --testsuite unit
../../../vendor/bin/phpunit --testsuite kernel
../../../vendor/bin/phpunit --testsuite functional

# Run with coverage
../../../vendor/bin/phpunit --coverage-html coverage
```

### Test Coverage

- **Unit Tests**: Core service logic, filename detection, sequencing algorithms
- **Kernel Tests**: Drupal integration, service availability, real file operations
- **Functional Tests**: UI integration, form behavior, permissions

### Test Files

- `tests/src/Unit/CleanFilenameManagerTest.php`: Service logic tests
- `tests/src/Kernel/CleanFilenameIntegrationTest.php`: Integration tests
- `tests/src/Functional/CleanFilenameFormTest.php`: UI and form tests

## Development

### Extending the Module

```php
// Get the clean filename manager service
$clean_filename_manager = \Drupal::service('clean_filename.manager');

// Set field context for custom implementations
$clean_filename_manager->setFieldContext($form_build_id, $field_info);
```

### Key Methods

- `handleFileUpload()`: Main processing method
- `getConflictingFile()`: Finds the specific file that conflicts
- `getNextAvailableSuffix()`: Determines next available suffix number
- `renameSingleFile()`: Renames the conflicting file
- `updateNewFileToOriginalName()`: Gives new file the clean name

### Hooks Used

- `hook_form_FORM_ID_alter()`: Adds field configuration options
- `hook_entity_presave()`: Handles file renaming logic
- `hook_field_widget_single_element_form_alter()`: Widget-level integration

## Troubleshooting

### Emergency Debug Mode

Add to `settings.php` for testing:
```php
$settings['clean_filename_debug'] = TRUE;
```

This forces clean filename to work regardless of field configuration.

### Troubleshooting

If the module is not working:

1. **Enable debug mode** in `settings.php`:
   ```php
   $settings['clean_filename_debug'] = TRUE;
   ```

2. **Check logs** at Reports > Recent log messages for `clean_filename` entries

3. **Clear caches**: `drush cr`

4. **Verify field configuration** that clean filename is enabled

**Common Issues:**
- **Files not being renamed**: Check field configuration and permissions
- **"Missing source file" errors**: Check file permissions and disk space
- **Settings not saving**: Ensure user has proper admin permissions
- **Module not running**: Ensure field has clean filename enabled

## API

### Service Usage

```php
// Get the clean filename manager service
$clean_filename_manager = \Drupal::service('clean_filename.manager');

// Set field context for custom implementations
$clean_filename_manager->setFieldContext($form_build_id, $field_info);

// Check if any fields have clean filename enabled
$has_enabled_fields = $clean_filename_manager->hasAnyFieldWithCleanFilenameEnabled();
```

### Hooks

The module integrates with Drupal using:

- `hook_form_FORM_ID_alter()`: Adds field configuration options
- `hook_entity_presave()`: Handles file renaming logic
- `hook_field_widget_single_element_form_alter()`: Widget-level integration

## Maintainers

Current maintainer(s):
 * Neeraj Singh (neerajsingh) - https://www.drupal.org/u/neerajsingh


## Contributing

Bug reports and feature requests should be reported in the [issue queue](https://www.drupal.org/project/issues/clean_filename).