(function ($, Drupal) {
  let player;
  let currentField, inField, titleField, outField;
  let inBtn, gotoInBtn, outBtn, gotoOutBtn, playBtn, playInOutBtn, prevBtn, nextBtn;

  Drupal.behaviors.clipMediaEntityEditForm = {
    attach: function (context, settings) {
      player = new ClipPlayer();

      inField = inField || onceFirst('in-field', '[id=edit-clip-in-ts]', context);
      currentField = currentField || onceFirst('current-field', '[id=edit-clip-current-ts]', context);
      outField = outField || onceFirst('out-field', '[id=edit-clip-out-ts]', context);
      titleField = titleField || onceFirst('title-field', '[id=edit-clip-title]', context);

      inBtn = inBtn || onceFirst('set-in', '#edit-control-buttons [name="set_in"]', context);
      if (inBtn) inBtn.addEventListener('click', function (event) {
        event.preventDefault();
        markInRequest();
      }, false);

      gotoInBtn = gotoInBtn || onceFirst('goto-in', '#edit-control-buttons [name="goto_in"]', context);
      if (gotoInBtn) gotoInBtn.addEventListener('click', function (event) {
        event.preventDefault();
        goToInRequest();
        pauseRequest();
      }, false);

      outBtn = outBtn || onceFirst('set-out', '#edit-control-buttons [name="set_out"]', context);
      if (outBtn) outBtn.addEventListener('click', function (event) {
        event.preventDefault();
        markOutRequest();
      }, false);

      gotoOutBtn = gotoOutBtn || onceFirst('goto-out', '#edit-control-buttons [name="goto_out"]', context);
      if (gotoOutBtn) gotoOutBtn.addEventListener('click', function (event) {
        event.preventDefault();
        goToOutRequest();
        pauseRequest();
      }, false);

      playBtn = playBtn || onceFirst('play', '#edit-control-buttons [name="play"]', context);
      if (playBtn) playBtn.addEventListener('click', function (event) {
        event.preventDefault();
        if (player) {
          player.toggle();
        }
      }, false);

      playInOutBtn = playInOutBtn || onceFirst('play-in-out', '#edit-control-buttons [name="play_in_out"]', context);
      if (playInOutBtn) playInOutBtn.addEventListener('click', function (event) {
        event.preventDefault();
        playInOutRequest();
      }, false);

      prevBtn = prevBtn || onceFirst('previous', '#edit-control-buttons [name="previous"]', context);
      if (prevBtn) prevBtn.addEventListener('click', function (event) {
        event.preventDefault();
        if (player) {
          if (player.isPlaying()) {
            player.pause();
          }
          player.seekToSecondRelative(-1);
        }
      }, false);

      nextBtn = nextBtn || onceFirst('next', '#edit-control-buttons [name="next"]', context);
      if (nextBtn) nextBtn.addEventListener('click', function (event) {
        event.preventDefault();
        if (player) {
          if (player.isPlaying()) {
            player.pause();
          }
          player.seekToSecondRelative(1);
        }
      }, false);

      let elements = once('clips-radio-buttons', '[id=clips-wrapper] input', context);
      if (elements.length) {
        clipRds = elements;
      }
      if (typeof clipRds !== 'undefined') {
        for (let i = 0; i < clipRds.length; i++) {
          clipRds[i].addEventListener('click', function (event) {
            loadClipFromRadioElement(event.target);
          }, false);
        }
      }

      // Support hotkeys when focused on the play button.
      // @todo use common editing hotkeys
      if (playBtn) playBtn.addEventListener('keypress', logKey);
      function logKey(e) {
        if (!player) return;
        e.preventDefault();
        switch(e.code) {
          case 'BracketLeft':
            player.volumeDown();
            break;

          case 'BracketRight':
            player.volumeUp();
            break;

          case 'Comma':
            goToOutRequest(inField, outField);
            break;

          case 'Period':
            goToInRequest(inField, outField);
            break;

          case 'KeyI':
          case 'KeyZ':
            markInRequest(inField, outField);
            break;

          case 'Minus':
          case 'KeyQ':
            previousFrameRequest(inField, outField);
            break;

          case 'Equal':
          case 'KeyE':
            nextFrameRequest(inField, outField);
            break;

          case 'KeyO':
          case 'KeyC':
            markOutRequest(inField, outField);
            break;

          case 'KeyA':
          case 'KeyJ':
            shuttleLeftRequest();
            break;

          case 'KeyS':
            normalSpeedRequest();
            break;

          case 'KeyK':
          case 'KeyW':
            shuttleStopRequest();
            break;

          case 'KeyD':
          case 'KeyL':
            shuttleRightRequest();
            break;

          case 'KeyF':
            plusSecondsRequest(60);
            break;

          case 'KeyG':
            plusSecondsRequest(90);
            break;

          case 'KeyH':
            plusSecondsRequest(180);
            break;

          case 'KeyX':
            normalSpeedRequest();
          case 'Enter':
            playInOutRequest();
            break;
        }
      }
    },

    onPlayerAttach: function () {
      if (!player) return;
      player.setVolume(drupalSettings.clipMediaEntityEditForm.defaultVolumeLevel);
      let clipRadioElement = getCurrentClip();
      if (typeof clipRadioElement !== 'undefined') {
        loadClipFromRadioElement(clipRadioElement);
      }
      setInterval(function () {
        if (!player || !player.isPlaying()) {
          return;
        }
        let currentTime = player.getCurrentTime();
        if (currentField) {
          currentField.setAttribute('value', player.getTimecode(currentTime));
        }
      }, 1000 / player.getFrameRate());
    },

    onPlayerPause: function () {
      if (playBtn) {
        playBtn.setAttribute('value', '\u25B6');
      }
    },

    onPlayerPlay: function () {
      if (playBtn) {
        playBtn.setAttribute('value', '\u23F8');
      }
    },

    getVideoSourceUrl() {
      return new URL(document.getElementById('edit-field-media-oembed-video-0-value').value);
    },
  };

  const onceFirst = function (id, query, context) {
    let elements = once(id, query, context);
    if (elements.length) {
      return elements[0];
    }
  }

  const getCurrentClip = function () {
    if (typeof clipRds === 'undefined') {
      return;
    }
    for(let i = 0; i < clipRds.length; i++) {
      if (clipRds[i].checked) {
        return clipRds[i];
      }
    }
  };

  const loadClipFromRadioElement = function (clipRadioElement) {
    let tds = clipRadioElement.parentElement.parentElement.parentElement.getElementsByTagName('td');
    titleField.value = tds[1].innerHTML;
    inField.value = tds[2].innerHTML;
    outField.value = tds[3].innerHTML;
    goToIn();
  };

  const playInOutRequest = function () {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = true;
    goToIn();
    if (!player.isPlaying()) {
      player.play();
    }
  };

  const pauseRequest = function () {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    player.pause();
  };

  const previousFrameRequest = function () {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    if (player.isPlaying()) {
      player.pause();
    }
    player.seekToFrameRelative(-1);
  };

  const nextFrameRequest = function () {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    if (player.isPlaying()) {
      player.pause();
    }
    player.seekToFrameRelative(1);
  };

  const shuttleLeftRequest = function () {
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    player.shuttleLeft();
  };

  const normalSpeedRequest = function () {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    player.setPlaybackRate(1);
    player.play();
  };

  const shuttleStopRequest = function () {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    player.pause();
  };

  const shuttleSlowRightRequest = function () {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    player.setPlaybackRate(.25);
    player.play();
  };

  const shuttleRightRequest = function () {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    player.shuttleRight();
  };

  const minusSecondsRequest = function (seconds) {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    player.seekToSecondRelative(seconds);
  };

  const plusSecondsRequest = function (seconds) {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    player.seekToSecond(player.getCurrentTime() + seconds);
  };

  const markInRequest = function () {
    if (!player) return;
    let timecode = player.getTimecode(player.getCurrentTime());
    let seconds = player.getTime(timecode);
    inField.value = timecode;
    if (outField.value && seconds > player.getTime(outField.value)) {
      outField.value = player.getTimecode(seconds + 10);
    }
  };

  const goToIn = function () {
    if (!player) return;
    player.seekToTimecode(inField.value);
  };

  const goToInRequest = function () {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    player.seekToTimecode(inField.value);
    player.pause();
  };

  const markOutRequest = function () {
    if (!player) return;
    let timecode = player.getTimecode(player.getCurrentTime());
    outField.value = timecode;
    if (player.getTime(inField.value) > player.getTime(outField.value)) {
      inField.value = timecode;
    }
  };

  const goToOutRequest = function () {
    if (!player) return;
    drupalSettings.clipMediaEntityEditForm.isInOutPlaybackMode = false;
    player.seekToTimecode(outField.value);
    player.pause();
  };

})(jQuery, Drupal);
