/**
 * Clip Lower Thirds Overlay JavaScript
 * Handles fetching and displaying lower thirds overlays for clip players
 */

(function ($, Drupal) {
  'use strict';

  Drupal.behaviors.clipLowerThirds = {
    attach: function (context, settings) {
      // Only run once per page
      if (context !== document) {
        return;
      }

      // Get media ID from URL or settings
      const mediaId = this.getMediaId();
      if (!mediaId) {
        return;
      }

      // Fetch lower thirds data
      this.fetchLowerThirds(mediaId).then(lowerThirds => {
        if (lowerThirds && lowerThirds.length > 0) {
          this.lowerThirdsData = lowerThirds;
          this.currentThirdIndex = 0;
          this.isVisible = false;

          // Set up overlay toggle on spacebar or click
          this.setupOverlayToggle();

          // Auto-show overlay after 2 seconds of playback
          this.setupAutoShow();
        }
      }).catch(error => {
        console.warn('Failed to load lower thirds:', error);
      });
    },

    getMediaId: function() {
      // Try to get from URL path
      const path = window.location.pathname;
      const mediaMatch = path.match(/\/media\/(\d+)/);
      if (mediaMatch) {
        return mediaMatch[1];
      }

      // Try to get from Drupal settings
      if (Drupal.settings && Drupal.settings.clip && Drupal.settings.clip.mediaId) {
        return Drupal.settings.clip.mediaId;
      }

      return null;
    },

    fetchLowerThirds: function(mediaId) {
      const url = `/clip/media/${mediaId}/lower-thirds`;
      return fetch(url)
        .then(response => response.json())
        .then(data => {
          return data;
        })
        .catch(error => {
          console.error('Error fetching lower thirds:', error);
          return null;
        });
    },

    setupOverlayToggle: function() {
      const self = this;

      // Spacebar toggle
      $(document).on('keydown.clipLowerThirds', function(e) {
        if (e.keyCode === 32 && !$(e.target).is('input, textarea')) { // Spacebar
          e.preventDefault();
          self.toggleOverlay();
        }
      });

      // Click toggle on overlay
      $('#clip-lower-thirds-overlay').on('click.clipLowerThirds', function() {
        self.toggleOverlay();
      });
    },

    setupAutoShow: function() {
      const self = this;

      // Show overlay after 2 seconds of playback
      setTimeout(function() {
        // Check if any player is playing
        if (self.isAnyPlayerPlaying()) {
          self.showOverlay();
        }
      }, 2000);

      // Rotate through lower thirds every 15 seconds if visible
      setInterval(function() {
        if (self.isVisible && self.lowerThirdsData && self.lowerThirdsData.length > 1) {
          self.nextLowerThird();
        }
      }, 15000);
    },

    isAnyPlayerPlaying: function() {
      // Check YouTube player
      if (typeof youTubeIframePlayer !== 'undefined' && youTubeIframePlayer) {
        try {
          return youTubeIframePlayer.getPlayerState() === YT.PlayerState.PLAYING;
        } catch (e) {}
      }

      // Check VideoJS player
      if (typeof videojs !== 'undefined') {
        const player = videojs('player');
        if (player) {
          return !player.paused();
        }
      }

      return false;
    },

    toggleOverlay: function() {
      if (this.isVisible) {
        this.hideOverlay();
      } else {
        this.showOverlay();
      }
    },

    showOverlay: function() {
      if (!this.lowerThirdsData || this.lowerThirdsData.length === 0) {
        return;
      }
      const currentThird = this.lowerThirdsData[this.currentThirdIndex];
      this.updateOverlayContent(currentThird);
      $('#clip-lower-thirds-overlay').fadeIn(300);
      this.isVisible = true;
    },

    hideOverlay: function() {
      $('#clip-lower-thirds-overlay').fadeOut(300);
      this.isVisible = false;
    },

    nextLowerThird: function() {
      if (!this.lowerThirdsData || this.lowerThirdsData.length <= 1) {
        return;
      }

      this.currentThirdIndex = (this.currentThirdIndex + 1) % this.lowerThirdsData.length;
      const currentThird = this.lowerThirdsData[this.currentThirdIndex];
      this.updateOverlayContent(currentThird);
    },

    updateOverlayContent: function(third) {
      $('#clip-lower-thirds-heading').text(third.heading || '');
      $('#clip-lower-thirds-text').text(third.text || '');

      if (third.qr) {
        $('#clip-lower-thirds-qr').attr('src', third.qr).show();
      } else {
        $('#clip-lower-thirds-qr').hide();
      }
    }
  };

})(jQuery, Drupal);
