<?php

namespace Drupal\clip\ChannelItemProvider;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\tv\ChannelItemProvider\TvChannelItemProviderInterface;
use Drupal\tv\Entity\TvChannel;
use Symfony\Component\DependencyInjection\Attribute\AutoconfigureTag;

/**
 * Provides clip media items for TV channels.
 */
#[AutoconfigureTag('tv.channel_item_provider')]
class ClipChannelItemProvider implements TvChannelItemProviderInterface {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getWeight(): int {
    return 10; // Higher than default video provider (0)
  }

  /**
   * {@inheritdoc}
   */
  public function getMediaIds(TvChannel $channel): array {
    $tags = $channel->getTags();
    if (empty($tags)) {
      return [];
    }

    // First, find parent media (video/remote_video) that have ALL channel tags.
    $parent_query = $this->entityTypeManager->getStorage('media')->getQuery()
      ->condition('bundle', ['video', 'remote_video'], 'IN')
      ->accessCheck(TRUE);

    // Require parent media to have ALL tags that the channel has.
    foreach ($tags as $tag) {
      $parent_query->condition($parent_query->andConditionGroup()
        ->condition('field_tags', $tag->id()));
    }

    // Allow other modules to alter the clip query before execution.
    \Drupal::moduleHandler()->alter('clip_channel_items_query', $parent_query, $channel);

    $qualifying_parent_ids = $parent_query->execute();

    // If no qualifying parents, return empty.
    if (empty($qualifying_parent_ids)) {
      return [];
    }

    // Query for published clips whose parent (field_source_media) is in the qualifying list.
    $clip_query = $this->entityTypeManager->getStorage('media')->getQuery()
      ->condition('bundle', 'clip')
      ->condition('status', 1)
      ->condition('field_source_media', $qualifying_parent_ids, 'IN')
      ->condition('field_duration', 0, '>')
      ->condition('field_description', NULL, 'IS NOT NULL')
      ->condition('field_thumbnail', NULL, 'IS NOT NULL')
      ->accessCheck(TRUE);

    return $clip_query->execute();
  }

}
