<?php

namespace Drupal\clone_entity_revision\Form;

use Drupal\Core\Url;
use Drupal\file\Entity\File;
use Drupal\node\NodeInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class NodeRevisionCloneForm extends ConfirmFormBase {

  /**
   * The node revision.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $revision;

  /**
   * The node storage.
   *
   * @var \Drupal\node\NodeStorageInterface
   */
  protected $nodeStorage;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a new NodeRevisionCloneForm.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $node_storage
   *   The node storage.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(EntityStorageInterface $node_storage, DateFormatterInterface $date_formatter, TimeInterface $time, FileSystemInterface $file_system) {
    $this->nodeStorage = $node_storage;
    $this->dateFormatter = $date_formatter;
    $this->time = $time;
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')->getStorage('node'),
      $container->get('date.formatter'),
      $container->get('datetime.time'),
      $container->get('file_system')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'node_revision_clone_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to clone revision to new node from %revision-date?', ['%revision-date' => $this->dateFormatter->format($this->revision->getRevisionCreationTime())]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.node.version_history', ['node' => $this->revision->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Clone');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?NodeInterface $node_revision = NULL) {
    $this->revision = $node_revision;
    $form = parent::buildForm($form, $form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Create duplicate but ensure we work with the default translation.
    $duplicate = $this->handleCloneParagraph($this->revision);
    $this->messenger()->addMessage($this->t('A node "@title" has been created with ID <a target="_blank" href="@link">@nid</a>', [
      '@title' => $duplicate->label(),
      '@link' => Url::fromRoute('entity.node.edit_form', ['node' => $duplicate->id()])->toString(),
      '@nid' => $duplicate->id(),
    ]));
    $form_state->setRedirect('entity.node.version_history', [
      'node' => $this->revision->id(),
    ]);
  }

  /**
   * Clone an entity and all of its child paragraphs.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The source entity (node or paragraph).
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface
   *   The cloned entity including cloned child paragraphs.
   */
  public function handleCloneParagraph(ContentEntityInterface $entity): ContentEntityInterface {
    $duplicate = $entity->createDuplicate();

    foreach ($duplicate->getFieldDefinitions() as $field_name => $field_definition) {
      $field_type = $field_definition->getType();

      // Clone paragraph fields
      if (
        $field_type === 'entity_reference_revisions'
        && $field_definition->getSetting('target_type') === 'paragraph'
        && $duplicate->hasField($field_name)
        && !$duplicate->get($field_name)->isEmpty()
      ) {
        $clone_values = [];
        foreach ($duplicate->get($field_name)->referencedEntities() as $referenced_entity) {
          $clone_referenced_entity = $this->handleCloneParagraph($referenced_entity);
          $clone_values[] = [
            'target_id' => $clone_referenced_entity->id(),
            'target_revision_id' => $clone_referenced_entity->getRevisionId(),
          ];
        }
        if (!empty($clone_values)) {
          $duplicate->set($field_name, $clone_values);
        }
      }

      // Clone file/image fields
      if (
        in_array($field_type, ['file', 'image'], TRUE) &&
        $duplicate->hasField($field_name) &&
        !$duplicate->get($field_name)->isEmpty()
      ) {
        $clone_values = [];
        foreach ($duplicate->get($field_name)->referencedEntities() as $delta => $file) {
          if ($file instanceof File) {
            $uri = $file->getFileUri();
            $directory = dirname($uri);
            $new_filename = uniqid('clone_revision_') . '_' . basename($uri);
            $new_uri = $directory . '/' . $new_filename;

            if ($this->fileSystem->copy($uri, $new_uri)) {
              $new_file = File::create([
                'uri' => $new_uri,
              ]);
              $new_file->save();

              $item = $duplicate->get($field_name)->get($delta)->getValue();
              $new_item = ['target_id' => $new_file->id()];

              foreach (['alt', 'title', 'width', 'height', 'description', 'display'] as $meta_key) {
                if (isset($item[$meta_key])) {
                  $new_item[$meta_key] = $item[$meta_key];
                }
              }

              $clone_values[] = $new_item;
            }
          }
        }
        if (!empty($clone_values)) {
          $duplicate->set($field_name, $clone_values);
        }
      }
    }

    // Update created/changed time
    $time = $this->time->getRequestTime();
    if ($duplicate->hasField('created')) {
      $duplicate->set('created', $time);
    }
    if ($duplicate->hasField('changed')) {
      $duplicate->set('changed', $time);
    }

    $duplicate->save();
    return $duplicate;
  }

}
