<?php

namespace Drupal\cloudfront_cache_path_invalidate\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Configure Automatic Cloudfront Cache settings for this site.
 */
class AutoCloudfrontCacheSettingForm extends ConfigFormBase {
  use StringTranslationTrait;

  /**
   * Drupal\Core\Entity\EntityTypeManagerInterface definition.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'cloudfront_cache_path_invalidate.settings';

  /**
   * Constructs a new Cloudfront Cache clear form object.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager
  ) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cloudfront_cache_path_invalidate_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      self::SETTINGS,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(self::SETTINGS);

    // Initialize group indices. This array will hold the numeric indices of all active groups.
    $group_indices = $form_state->get('group_indices');
    $next_index = $form_state->get('next_index');

    if ($group_indices === NULL) {
      // If no group indices are set in form_state, load from config or start with one group.
      $saved_count = $config->get('ecgroupcount') ? $config->get('ecgroupcount') : 1;
      $group_indices = range(0, $saved_count - 1); // Create initial indices from 0 to saved_count-1
      $next_index = $saved_count; // The next available index for a new group.
      $form_state->set('group_indices', $group_indices);
      $form_state->set('next_index', $next_index);
    }

    $form['#tree'] = TRUE;
    $form['ec_fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Automatic Cloudfront Cache Path Invalidate Settings for hook_enity_*()'),
      '#prefix' => '<div id="names-fieldset-wrapper">',
      '#suffix' => '</div>',
    ];

    $entity_options = [];
    // Populate entity_options with content entity types that have bundles.
    // We check if getBundleEntityType() returns a non-NULL value, which means
    // this entity type has associated bundles (e.g., 'node' has 'node_type').
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      if ($entity_type->getBundleEntityType() == NULL) {
        $entity_options[$entity_type_id] = $entity_type->getLabel();
      }
    }
    
    asort($entity_options);
    // Add a placeholder option at the beginning. Its key will be 0.
    array_unshift($entity_options, $this->t('--Select--'));

    // Retrieve saved values from form_state for persistence across AJAX rebuilds.
    // This variable holds the submitted values from previous AJAX interactions.
    $saved_fieldset_values = $form_state->get('saved_fieldset_values') ?? [];

    // Loop through the active group indices to build each group.
    foreach ($group_indices as $i) {
      // Each group is a container to allow for individual removal.
      $form['ec_fieldset'][$i] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['cloudfront-cache-group']],
        '#weight' => $i, // Ensure order is maintained
      ];

      // Determine default value for Entity Type dropdown.
      $selected_entity_type = 0; // Default to '--Select--'

      // 1. Check if the value is present in the current form_state (e.g., if this field was just changed via AJAX).
      $value_from_current_form_state = $form_state->getValue(['ec_fieldset', $i, 'ecentitytype']);
      if ($value_from_current_form_state !== NULL) {
          $selected_entity_type = $value_from_current_form_state;
      }
      // 2. Fallback to our explicitly saved values from previous AJAX rebuilds.
      elseif (isset($saved_fieldset_values[$i]['ecentitytype'])) {
          $selected_entity_type = $saved_fieldset_values[$i]['ecentitytype'];
      }
      // 3. Fallback to stored configuration for initial load.
      elseif (isset($config->get('ecentitytype')[$i])) {
          $selected_entity_type = $config->get('ecentitytype')[$i];
      }


      $form['ec_fieldset'][$i]['ecentitytype'] = [
        '#type' => 'select',
        '#title' => $this->t('Entity Type'),
        '#options' => $entity_options,
        '#required' => TRUE,
        '#weight' => -1,
        '#default_value' => $selected_entity_type,
        '#ajax' => [
          'callback' => '::entityTypeBundle',
          'wrapper' => "entity-bundle-container_$i",
          'event' => 'change',
        ],
      ];

      $entity_bundle_options = [];
      // Only attempt to load bundles if a valid entity type (not the placeholder '0') is selected.
      if (!empty($selected_entity_type) && $selected_entity_type !== 0) {
        // Get the bundle entity type definition for the selected entity type.
        // $bundle_entity_type_id = $this->entityTypeManager->getDefinition($selected_entity_type)->getBundleEntityType();
        // if ($bundle_entity_type_id) {
        //   // Load the bundles using the bundle entity type ID.
        //   foreach ($this->entityTypeManager->getStorage($bundle_entity_type_id)->loadMultiple() as $entity_bundle) {
        //     $entity_bundle_options[$entity_bundle->id()] = $entity_bundle->label();
        //   }
        // }
       
          // Load the bundles using the bundle entity type ID.
          foreach ($this->entityTypeManager->getStorage($selected_entity_type)->loadMultiple() as $entity_bundle) {
            $entity_bundle_options[$entity_bundle->id()] = $entity_bundle->label();
          }

      }
      asort($entity_bundle_options);
      array_unshift($entity_bundle_options, $this->t('--Select--'));

      $form['ec_fieldset'][$i]['entity_bundle_container'] = [
        '#type' => 'container',
        '#attributes' => ['id' => "entity-bundle-container_$i"],
      ];

      // Determine the default value for the Entity Bundle field.
      $default_bundle_value = 0; // Default to '--Select--'
      $triggered_element = $form_state->getTriggeringElement();
      $triggered_element_parents = $triggered_element['#parents'] ?? [];
      $is_current_group_entity_type_ajax = (
        isset($triggered_element_parents[1]) &&
        $triggered_element_parents[1] == $i &&
        isset($triggered_element_parents[2]) &&
        $triggered_element_parents[2] == 'ecentitytype'
      );

      if ($is_current_group_entity_type_ajax) {
        // If the entity type in *this specific group* triggered the AJAX,
        // reset the bundle dropdown for *this group* to --Select--.
        $default_bundle_value = 0;
      } else {
        // For other groups, or if the AJAX was triggered by something else (like remove/add group buttons),
        // prioritize from form_state's saved values, then config.

        // 1. Check current form_state->getValue() (values from the last AJAX submission).
        $value_from_form_state_get_value = $form_state->getValue(['ec_fieldset', $i, 'entity_bundle_container', 'ecentitytypebundle']);

        if ($value_from_form_state_get_value !== NULL) {
            $default_bundle_value = $value_from_form_state_get_value;
        }
        // 2. Fallback to our explicitly saved values from previous AJAX rebuilds.
        elseif (isset($saved_fieldset_values[$i]['entity_bundle_container']['ecentitytypebundle'])) {
            $default_bundle_value = $saved_fieldset_values[$i]['entity_bundle_container']['ecentitytypebundle'];
        }
        // 3. Fallback to stored configuration for initial load.
        elseif (isset($config->get('ecentitytypebundle')[$i])) {
            $default_bundle_value = $config->get('ecentitytypebundle')[$i];
        }
      }

      $form['ec_fieldset'][$i]['entity_bundle_container']['ecentitytypebundle'] = [
        '#type' => 'select',
        '#title' => $this->t('Entity Bundle'),
        '#options' => $entity_bundle_options,
        '#required' => TRUE,
        '#weight' => 2,
        '#default_value' => $default_bundle_value,
      ];

      $form['ec_fieldset'][$i]['detail_page'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Detail Page'),
        '#default_value' => $form_state->getValue(['ec_fieldset', $i, 'detail_page'], $config->get('detail_page')[$i] ?? FALSE),
        '#weight' => 3,
      ];

      $form['ec_fieldset'][$i]['ec_cloudfront_url'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Cloudfront URL to invalidate'),
        '#description' => $this->t('Specify the existing path you wish to invalidate. For example: /sector/*, /state/*. Enter one value per line.'),
        '#placeholder' => $this->t('Cloudfront URL'),
        '#weight' => 4,
        '#default_value' => $form_state->getValue(['ec_fieldset', $i, 'ec_cloudfront_url'], $config->get('ec_cloudfront_url')[$i] ?? ''),
      ];

      // Add a remove button for this specific group.
      $form['ec_fieldset'][$i]['remove_group'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove this group'),
        '#name' => 'remove_group_' . $i, // Unique name to identify which group to remove.
        '#submit' => ['::removeGroup'],
        '#limit_validation_errors' => [], // Do not validate other fields when removing.
        '#ajax' => [
          'callback' => '::addMoreCallback',
          'wrapper' => 'names-fieldset-wrapper',
        ],
        '#weight' => 5,
      ];
    }

    // Actions for adding new groups.
    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['add_more_ec'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another group'),
      '#submit' => ['::addMore'],
      '#ajax' => [
        'callback' => '::addMoreCallback',
        'wrapper' => 'names-fieldset-wrapper',
      ],
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * Callback for both ajax-enabled buttons.
   *
   * Selects and returns the fieldset with the names in it.
   */
  public function addMoreCallback(array &$form, FormStateInterface $form_state) {
    return $form['ec_fieldset'];
  }

  /**
   * Callback for entity type bundle selection via AJAX.
   *
   * Selects and returns the entity bundle container for a specific group.
   */
  public function entityTypeBundle(array &$form, FormStateInterface $form_state) {
    $triggered_element = $form_state->getTriggeringElement();
    // The index of the group is the second element in the #parents array.
    $group_index = $triggered_element['#parents'][1];
    return $form['ec_fieldset'][$group_index]['entity_bundle_container'];
  }

  /**
   * Submit handler for the "Add another group" button.
   *
   * Increments the max counter and causes a rebuild.
   */
  public function addMore(array &$form, FormStateInterface $form_state) {
    $group_indices = $form_state->get('group_indices');
    $next_index = $form_state->get('next_index');

    // Add the next available index to the list of active groups.
    $group_indices[] = $next_index;
    $form_state->set('group_indices', $group_indices);
    // Increment the next_index for the next 'Add more' click.
    $form_state->set('next_index', $next_index + 1);

    // Capture current values from user input for persistence across rebuilds.
    // Use getUserInput() to get all raw submitted values.
    $user_input_fieldset_values = $form_state->getUserInput()['ec_fieldset'] ?? [];
    $form_state->set('saved_fieldset_values', $user_input_fieldset_values);

    $form_state->setRebuild();
  }

  /**
   * Submit handler for the individual "Remove this group" buttons.
   *
   * Removes the specific group's index from the list of active groups and causes a form rebuild.
   */
  public function removeGroup(array &$form, FormStateInterface $form_state) {
    $triggered_element = $form_state->getTriggeringElement();
    // Extract the group index from the button's #name (e.g., 'remove_group_X').
    preg_match('/remove_group_(\d+)/', $triggered_element['#name'], $matches);
    $index_to_remove = isset($matches[1]) ? (int) $matches[1] : NULL;

    if ($index_to_remove !== NULL) {
      $group_indices = $form_state->get('group_indices');
      // Find the key of the index to remove in the group_indices array.
      $key_to_remove = array_search($index_to_remove, $group_indices);

      if ($key_to_remove !== FALSE) {
        // Remove the index from the array.
        unset($group_indices[$key_to_remove]);
        // Re-index the array to avoid issues with non-sequential keys if needed,
        // though Drupal's form API usually handles this well.
        $group_indices = array_values($group_indices);
        $form_state->set('group_indices', $group_indices);
      }
    }
    
    // Capture current values from user input for persistence across rebuilds.
    $user_input_fieldset_values = $form_state->getUserInput()['ec_fieldset'] ?? [];
    // If a group was actually removed, we need to adjust saved_fieldset_values
    // to reflect the removal for the next build.
    if ($index_to_remove !== NULL && isset($user_input_fieldset_values[$index_to_remove])) {
        unset($user_input_fieldset_values[$index_to_remove]);
    }
    $form_state->set('saved_fieldset_values', $user_input_fieldset_values);


    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $fieldset_values = $form_state->getValue(['ec_fieldset']);
    
    // Validate only the groups that are currently present in the form_state.
    $active_group_indices = $form_state->get('group_indices');

    foreach ($active_group_indices as $group_index) {
      // Ensure the group actually exists in the submitted values.
      if (isset($fieldset_values[$group_index])) {
        $value = $fieldset_values[$group_index];

        if (!empty($value['ec_cloudfront_url'])) {
          $url_value = explode("\n", $value['ec_cloudfront_url']);
          foreach ($url_value as $url) {
            if (trim($url) !== '' && substr(trim($url), 0, 1) !== '/') {
              $form_state->setErrorByName("ec_fieldset][$group_index][ec_cloudfront_url", $this->t('The Cloudfront URL must start with a forward slash (/).'));
              break;
            }
          }
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $submitted_values = $form_state->getValue(['ec_fieldset']);
    $active_group_indices = $form_state->get('group_indices');
    $ecentitytype = [];
    $ecentitytypebundle = [];
    $detail_page = [];
    $ec_cloudfront_url = [];

    // Iterate through the active group indices to collect values in the correct order.
    foreach ($active_group_indices as $index) {
      // Ensure the submitted value for this index exists.
      if (isset($submitted_values[$index])) {
        $value = $submitted_values[$index];
        $ecentitytype[] = $value['ecentitytype'] ?? '';
        $ecentitytypebundle[] = $value['entity_bundle_container']['ecentitytypebundle'] ?? '';
        $detail_page[] = $value['detail_page'] ?? '';
        $ec_cloudfront_url[] = $value['ec_cloudfront_url'] ?? '';
      }
    }

    $this->config(self::SETTINGS)
      ->set('ecentitytype', $ecentitytype)
      ->set('ecentitytypebundle', $ecentitytypebundle)
      ->set('detail_page', $detail_page)
      ->set('ec_cloudfront_url', $ec_cloudfront_url)
      ->set('ecgroupcount', count($active_group_indices)) // Update count based on active groups.
      ->save();
    $this->messenger()->addStatus($this->t('Cloudfront Cache Setting has been saved.'));
  }

}
