<?php

namespace Drupal\cloudfront_invalidate_all\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure CloudFront Invalidate All settings for this site.
 */
class CloudFrontInvalidateAllSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cloudfront_invalidate_all_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['cloudfront_invalidate_all.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('cloudfront_invalidate_all.settings');

    $form['disabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Temporarily disable CloudFront invalidations'),
      '#description' => $this->t('When checked, no invalidation requests will be sent to CloudFront.'),
      '#default_value' => $config->get('disabled') ?? FALSE,
    ];

    $form['distribution_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CloudFront Distribution ID'),
      '#description' => $this->t('The CloudFront distribution ID to invalidate (e.g., E1EXAMPLE123).'),
      '#default_value' => $config->get('distribution_id'),
    ];

    $form['region'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWS Region'),
      '#description' => $this->t('The AWS region for CloudFront API calls. Default is us-east-1.'),
      '#default_value' => $config->get('region') ?: 'us-east-1',
    ];

    $form['whitelist'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Cache Tag Whitelist'),
      '#description' => $this->t('Enter cache tag prefixes to allow, one per line. Only tags that start with these values will trigger CloudFront invalidation. For example: "node_list" or "node:" (which will match node:1, node:2, etc.)'),
      '#default_value' => implode("\n", $config->get('whitelist') ?: []),
      '#rows' => 10,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $distribution_id = $form_state->getValue('distribution_id');
    if (!empty($distribution_id) && !preg_match('/^[A-Z0-9]+$/', $distribution_id)) {
      $form_state->setErrorByName('distribution_id', $this->t('Distribution ID must contain only uppercase letters and numbers.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $whitelist = array_filter(array_map('trim', explode("\n", $form_state->getValue('whitelist'))));

    $this->config('cloudfront_invalidate_all.settings')
      ->set('disabled', $form_state->getValue('disabled'))
      ->set('distribution_id', $form_state->getValue('distribution_id'))
      ->set('region', $form_state->getValue('region'))
      ->set('whitelist', array_values($whitelist))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
