<?php

namespace Drupal\cloudfront_invalidate_all;

use Aws\CloudFront\CloudFrontClient;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;

/**
 * Invalidates CloudFront cache when Drupal cache tags are invalidated.
 */
final class CloudFrontInvalidator implements CacheTagsInvalidatorInterface {

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  private LoggerChannelInterface $logger;

  /**
   * Constructs a CloudFrontInvalidator object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger channel factory.
   */
  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->logger = $loggerFactory->get('cloudfront_invalidate_all');
  }

  /**
   * {@inheritdoc}
   */
  public function invalidateTags(array $tags) {
    // Don't proceed if no tags to invalidate.
    if (empty($tags)) {
      return;
    }

    $config = $this->configFactory->get('cloudfront_invalidate_all.settings');

    // Check if invalidation is disabled.
    if ($config->get('disabled')) {
      return;
    }

    $distributionId = $config->get('distribution_id');
    if (!$distributionId) {
      $this->logger->error('CloudFront distribution id not configured');
      return;
    }

    // Only process whitelisted tags.
    $whitelistedTagPrefixes = $config->get('whitelist') ?? [];
    if (empty($whitelistedTagPrefixes)) {
      // No whitelist means nothing should trigger invalidation.
      return;
    }
    // Debug: Log whitelist configuration.
    if ($config->get('debug')) {
      $this->logger->debug(
        'Processing @count tags with @whitelist_count whitelist prefixes: @whitelist',
        [
          '@count' => count($tags),
          '@whitelist_count' => count($whitelistedTagPrefixes),
          '@whitelist' => implode(', ', $whitelistedTagPrefixes),
        ]
      );
    }

    $contentTags = [];

    foreach ($tags as $tag) {
      $whitelisted = FALSE;
      foreach ($whitelistedTagPrefixes as $prefix) {
        if (strpos($tag, $prefix) === 0) {
          $whitelisted = TRUE;
          break;
        }
      }
      if ($whitelisted) {
        $contentTags[] = $tag;
      }
    }

    // Don't invalidate CloudFront if no content-related tags.
    if (empty($contentTags)) {
      // Debug log when skipping due to whitelist.
      if ($config->get('debug')) {
        $this->logger->debug(
          'Skipped CloudFront invalidation - none of @count tags matched whitelist: @tags',
          [
            '@count' => count($tags),
            '@tags' => implode(', ', $tags),
          ]
        );
      }
      return;
    }

    try {
      $client = new CloudFrontClient([
        'region'  => $config->get('region') ?? 'us-east-1',
        'version' => 'latest',
      ]);

      $result = $client->createInvalidation([
        'DistributionId'    => $distributionId,
        'InvalidationBatch' => [
          'CallerReference' => microtime(TRUE),
          'Paths' => [
            'Quantity' => 1,
            'Items'    => ['/*'],
          ],
        ],
      ]);

      $this->logger->notice(
        'CloudFront invalidation sent (ID: @id) for @content_count content tags (filtered from @total_count total tags): @tags.',
        [
          '@id' => $result['Invalidation']['Id'] ?? 'unknown',
          '@content_count' => count($contentTags),
          '@total_count' => count($tags),
          '@tags' => implode(', ', $contentTags),
        ]
      );
    }
    catch (\Throwable $e) {
      $this->logger->error(
        'CloudFront invalidation failed: @msg',
        ['@msg' => $e->getMessage()]
      );
    }
  }

}
