<?php

namespace Drupal\cm_tools\Plugin\Field\FieldWidget;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\field\FieldConfigInterface;
use Drupal\paragraphs\Plugin\Field\FieldWidget\ParagraphsWidget;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'cm_tools_paragraphs_table_widget' widget.
 *
 * @FieldWidget(
 *   id = "cm_tools_paragraphs_table_widget",
 *   label = @Translation("Paragraphs table - CM Tools"),
 *   description = @Translation("Paragraphs table form widget."),
 *   field_types = {
 *     "entity_reference_revisions"
 *   }
 * )
 */
class ParagraphsTableWidget extends ParagraphsWidget {

  /**
   * Constructs a ParagraphsWidget object.
   *
   * @param string $plugin_id
   *   The plugin_id for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the widget is associated.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   Entity field manager Service.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entityDisplayRepository
   *   Entity display service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager service.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    EntityFieldManagerInterface $entity_field_manager,
    protected EntityDisplayRepositoryInterface $entityDisplayRepository,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings, $entity_field_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new self(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_field.manager'),
      $container->get('entity_display.repository'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    // We only really support the form_display_mode setting.
    foreach (Element::children($elements) as $child) {
      if ($child != 'form_display_mode') {
        $elements[$child]['#access'] = FALSE;
      }
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $summary[] = $this->t('Form display mode: @form_display_mode', [
      '@form_display_mode' => $this->getSetting('form_display_mode')
    ]);

    return $summary;
  }

  /**
   * For multiple elements.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function formMultipleElements(FieldItemListInterface $items, array &$form, FormStateInterface $form_state) {
    $elements = parent::formMultipleElements($items, $form, $form_state);
    $settings = $this->fieldDefinition->getSettings();
    $handler = $settings['handler_settings'];
    if (!empty($handler["target_bundles"]) && count($handler["target_bundles"]) > 1) {
      return $elements;
    }
    // This flags to our cm_tools_preprocess_field_multiple_value_form.
    $elements["#cm_tools_paragraphsTable"] = [];
    $target_type = $this->getFieldSetting('target_type');
    $default_type = $this->getDefaultParagraphTypeMachineName();
    // Look to remove all these.
    if (empty($default_type)) {
      $default_type = array_key_first($this->getAllowedTypes());
    }
    if (empty($default_type)) {
      return $elements;
    }
    $formDisplay = $this->entityDisplayRepository
      ->getFormDisplay($target_type, $default_type, $this->getSetting('form_display_mode'));
    $components = $formDisplay->getComponents();
    uasort($components, 'Drupal\Component\Utility\SortArray::sortByWeightElement');

    /** @var \Drupal\paragraphs\ParagraphInterface $paragraphs_entity */
    $paragraphs_entity = $this->entityTypeManager->getStorage($target_type)
      ->create(['type' => $default_type]);
    $field_definitions = $paragraphs_entity->getFieldDefinitions();

    foreach ($components as $name => $setting) {
      if (!empty($field_definitions[$name]) && $field_definitions[$name] instanceof FieldConfigInterface) {
        $elements["#cm_tools_paragraphsTable"]['#fields'][$name] = $field_definitions[$name];
      }
    }
    // Remove fieldgroups we don't need it.
    if (!empty($elements[0]) && $elements[0]['subform']) {
      foreach (range(0, $elements["#max_delta"]) as $delta) {
        $elements[$delta]["subform"]["#fieldgroups"] = [];
      }
    }
    return $elements;
  }

  /**
   * Implements hook_preprocess_HOOK().
   */
  public static function preprocessFieldMultipleValueForm(&$variables) {
    $element = $variables['element'];
    if (empty($element['#cm_tools_paragraphsTable'])) {
      return FALSE;
    }
    if (!empty($variables['table'])) {
      // Case number of values unlimited - cardinality = -1.
      if (!empty($element['#cm_tools_paragraphsTable']) && !empty($variables["table"]["#tabledrag"])) {

        // Add caption header.
        if (!empty($element["#title"])) {
          $variables["table"]["#caption"] = $variables["element"]["#title"];
          $variables["table"]["#attributes"]["class"][] = 'caption-top';
        }
        if (!empty($element["#title_display"]) && $element["#title_display"] == 'invisible') {
          unset($variables["table"]["#caption"]);
        }

        $row_weight = end($variables["table"]["#header"]);
        if ($element["#cardinality"] == 1) {
          $row_weight = FALSE;
          unset($variables['table']['#tabledrag']);
        }

        $is_new = isset($variables['table']['#rows']) && count($variables['table']['#rows']) == 0;
        $variables["table"]["#header"] = static::_cm_tools_paragraphs_table_header($element["#cm_tools_paragraphsTable"]["#fields"] ?? [], $row_weight, $is_new);
        $remove_row = FALSE;
        foreach ($variables["table"]["#rows"] as $delta => $rows) {
          $row = static::_cm_tools_paragraphs_table_row($element["#cm_tools_paragraphsTable"]["#fields"] ?? [], $rows['data'], FALSE);
          if (!empty($row)) {
            $variables["table"]["#rows"][$delta]['data'] = $row;
          }
          else {
            $remove_row = TRUE;
            unset($variables["table"]["#rows"][$delta]);
          }
        }
        if ($remove_row) {
          $variables["table"]["#rows"] = array_merge($variables["table"]["#rows"]);
        }
        if (empty($variables["table"]["#rows"]) || !empty($element["#no_header"])) {
          unset($variables["table"]["#header"]);
          unset($variables["table"]["#caption"]);
        }

      }
    }
    elseif (!empty($element["#cardinality"]) && $element["#cardinality"] == 1) {
      // Case single value.
      $variables["elements"][0]["subform"] = static::_cm_tools_paragraphs_table_1_row($element["#cm_tools_paragraphsTable"]["#fields"], $variables["elements"][0]["subform"]);
      $variables["elements"][0]["subform"]["#header"] = static::_cm_tools_paragraphs_table_header($element["#cm_tools_paragraphsTable"]["#fields"]);

    }
  }

  /**
   * Get table header.
   */
  public static function _cm_tools_paragraphs_table_header($fields, $rowWeightCol = FALSE, $is_new = FALSE) {
    $header = ['data' => ''];
    if (!empty($fields)) {
      foreach ($fields as $field_name => $field) {
        $header[] = [
          'data' => $field->getLabel(),
          'class' => [$field_name],
        ];
      }
    }
    if (!$is_new) {
      $header[] = ['data' => '', 'class' => ['action']];
      if (!empty($rowWeightCol)) {
        $header[] = $rowWeightCol;
      }
    }
    else {
      $header[0] = $header[1];
    }
    return $header;
  }

  /**
   * Hide field label.
   */
  public static function _cm_tools_paragraphs_table_hidden_label(&$field_form) {
    $label = '';
    if (!empty($field_form['#title_display'])) {
      $label = $field_form['#title'] ?? '';
      $field_form['#title_display'] = 'invisible';
    }
    if (empty($label) && !empty($field_form['target_id']['#title'])) {
      $label = $field_form['target_id']['#title'];
    }
    // Remove title field date.
    if (!empty($field_form['#type'])) {
      $invisibleType = ['select', 'managed_file', 'entity_autocomplete'];
      if (in_array($field_form['#type'], $invisibleType)) {
        $field_form['#title_display'] = 'invisible';
      }
      if ($field_form['#type'] == 'datetime') {
        $label = $field_form['#title'] ?? '';
        $field_form['#title'] = '';
      }
      if (in_array($field_form['#type'], ['radios', 'checkboxes'])) {
        return $label;
      }
    }
    // Remove fieldset field date & time.
    if (!empty($field_form["#theme_wrappers"][0]) && in_array($field_form["#theme_wrappers"][0], ['fieldset'])) {
      array_shift($field_form["#theme_wrappers"]);
    }
    foreach (Element::children($field_form) as $child) {
      $label = static::_cm_tools_paragraphs_table_hidden_label($field_form[$child]);
    }
    return $label;
  }

  /**
   * Get table row.
   */
  public static function _cm_tools_paragraphs_table_1_row($fields, $rows_data, $show_all = FALSE) {
    $rows['#type'] = 'table';
    foreach ($fields as $field_name => $field) {
      $field_form = $rows_data[$field_name];
      static::_cm_tools_paragraphs_table_hidden_label($field_form['widget']);
      $rows[0][$field_name] = $field_form;
    }
    return $rows;
  }

  /**
   * Generate table row.
   */
  public static function _cm_tools_paragraphs_table_row($fields, $rows_data, $show_all = FALSE) {
    $row[] = $rows_data[0];
    if (isset($rows_data[1]["data"]["#access"]) &&
      $rows_data[1]["data"]["#access"] === FALSE) {
      return FALSE;
    }
    if (!empty($rows_data[1]["data"]["subform"]) && !empty($rows_data[1]['data']['#paragraph_type'])) {
      $subform = $rows_data[1]["data"]["subform"];
    }
    else {
      return FALSE;
    }
    if (!empty($fields)) {
      foreach ($fields as $field_name => $field) {
        if (!empty($subform[$field_name])) {
          $field_form = $subform[$field_name];
          if (!empty($field_form['widget'])) {
            static::_cm_tools_paragraphs_table_hidden_label($field_form['widget']);
          }
          $row[$field_name] = [
            'data' => $field_form,
            'class' => $field_name,
          ];
          unset($subform[$field_name]);
        }
      }
    }
    $actions = !empty($rows_data[1]["data"]["top"]["actions"]) ? $rows_data[1]["data"]["top"]["actions"] : [];
    unset($actions["actions"]["collapse_button"]);
    $row[] = [
      'data' => $actions,
      'class' => ['action'],
    ];
    // Add weight.
    $row[] = end($rows_data);
    return $row;
  }
}
