<?php

namespace Drupal\cm_tools\Plugin\Field\FieldFormatter;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceFormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\field\FieldConfigInterface;
use Drupal\paragraphs\ParagraphInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'cm_tools_paragraphs_table_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "cm_tools_paragraphs_table_formatter",
 *   label = @Translation("Paragraphs table - CM Tools"),
 *   field_types = {
 *     "entity_reference_revisions"
 *   }
 * )
 */
class ParagraphsTableFormatter extends EntityReferenceFormatterBase {

  /**
   * Constructs a ParagraphsTableFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entityDisplayRepository
   *   The entity display repository.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The render service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    protected EntityDisplayRepositoryInterface $entityDisplayRepository,
    protected RendererInterface $renderer,
    protected ModuleHandlerInterface $moduleHandler,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('entity_display.repository'),
      $container->get('renderer'),
      $container->get('module_handler'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    $target_type = $field_definition->getSetting('target_type');
    $paragraph_type = \Drupal::entityTypeManager()->getDefinition($target_type);
    if ($paragraph_type) {
      return $paragraph_type->entityClassImplements(ParagraphInterface::class);
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    // Implement default settings.
    return [
      'view_mode' => 'default',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $settingForm = [
      'view_mode' => [
        '#type' => 'select',
        '#options' => $this->entityDisplayRepository->getViewModeOptions($this->getFieldSetting('target_type')),
        '#title' => $this->t('View mode'),
        '#default_value' => $this->getSetting('view_mode'),
        '#required' => TRUE,
      ],
    ];
    return $settingForm + parent::settingsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    // Implement settings summary.
    $view_modes = $this->entityDisplayRepository->getViewModeOptions($this->getFieldSetting('target_type'));
    $view_mode = $this->getSetting('view_mode');
    $summary[] = $this->t('Rendered as @view_mode', ['@view_mode' => !empty($view_modes[$view_mode]) ? $view_modes[$view_mode] : $view_mode]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $setting = $this->getSettings();
    $targetType = $this->getFieldSetting('target_type');
    $field_definition = $items->getFieldDefinition();
    $output = [];
    $handler = $field_definition->getSetting("handler_settings");
    if (empty($handler["target_bundles"])) {
      return $output;
    }
    $entities = $this->getEntitiesToView($items, $langcode);

    foreach ($handler['target_bundles'] as $targetBundle) {
      $table_header = $fields = [];

      /** @var \Drupal\paragraphs\ParagraphInterface $paragraphs_entity */
      $paragraphs_entity = $this->entityTypeManager->getStorage($targetType)
        ->create(['type' => $targetBundle]);
      $field_definitions = $paragraphs_entity->getFieldDefinitions();
      $view_mode = $setting['view_mode'];
      $viewDisplay = $this->entityDisplayRepository->getViewDisplay($targetType, $targetBundle, $view_mode);
      $components = $viewDisplay->getComponents();
      uasort($components, 'Drupal\Component\Utility\SortArray::sortByWeightElement');

      foreach ($components as $field_name => $component) {
        if (!empty($field_definitions[$field_name]) && $field_definitions[$field_name] instanceof FieldConfigInterface) {
          $fields[$field_name] = $field_definitions[$field_name];
          $table_header[$field_name] = $this->t("@name",
            ["@name" => $field_definitions[$field_name]->getLabel()]
          );
        }
      }

      $table_rows = $this->getPreparedRenderedEntities($entities, $fields, $viewDisplay);
      $table = $this->getTable($table_header, $table_rows, $entities);

      $table_id = implode('-', [$targetType, $targetBundle]);
      $table['#attributes']['id'] = $table_id;
      $table['#attributes']['class'][] = 'paragraphs-table';

      // Alter table results.
      $this->moduleHandler
        ->alter('paragraphs_table_formatter', $table, $table_id);
      $output[] = $table;
    }

    return $output;
  }

  /**
   * {@inheritdoc}
   */
  public function getTable($table_columns, $table_rows, $entities) {
    $table = [
      '#theme' => 'table',
      '#rows' => $table_rows,
      '#header' => $table_columns,
    ];
    $this->cacheMetadata($entities, $table);
    return $table;
  }

  /**
   * Cache.
   */
  protected function cacheMetadata($entities, &$table) {
    $cache_metadata = new CacheableMetadata();
    foreach ($entities as $entity) {
      $cache_metadata->addCacheableDependency($entity);
      $cache_metadata->addCacheableDependency($entity->access('view', NULL, TRUE));
    }
    $cache_metadata->applyTo($table);
  }

  /**
   * Prepare all the given entities for rendering with applicable fields.
   */
  protected function getPreparedRenderedEntities($entities, $fields, EntityViewDisplayInterface $viewDisplay) {
    $rows = [];
    foreach ($entities as $delta => $entity) {
      $entity_bundle = $entity->bundle();
      if ($viewDisplay->getTargetBundle() != $entity_bundle) {
        continue;
      }
      $table_entity = $viewDisplay->build($entity);
      $paragraphs_id = $entity->get('id')->value;

      foreach ($fields as $field_name => $field) {
        $table_entity[$field_name]['#label_display'] = 'hidden';
        $value = $this->renderer->render($table_entity[$field_name]);
        $rows[$delta]['data'][$field_name] = $value;
      }

      $rows[$delta]["data-id"] = $paragraphs_id;
    }
    return $rows;
  }

}
