<?php

namespace Drupal\cm_tools\Plugin\WebformHandler;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\Utility\WebformHtmlHelper;
use Drupal\webform\WebformSubmissionInterface;

/**
 * Webform validation handler.
 *
 * @WebformHandler(
 *   id = "cm_tools_page_level_validation",
 *   label = @Translation("Page level validation"),
 *   category = @Translation("Validation"),
 *   description = @Translation("Provides a way to set validation errors on a whole form."),
 *   cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_UNLIMITED,
 *   results = \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_IGNORED,
 *   submission = \Drupal\webform\Plugin\WebformHandlerInterface::SUBMISSION_OPTIONAL,
 * )
 */
class CmToolsPageLevelValidation extends WebformHandlerBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'message' => '',
      'pages_to_set_on' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['message'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Error message'),
    ];

    $form['message']['message'] = [
      '#type' => 'webform_html_editor',
      '#title' => $this->t('Page level message'),
      '#description' => $this->t("Use the 'Conditions' tab to set when this error should be shown."),
      '#default_value' => $this->configuration['message'],
      '#required' => TRUE,
    ];

    $page_elements = [];

    /** @var \Drupal\webform_ui\Form\WebformUiElementEditForm $form_object */
    $form_object = $form_state->getFormObject();
    $webform = $form_object->getWebform();
    foreach ($webform->getElementsInitializedAndFlattened() as $key => $element) {
      if (isset($element['#type']) && $element['#type'] == 'webform_wizard_page') {
        $page_elements[$key] = $this->t('@title - @key', [
          '@title' => $element['#title'],
          '@key' => $key,
        ]);
      }
    }

    // Optionally restrict to setting the error on specific pages.
    $form['message']['pages_to_set_on'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Only set errors on the selected pages'),
      '#options' => $page_elements,
      '#access' => count($page_elements) > 1,
      '#default_value' => $this->configuration['pages_to_set_on'],
      '#description' => $this->t('If the elements you are conditional on appear pages late in the form, you might want to prevent setting the validation message until the visitor is at those pages.'),
    ];

    return $this->setSettingsParents($form);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $form_state->setValue('pages_to_set_on', array_keys(array_filter($form_state->getValue('pages_to_set_on'))));
    $this->applyFormStateToConfiguration($form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $webform_submission) {
    parent::validateForm($form, $form_state, $webform_submission);

    $current_page = $form_state->has('current_page') ? $form_state->get('current_page') : '';
    // Optionally check that we are on the right page.
    if (empty($this->configuration['pages_to_set_on']) || in_array($current_page, $this->configuration['pages_to_set_on'], TRUE)) {
      // We assume that someone has used the webform conditions section to set
      // appropriate conditions, so we do not need to do further processing
      // ourselves, we can simply set the error message on the whole form.
      $form_state->setErrorByName('', WebformHtmlHelper::toHtmlMarkup($this->configuration['message']));
    }
  }

}
