<?php

namespace Drupal\cmrf_form_processor_mollie\EventSubscriber;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\mollie\Entity\Payment;
use Drupal\mollie\Events\MollieNotificationEvent;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for Mollie form processor.
 */
class MollieFormProcessorSubscriber implements EventSubscriberInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private $entityTypeManager;

  /**
   * MollieFormProcessorSubscriber constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public function callFormProcessor(MollieNotificationEvent $event): void {
    $httpCode = 200;

    // Fetch the transaction.
    try {
      $submission = $this->entityTypeManager
        ->getStorage('webform_submission')
        ->load($event->getContextId());
      if (!$submission instanceof WebformSubmissionInterface) {
        return;
      }

      $webform = $submission->getWebform();
      if (!$webform instanceof WebformInterface) {
        return;
      }

      $handlers = iterator_to_array($webform->getHandlers()->getIterator());
      $handlers = array_filter($handlers, function ($handler) {
        $plugin_definition = $handler->getPluginDefinition();
        return $plugin_definition['provider'] === 'cmrf_form_processor';
      });

      $transaction = $this->entityTypeManager
        ->getStorage('mollie_payment')
        ->load($event->getTransactionId());
      if (!$transaction instanceof Payment) {
        return;
      }

      foreach ($handlers as $handler) {
        if ($handler->checkConditions($submission)) {
          $handler->postSave($submission, TRUE, [
            'mollie_payment_id' => $event->getTransactionId(),
            'mollie_payment_status' => $transaction->getStatus(),
          ]);
        }
      }
    }
    catch (\Exception $e) {
      if (function_exists('watchdog_exception')) {
        // For Drupal 10.0 and earlier, use the deprecated watchdog_exception function.
        watchdog_exception('cmrf_form_processor_mollie', $e);
      }
      else {
        // For Drupal 10.1 and later, use the logger service.
        \Drupal\Core\Utility\Error::logException(\Drupal::logger('cmrf_form_processor_mollie'), $e);
      }
      $httpCode = 500;
    }

    // Set the HTTP code to return to Mollie.
    $event->setHttpCode($httpCode);
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      MollieNotificationEvent::EVENT_NAME => 'callFormProcessor',
    ];
  }

}
