<?php

namespace Drupal\cms_content_sync\Controller;

use Psr\Log\LoggerInterface;

/**
 * Proxy logging to Drupal's regular logger but also cache messages statically
 * at runtime to be able to provide them to the Sync Core on failure.
 * Only used for logging related to pushing or pulling content.
 */
class LoggerProxy implements LoggerInterface {
  /**
   * All captured log messages as an array of associative arrays, ready to be
   * serialized at the REST interface. Includes:
   * - level
   * - message
   * - context
   *
   * @var array
   */
  public static $messages = [];

  /**
   * Get singleton.
   *
   * @return LoggerProxy
   */
  public static function get() {
    static $logger = new LoggerProxy();
    return $logger;
  }

  /**
   * Add a log message.
   *
   * @param string $level
   *   One of Drupal's log levels.
   * @param mixed $message
   *   A string or Stringable object.
   * @param array $context
   *   The tokens to be replaced in the message.
   *
   * @return void
   */
  public static function add(string $level, $message, array $context): void {
    unset($context['@request_body']);
    self::$messages[] = [
      'level' => $level,
      'message' => is_string($message) ? $message : $message . '',
      'context' => $context,
      'timestamp' => round(microtime(TRUE) * 1_000),
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function emergency($message, array $context = []): void {
    \Drupal::logger('cms_content_sync')->emergency($message, $context);
    self::add('emergency', $message, $context);
  }

  /**
   * {@inheritDoc}
   */
  public function alert($message, array $context = []): void {
    \Drupal::logger('cms_content_sync')->alert($message, $context);
    self::add('alert', $message, $context);
  }

  /**
   * {@inheritDoc}
   */
  public function critical($message, array $context = []): void {
    \Drupal::logger('cms_content_sync')->critical($message, $context);
    self::add('critical', $message, $context);
  }

  /**
   * {@inheritDoc}
   */
  public function error($message, array $context = []): void {
    \Drupal::logger('cms_content_sync')->error($message, $context);
    self::add('error', $message, $context);
  }

  /**
   * {@inheritDoc}
   */
  public function warning($message, array $context = []): void {
    \Drupal::logger('cms_content_sync')->warning($message, $context);
    self::add('warning', $message, $context);
  }

  /**
   * {@inheritDoc}
   */
  public function notice($message, array $context = []): void {
    \Drupal::logger('cms_content_sync')->notice($message, $context);
    self::add('notice', $message, $context);
  }

  /**
   * {@inheritDoc}
   */
  public function info($message, array $context = []): void {
    \Drupal::logger('cms_content_sync')->info($message, $context);
    self::add('info', $message, $context);
  }

  /**
   * {@inheritDoc}
   */
  public function debug($message, array $context = []): void {
    \Drupal::logger('cms_content_sync')->debug($message, $context);
    self::add('debug', $message, $context);
  }

  /**
   * {@inheritDoc}
   */
  public function log($level, $message, array $context = []): void {
    \Drupal::logger('cms_content_sync')->log($message, $context);
    self::add('log', $message, $context);
  }

}
