<?php

namespace Drupal\cms_content_sync\Commands;

use Drupal\cms_content_sync\Cli\CliService;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Input\InputOption;

/**
 * Content Sync Drush Commands.
 */
class CMSContentSyncCommands extends DrushCommands {
  /**
   * The interoperability cli service.
   *
   * @var \Drupal\cms_content_sync\Cli\CliService
   */
  protected $cliService;

  /**
   * CMS Content Sync constructor.
   *
   * @param \Drupal\cms_content_sync\Cli\CliService $cliService
   *   The CLI service which allows interoperability.
   */
  public function __construct(CliService $cliService) {
    $this->cliService = $cliService;

    parent::__construct();
  }

  /**
   * Export the configuration to the Sync Core.
   *
   * @param array $options
   *   An array containing the option parameters provided by Drush.
   *
   * @command cms_content_sync:configuration-export
   *
   * @aliases cse csce
   *
   * @options force
   *  Whether to ignore that another site is already using the same site ID.
   *  Useful if you change the URL of a site.
   * @options mode
   *  - all: Push entity types, flows, pools.
   *  - cs: Push flows, pools.
   *  - entity-types: Push entity types.
   *  - old: Use old, synchronous update mode (significantly slower, only use in case of issues as a fallback).
   *
   * @throws \Exception
   */
  public function configurationExport(
    array $options = [
      'force' => FALSE,
      'mode' => 'all',
    ],
  ) {
    $this->cliService->configurationExport($this->io(), $options);
  }

  /**
   * Pull entities for a give flow.
   *
   * Kindly ask the Sync Core to pull all entities for a specific flow, or to
   * force pull one specific entity.
   *
   * @param string $flow_id
   *   The flow the entities should be pulled from.
   * @param array $options
   *   An array containing the option parameters provided by Drush.
   *
   * @command cms_content_sync:pull
   *
   * @aliases cs-pull
   *
   * @options force
   *  Also update entities which have already been pulled.
   * @options entity_type
   *  The type of the entity that should be pulled, e.g. "node".
   * @options entity_uuid
   *  The uuid of the entity that should be pulled.
   *
   * @usage cms_content_sync:pull example_flow
   *   Pulls all entities from the example flow.
   * @usage cms_content_sync:pull example_flow --force
   *   Pull all entities from the "example_flow" and force entities which already have been pulled to be updated as well.
   * @usage cms_content_sync:pull example_flow --entity_type="node" --entity_uuid="3a150294-90eb-48c2-911d-672043a45683"
   *   Force pull the node having the uuid 3a150294-90eb-48c2-911d-672043a45683 from the example flow.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \EdgeBox\SyncCore\Exception\SyncCoreException
   */
  public function pull(
    $flow_id,
    array $options = [
    'force' => FALSE,
    'entity_type' => NULL,
    'entity_uuid' => NULL
    ],
  ) {
    $this->cliService->pull($this->io(), $flow_id, $options);
  }

  /**
   * Push all entities for a specific flow.
   *
   * @param string $flow_id
   *   The flow the entities should be pushed for.
   * @param array $options
   *   An array containing the option parameters provided by Drush.
   *
   * @command cms_content_sync:push
   *
   * @aliases cs-push
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \GuzzleHttp\Exception\GuzzleException
   *
   * @options push_mode
   *  Allows to overwrite the default push mode.
   *
   * @usage cms_content_sync:push example_flow
   *   Push all entities from the "example_flow" which have push configured as "automatically".
   * @usage cms_content_sync:push example_flow --push_mode="automatic_manual"
   *   Push all entities from the "example_flow" which have push configured as "automatically" or "manually". Only exports manually exported entities which have not been exported before.
   * @usage cms_content_sync:push example_flow --push_mode="automatic_manual_force"
   *   Push all entities from the "example_flow" which have push configured as "automatically" or "manually". Also exports entities which have not been exported before by the manual push.
   */
  public function push($flow_id, array $options = ['push_mode' => NULL]) {
    $this->cliService->push($this->io(), $flow_id, $options);
  }

  /**
   * Reset the status entities for a specific or all pool/s.
   *
   * @param array $options
   *   An array containing the option parameters provided by Drush.
   *
   * @command cms_content_sync:reset-status-entities
   *
   * @aliases csrse
   *
   * @options pool_id
   *  The machine name of the pool the status entities should be reset for.
   *
   * @usage cms_content_sync:reset-status-entities
   *   Reset all status entities for all pools.
   * @usage cms_content_sync:reset-status-entities --pool_id='example_pool'
   *   Reset all status entities for the "example_pool".
   *
   * @throws \Drush\Exceptions\UserAbortException
   */
  public function resetStatusEntities(array $options = ['pool_id' => InputOption::VALUE_OPTIONAL]) {
    $this->cliService->resetStatusEntities($this->io(), $options);
  }

  /**
   * Check the flags for an entity.
   *
   * @param string $entity_uuid
   *   The entities uuid you would like to check for.
   * @param array $options
   *   An array containing the option parameters provided by Drush.
   *
   * @command cms_content_sync:check-entity-flags
   *
   * @aliases cscef
   *
   * @options flag The flag to check for, allowed values are: FLAG_IS_SOURCE_ENTITY, FLAG_PUSH_ENABLED, FLAG_PUSHED_AS_DEPENDENCY, FLAG_EDIT_OVERRIDE, FLAG_USER_ENABLED_PUSH, FLAG_DELETED
   *
   * @usage cms_content_sync:check-entity-flags 16cc0d54-d93d-45b8-adf2-071de9d2d32b
   *   Get all flags for the entity having the uuid = "16cc0d54-d93d-45b8-adf2-071de9d2d32b".
   * @usage cms_content_sync:check-entity-flags 16cc0d54-d93d-45b8-adf2-071de9d2d32b --flag="FLAG_EDIT_OVERRIDE"
   *   Check if the entity having the uuid = "16cc0d54-d93d-45b8-adf2-071de9d2d32b" is overridden locally.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function checkEntityFlags($entity_uuid, array $options = ['flag' => InputOption::VALUE_OPTIONAL]) {
    $this->cliService->checkEntityFlags($this->io(), $entity_uuid, $options);
  }

  /**
   * Register this site to the Content Sync backend. Please visit https://app.content-sync.io/sites/register-multiple to get the required IDs and token.
   *
   * @param string $environment_type
   *   Either production, staging, testing or local.
   * @param string $contract
   *   The UUID of the contract. Please visit https://app.content-sync.io/sites/register-multiple to get the required IDs and token.
   * @param string $space
   *   The UUID of the space. Please visit https://app.content-sync.io/sites/register-multiple to get the required IDs and token.
   * @param string $token
   *   A JWT token to verify your request. Please visit https://app.content-sync.io/sites/register-multiple to get the required IDs and token.
   * @param array $options
   *   An array containing the option parameters provided by Drush.
   *
   * @command cms_content_sync:register
   *
   * @aliases csr
   *
   * @options require_fixed_ip If set to true, requests to this site will come from a fixed IP. Only use this if you need it as it will proxy requests, slowing down the update speed.
   *
   * @usage cms_content_sync:register production 00000000-0000-0000-0000-000000000000 00000000-0000-0000-0000-000000000000 eyJ...
   *   Register this site as a new production site.
   * @usage cms_content_sync:register staging 00000000-0000-0000-0000-000000000000 00000000-0000-0000-0000-000000000000 eyJ... --use_proxy=true
   *   Register this site as a new staging site that is not public and therefore requires a fixed IP.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function register(string $environment_type, string $contract, string $space, string $token, array $options = ['require_fixed_ip' => InputOption::VALUE_OPTIONAL]) {
    $this->cliService->register($this->io(), $environment_type, $contract, $space, $token, $options);
  }

}
