<?php

namespace Drupal\cms_content_sync\Helper;

use Drupal\Core\Field\FieldDefinitionInterface;

/**
 * Provides utility methods for working with field definitions.
 */
class FieldHelper {

  /**
   * Gets the allowed target bundles for an entity reference field.
   *
   * This method extracts the 'target_bundles' setting or computes the allowed
   * target bundles from the handler settings of the given field definition.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $fieldDefinition
   *   The field definition to check for target bundles.
   *
   * @return array
   *   An array of allowed target bundle IDs. If no specific bundles are
   *   defined, an empty array is returned.
   */
  public function getEntityReferenceFieldAllowedTargetBundles (FieldDefinitionInterface $fieldDefinition): array {
    // Check the field definition for the 'target_bundles' setting.
    $targetBundles = $fieldDefinition->getSetting('target_bundles');
    if (!empty($targetBundles)) {
      return $targetBundles;
    }
    // Check if the field definition has a handler_settings array.
    $handlerSettings = $fieldDefinition->getSetting('handler_settings');
    if (!empty($handlerSettings)) {
      $targetBundles = [];
      // Check if we have 'target_bundles' in the handler_settings array.
      if (!empty($handlerSettings['target_bundles'])) {
        $targetBundles = array_values($handlerSettings['target_bundles']);
      }
      // Checks if we have a setting for negating in the handler_settings array,
      // and it is enabled. This is only available in entity_reference_revisions
      // fields (paragraphs).
      if (isset($handlerSettings['negate']) && $handlerSettings['negate']) {
        // Available target bundles are stored in the 'target_bundles_drag_drop'
        // handler setting.
        $targetBundles = array_diff(array_keys($handlerSettings['target_bundles_drag_drop']), $targetBundles);
      }
      return array_values($targetBundles);
    }
    return [];
  }
}
