<?php

namespace Drupal\cms_content_sync\Form;

use Drupal\cms_content_sync\Controller\ContentSyncSettings;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\Entity\Role;
use EdgeBox\SyncCore\Interfaces\IApplicationInterface;

/**
 * Content Sync general settings form.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    global $base_url;
    $settings = ContentSyncSettings::getInstance();

    $form['cms_content_sync_base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Base URL'),
      '#default_value' => $settings->getSiteBaseUrl(TRUE),
      '#description' => $this->t('Do not include a trailing slash. By default the global base_url provided by Drupal is used for the communication between the Content Sync backend and Drupal. However, this setting allows you to override the base_url that should be used for the communication.
      Once this is set, all settings must be exported again. This can be done by either saving them, or using <i>drush cse</i>.<br>The cms content sync base url can also be set in your settings.php file by adding: <i>$settings["cms_content_sync_base_url"] = "http://example.com";</i> to it.'),
      '#attributes' => [
        'placeholder' => $base_url,
      ],
    ];

    $auth_options = [
      IApplicationInterface::AUTHENTICATION_TYPE_COOKIE => $this->t('Standard (Cookie)'),
    ];
    if (\Drupal::service('module_handler')->moduleExists('basic_auth')) {
      $auth_options[IApplicationInterface::AUTHENTICATION_TYPE_BASIC_AUTH] = $this->t('Basic Auth');
    }

    $form['cms_content_sync_authentication_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Authentication'),
      '#default_value' => $settings->getAuthenticationType(),
      '#description' => $this->t('Define how the Sync Core should authenticate against this page. Using Basic Auth can improve startup times when connecting a lot of sites and resolves issues when hiding the default /user/login route for security reasons. When using the SaaS product, we recommend using the standard Drupal login. To enable Basic Auth, please enable the basic_auth module.'),
      '#options' => $auth_options,
      '#required' => TRUE,
    ];

    $id = $settings->getSiteId();
    $site_name = '';

    try {
      $site_name = $settings->getSiteName();
    }
    catch (\Exception $e) {
      \Drupal::messenger()->addWarning(\Drupal::translation()->translate('Failed to get site name: %error', ['%error' => $e->getMessage()]));
    }

    if ($id) {
      $form['cms_content_sync_name'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Site name'),
        '#default_value' => $site_name,
        '#description' => $this->t(
                'Any name used to identify this site from other sites. Your site ID is %id.<br>
          The site name can also be set in your settings.php file by adding: <i>$settings["cms_content_sync_site_name"] = "My site name";</i> to it.<br>
          The same works for the site id by using <i>$settings["cms_content_sync_site_id"] = "my_site_id";</i><br>
          <b><i>The site name and site id must only be overwritten <u>after</u> the initial configuration export to the sync core and <u>must</u> always be unique.<br>
          If this is getting ignored, you might run into several issues which are not always undoable without wiping your data from the sync core.</i></b>',
                ['%id' => $id]
        ),
      ];
    }

    $form['cms_content_sync_disable_dynamic_pool_assignment'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('DISABLE dynamic pool assignment'),
      '#default_value' => $settings->isDynamicPoolAssignmentDisabled(),
      '#description' => $this->t('Don\'t allow editors to unassign and re-assign Pools. Content may be deleted on other sites when a Pool assignment is removed!'),
      '#suffix' => $settings->isDynamicPoolAssignmentDisabled() ? '<div class="messages messages--warning">' . $this->t('If you already pushed content on this site you have to first push all content again before enabling dynamic pool assignments! Otherwise content may be deleted from other sites!')->__toString() . '</div>' : NULL,
    ];

    $form['import_dashboard'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Pull dashboard configuration'),
    ];

    $form['import_dashboard']['cms_content_sync_enable_preview'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable preview'),
      '#default_value' => $settings->isPreviewEnabled(),
      '#description' => $this->t('If you want to pull content from this site on other sites via the UI ("Manual" pull action) and you\'re using custom Preview display modes, check this box to actually push them so they become available on remote sites.'),
    ];

    // The extended entity export/import logging is stored within the key value table
    // since we do not want this stored within the Drupal configuration.
    $form['extended_entity_export_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Extended Entity Export logging'),
      '#default_value' => $settings->getExtendedEntityExportLogging() ?? 0,
      '#description' => $this->t('When the "Extended Entity Export logging" is enabled, Content Sync is going to add a log entry to Watchdog
                                        showing all entity values processed by content sync after the <b><u>export</u></b>. This is helpful to debug outgoing entities.<br>
                                        <b>This will create many large log messages, so only use this for a short period of time and disable it immediately after your debugging session.</b>'),
    ];

    $form['extended_entity_import_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Extended Entity Import logging'),
      '#default_value' => $settings->getExtendedEntityImportLogging() ?? 0,
      '#description' => $this->t('When the "Extended Entity Import logging" is enabled, Content Sync is going to add a log entry to Watchdog
                                        showing all entity values processed by content sync after the <b><u>import</u></b>. This is helpful to identify if an entity
                                        has changed after content sync has processed it.<br>
                                        <b>This will create many large log messages, so only use this for a short period of time and disable it immediately after your debugging session.</b>'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $base_url = $form_state->getValue('cms_content_sync_base_url');

    if (!empty($base_url) && '/' === mb_substr($base_url, -1)) {
      $form_state->setErrorByName('cms_content_sync_base_url', 'Do not include a trailing slash.');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $settings = ContentSyncSettings::getInstance();
    $settings->setSiteBaseUrl($form_state->getValue('cms_content_sync_base_url'));
    $settings->setAuthenticationType($form_state->getValue('cms_content_sync_authentication_type'));
    $settings->setPreviewEnabled(
          boolval($form_state->getValue('cms_content_sync_enable_preview'))
      );

    $settings->disableDynamicPoolAssignment(
          boolval($form_state->getValue('cms_content_sync_disable_dynamic_pool_assignment'))
      );

    $settings->setExtendedEntityExportLogging($form_state->getValue('extended_entity_export_logging'));
    $settings->setExtendedEntityImportLogging($form_state->getValue('extended_entity_import_logging'));
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'cms_content_sync.settings',
    ];
  }

}
