<?php

namespace Drupal\cms_content_sync\Plugin\rest\resource;

use Drupal\cms_content_sync\Controller\LoggerProxy;
use Drupal\cms_content_sync\SyncCoreInterface\DrupalApplication;
use Drupal\Core\Extension\ExtensionList;
use Drupal\rest\Plugin\ResourceBase;
use EdgeBox\SyncCore\V2\Raw\Model\RemoteRequestQueryParamsSiteStatus;
use EdgeBox\SyncCore\V2\Raw\Model\RemoteSiteStatusRequestMode;
use EdgeBox\SyncCore\V2\Raw\Model\SiteExtension;
use EdgeBox\SyncCore\V2\Raw\Model\SiteStatusResponse;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a status for Content Sync, allowing Sync Core v2 to
 * retrieve information about this site like the Drupal version.
 *
 * @RestResource(
 *   id = "cms_content_sync_sync_core_site_status",
 *   label = @Translation("Content Sync: Sync Core: Site Status"),
 *   uri_paths = {
 *     "canonical" = "/rest/cms-content-sync/v2/status"
 *   }
 * )
 */
class SyncCoreSiteStatusResource extends ResourceBase implements ContentSyncRestInterface {
  use ContentSyncRestTrait;

  /**
   * @var \Drupal\Core\Extension\ExtensionList
   */
  protected $extensionListModule;

  /**
   * Constructs an object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param array $serializer_formats
   *   The available serialization formats.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   * @param \Drupal\Core\Extension\ExtensionList $extension_list_module
   *   An extension list instance.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    ExtensionList $extension_list_module,
  ) {
    parent::__construct(
          $configuration,
          $plugin_id,
          $plugin_definition,
          $serializer_formats,
          $logger
      );

    $this->extensionListModule = $extension_list_module;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
    $configuration,
    $plugin_id,
    $plugin_definition,
    $container->getParameter('serializer.formats'),
    LoggerProxy::get(),
    $container->get('extension.list.module')
    );
  }

  /**
   *
   */
  public function get() {
    $application = DrupalApplication::get();
    $response = new SiteStatusResponse();
    $response->setName($application->getSiteName());
    $response->setBaseUrl($application->getSiteBaseUrl());
    $response->setAppVersion($application->getApplicationVersion());
    $response->setAppModuleVersion($application->getApplicationModuleVersion());
    $response->setUuid($application->getSiteUuid());
    $response->setFeatureFlags($application->getFeatureFlags());

    $response->setDomains(_cms_content_sync_get_domains());

    $query = \Drupal::request()->query->all();
    $params = new RemoteRequestQueryParamsSiteStatus($query);

    $mode = $params->getMode() ?? RemoteSiteStatusRequestMode::ALL;

    if (($mode === RemoteSiteStatusRequestMode::ALL || $mode === RemoteSiteStatusRequestMode::EXTENSIONS) && empty(getenv("CMS_CONTENT_SYNC_DISABLE_SITE_STATUS_EXTENSIONS"))) {
      $extensions = [];

      // Provide a list of all installed modules and their versions so we can help
      // customers troubleshoot issues faster.
      $installed = \Drupal::service('extension.list.module')->getAllInstalledInfo();
      foreach ($installed as $name => $info) {
        $extension = new SiteExtension();
        $extension->setName($name);
        if (!empty($info['version'])) {
          $extension->setVersion($info['version']);
        }
        $extensions[] = $extension;
      }

      // See if we can find out what version of our library is used so we can help
      // customers identify version mismatches (forgetting to run composer with
      // the --with-dependencies flag).
      $composer_installed_path = DRUPAL_ROOT . '/../vendor/composer/installed.json';
      if (file_exists($composer_installed_path)) {
        try {
          $content = json_decode(file_get_contents($composer_installed_path), TRUE);
          $packages = $content['packages'] ?? [];
          foreach ($packages as $package) {
            if (($package['name'] ?? '') === 'edge-box/sync-core') {
              $extension = new SiteExtension();
              $extension->setName($package['name']);
              if (!empty($package['version'])) {
                $extension->setVersion($package['version']);
              }
              $extensions[] = $extension;
            }
          }
        }
        catch (\Exception $e) {
          throw $e;
        }
      }

      $response->setExtensions($extensions);
    }

    if (($mode === RemoteSiteStatusRequestMode::ALL || $mode === RemoteSiteStatusRequestMode::HEALTH) && empty(getenv("CMS_CONTENT_SYNC_DISABLE_SITE_STATUS_REQUIREMENTS"))) {
      $system_manager = \Drupal::service('system.manager');
      $requirements_raw = $system_manager->listRequirements();
      $requirements = [];
      foreach ($requirements_raw as $name => $value_raw) {
        $value = [];
        if (!empty($value_raw['title'])) {
          $value['title'] = $value_raw['title'] . '';
        }
        if (!empty($value_raw['value'])) {
          $value['value'] = $value_raw['value'] . '';
        }
        if (!empty($value_raw['description'])) {
          $value['description'] = $value_raw['description'] . '';
        }
        if (!empty($value_raw['severity'])) {
          $value['severity'] = (int) $value_raw['severity'];
        }
        if (!empty($value_raw['weight'])) {
          $value['weight'] = (int) $value_raw['weight'];
        }
        $requirements[$name] = $value;
      }

      $response->setRequirements($requirements);
    }

    $body = $response->jsonSerialize();

    // Turn objects into arrays.
    return $this->respondWith(json_decode(json_encode($body), TRUE), self::CODE_OK, FALSE);
  }

}
