<?php

namespace Drupal\cms_content_sync\Controller;

use Drupal\Core\KeyValueStore\KeyValueDatabaseFactory;
use Drupal\encrypt\Entity\EncryptionProfile;
use Drupal\user\Entity\User;
use Drupal\user\UserDataInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class AuthenticationByUser.
 *
 * Ask the Sync Core to use a local user account to authenticate against this site.
 */
class AuthenticationByUser {
  /**
   * The username key.
   *
   * @var string KEY_USERNAME
   */
  public const KEY_USERNAME = 'userName';

  /**
   * The user password key.
   *
   * @var string KEY_PASSWORD
   */
  public const KEY_PASSWORD = 'userPass';

  /**
   * The key value database.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueDatabaseFactory
   */
  protected $keyValueDatabase;

  /**
   * The user data interface.
   *
   * @var \Drupal\user\Entity\UserDataInterface
   */
  protected $userData;

  /**
   * Constructs a new EntityListBuilder object.
   *
   * @param \Drupal\Core\KeyValueStore\KeyValueDatabaseFactory $key_value_database
   *   The key value database.
   * @param \Drupal\user\Entity\UserDataInterface $user_data
   *   The user data interface.
   */
  public function __construct(KeyValueDatabaseFactory $key_value_database, UserDataInterface $user_data) {
    $this->keyValueDatabase = $key_value_database;
    $this->userData = $user_data;
  }

  /**
   * Singleton.
   */
  public static function getInstance() {
    static $instance = NULL;
    if ($instance) {
      return $instance;
    }

    return $instance = self::createInstance(\Drupal::getContainer());
  }

  /**
   * Create a new instance of the controller with the services of the given container.
   */
  public static function createInstance(ContainerInterface $container) {
    return new static(
    $container->get('keyvalue.database'),
    $container->get('user.data')
    );
  }

  /**
   * Get the login data.
   *
   * @throws \Exception
   *
   * @returns array
   *   The login data.
   */
  public function getLoginData() {
    static $loginData = NULL;

    if (!empty($loginData)) {
      return $loginData;
    }

    $user = User::load(_cms_content_sync_user_id());

    if (!$user) {
      throw new \Exception("Content Sync User not found. Encrypted data can't be read or saved.");
    }

    $loginData = $this->userData->get('cms_content_sync', $user->id(), 'sync_data');

    if (!$loginData) {
      throw new \Exception('No credentials for sync user found.');
    }

    $encryption_profile = EncryptionProfile::load(CMS_CONTENT_SYNC_ENCRYPTION_PROFILE_NAME);

    foreach ($loginData as $key => $value) {
      $loginData[$key] = \Drupal::service('encryption')
        ->decrypt($value, $encryption_profile);
    }

    return $loginData;
  }

  /**
   * Get user name.
   *
   * @return string
   *   Returns the user name.
   */
  public function getUsername() {
    return $this->getLoginData()[self::KEY_USERNAME];
  }

  /**
   * Get user password.
   *
   * @return string
   *   Returns the user password.
   */
  public function getPassword() {
    return $this->getLoginData()[self::KEY_PASSWORD];
  }

}
