<?php

namespace Drupal\cms_content_sync\Form;

use Drupal\cms_content_sync\Entity\EntityStatus;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Entity\EntityConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Builds the form to delete an Flow.
 */
class FlowDeleteForm extends EntityConfirmFormBase {
  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The config factory to load configuration.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $configFactory;

  /**
   * FlowDeleteForm constructor.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Config\ConfigFactory $config_factory
   *   The config factory.
   */
  public function __construct(MessengerInterface $messenger, ConfigFactory $config_factory) {
    $this->messenger = $messenger;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
    $container->get('messenger'),
    $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete %name? This will also delete all synchronisation status entities!', ['%name' => $this->entity->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $form['keep_status_entities'] = [
      '#type' => 'checkbox',
      '#title' => t('Keep status entities. <strong>This may cause Exceptions until a Flow with the same machine name is re-created. You should only use this to migrate between different Flow variants with the same purpose.</strong>'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $keep_status_entities = boolval($form_state->getValue('keep_status_entities'));
    if (!$keep_status_entities) {
      // Delete config related status entities.
      $entity_status = \Drupal::entityTypeManager()->getStorage('cms_content_sync_entity_status')
        ->loadByProperties(['flow' => $this->getEntity()->id()]);

      foreach ($entity_status as $status) {
        $entity = EntityStatus::load($status->id());
        $entity->delete();
      }
    }

    $links = \Drupal::entityTypeManager()->getStorage('menu_link_content')
      ->loadByProperties(['link__uri' => 'internal:/admin/content/cms_content_synchronization/' . $this->entity->id()]);

    if ($link = reset($links)) {
      $link->delete();
      \Drupal::cache('menu')->invalidateAll();
    }

    $moduleHandler = \Drupal::service('module_handler');
    if ($moduleHandler->moduleExists('cms_content_sync_developer')) {
      $config_factory = $this->configFactory;
      $developer_config = $config_factory->getEditable('cms_content_sync.developer');
      $mismatching_versions = $developer_config->get('version_mismatch');
      if (!empty($mismatching_versions)) {
        unset($mismatching_versions[$this->entity->id()]);
        $developer_config->set('version_mismatch', $mismatching_versions)->save();
      }
    }

    $this->entity->delete();
    $this->messenger->addMessage($this->t('Flow %label has been deleted.', ['%label' => $this->entity->label()]));

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.cms_content_sync_flow.collection');
  }

}
