<?php

namespace Drupal\cms_content_sync\Plugin\cms_content_sync\entity_handler;

use Drupal\cms_content_sync\Plugin\cms_content_sync\field_handler\DefaultFormattedTextHandler;
use Drupal\cms_content_sync\PullIntent;
use Drupal\cms_content_sync\PushIntent;
use Drupal\cms_content_sync\SyncIntent;
use Drupal\Core\Entity\EntityInterface;
use EdgeBox\SyncCore\Interfaces\Configuration\IDefineEntityType;
use EdgeBox\SyncCore\V2\Configuration\DefineProperty;

/**
 * Support Cohesion / Site Studio layouts.
 *
 * @EntityHandler(
 *   id = "cms_content_sync_default_cohesion_layout_handler",
 *   label = @Translation("Default Cohesion Layout"),
 *   weight = 90
 * )
 */
class DefaultCohesionLayoutHandler extends DefaultContentEntityHandler {

  /**
   * {@inheritdoc}
   */
  public static function supports($entity_type, $bundle) {
    // Whitelist supported entity types.
    $entity_types = [
      'cohesion_layout',
    ];

    return in_array($entity_type, $entity_types);
  }

  /**
   * {@inheritdoc}
   */
  public function updateEntityTypeDefinition(IDefineEntityType|DefineProperty &$definition) {
    $definition->addReferenceProperty('dependencies', 'Dependencies', TRUE, FALSE, 'cohesion_dependencies')->setLocalized(TRUE)->setShared(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function pull(PullIntent $intent) {
    if (SyncIntent::ACTION_DELETE === $intent->getAction()) {
      return parent::pull($intent);
    }

    // Serialized data will be fine as Cohesion is also using UUIDs for all references.
    // But we need to go through the referenced entities and import them as dependencies
    // so that when saving the layout all references can be resolved.
    $dependencies = $intent->getProperty('dependencies') ?? [];
    foreach ($dependencies as $dependency) {
      $intent->loadEmbeddedEntity($dependency);
    }

    return parent::pull($intent);
  }

  /**
   * {@inheritdoc}
   */
  public function push(PushIntent $intent, ?EntityInterface $entity = NULL) {
    $result = parent::push($intent, $entity);

    if (!$result || SyncIntent::ACTION_DELETE === $intent->getAction()) {
      return $result;
    }

    /**
     * @var \Drupal\cohesion_elements\Entity\CohesionLayout $entity
     */
    if (!$entity) {
      $entity = $intent->getEntity();
    }

    // Parent will serialize fine as Cohesion is also using UUIDs for all references.
    // But we need to go through the referenced entities and add them as dependencies
    // so they appear at the target site as well.
    $json = json_decode($entity->getJsonValues(), TRUE);
    $model = $json['model'];
    $dependencies = [];
    foreach ($model as /*$component_uuid=>*/$component) {
      foreach ($component as /*$property_uuid=>*/$property) {
        if (!empty($property['entity']['entityType']) && !empty($property['entity']['entityId'])) {
          if (isset($property['entity']['entityUUID'])) {
            $repository = \Drupal::service('entity.repository');
            $entity = $repository->loadEntityByUuid($property['entity']['entityType'], $property['entity']['entityUUID']);
          }
          else {
            $storage = \Drupal::entityTypeManager()->getStorage($property['entity']['entityType']);
            $entity = $storage->load($property['entity']['entityId']);
          }

          if ($entity) {
            $dependencies[] = $intent->embed($entity);
          }
        }
        // Text may also have entities embedded that we need to add as dependencies.
        elseif (!empty($property['text']) && !empty($property['textFormat'])) {
          $dependencies = array_merge($dependencies, DefaultFormattedTextHandler::extractDependenciesFromHtml($intent, $property['text'], $this->logger));
        }
      }
    }

    $intent->setProperty('dependencies', $dependencies);

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function getForbiddenFields() {
    // Styles and template are calculated in the pre-save hook of the cohesion
    // layout, so we don't need to sync them.
    return array_merge(parent::getForbiddenFields(), ['styles', 'template']);
  }

}
