<?php

namespace Drupal\cms_content_sync_health\Controller;

use Drupal\cms_content_sync\Entity\Flow;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\Core\Entity\EntityTypeBundleInfo;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Show all version mismatches of all entity types.
 *
 * Can take some time toexecute depending on the number of entity types
 * configured, so is run as a batch operation.
 */
class VersionMismatches extends ControllerBase {

  /**
   * {@inheritdoc}
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfo
   */
  protected $entityTypeBundleInfo;

  /**
   * VersionMismatches constructor to handle dependency injection.
   */
  public function __construct(EntityTypeBundleInfo $entityTypeBundleInfo) {
    $this->entityTypeBundleInfo = $entityTypeBundleInfo;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
    $container->get('entity_type.bundle.info')
    );
  }

  /**
   * Prepare batch operation.
   */
  public function aggregate() {
    $operations = [];

    $entity_types = $this->entityTypeBundleInfo->getAllBundleInfo();
    ksort($entity_types);
    foreach ($entity_types as $type_key => $entity_type) {
      if ('cms_content_sync' == substr($type_key, 0, 16)) {
        continue;
      }

      ksort($entity_type);

      foreach ($entity_type as $entity_bundle_name => $entity_bundle) {
        $any_handler = FALSE;
        foreach (Flow::getAll() as $flow) {
          $config = $flow->getController()->getEntityTypeConfig($type_key, $entity_bundle_name, TRUE);
          if (empty($config) || Flow::HANDLER_IGNORE == $config['handler']) {
            continue;
          }
          $any_handler = TRUE;

          break;
        }

        if (!$any_handler) {
          continue;
        }

        $operations[] = [
          '\Drupal\cms_content_sync_health\Controller\VersionMismatches::batch',
              [$type_key, $entity_bundle_name],
        ];
      }
    }

    $batch = [
      'title' => $this->t('Check version mismatches...'),
      'operations' => $operations,
      'finished' => '\Drupal\cms_content_sync_health\Controller\VersionMismatches::batchFinished',
    ];

    batch_set($batch);

    return batch_process();
  }

  /**
   * Batch push finished callback.
   *
   * @param bool $success
   *   Indicate that the batch API tasks were all completed successfully.
   * @param array $results
   *   An array of all the results that were updated.
   * @param array $operations
   *   A list of all the operations that had not been completed by the
   *   batch API.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect the user to the sync health overview.
   */
  public static function batchFinished($success, array $results, array $operations) {
    $list = _cms_content_sync_display_entity_type_differences_recursively_render($results);

    if (empty($list)) {
      \Drupal::messenger()->addStatus(\Drupal::translation()->translate('No differences found; all other connected sites use the same entity type definition as this site.'));
    }
    else {
      \Drupal::messenger()->addError(\Drupal::translation()->translate('Some connected sites use other entity type definitions than this site:<br><br>@list', ['@list' => new FormattableMarkup($list, [])]));
    }

    return new RedirectResponse(Url::fromRoute('entity.cms_content_sync.sync_health')->toString());
  }

  /**
   * Batch push callback for the push-all operation.
   *
   * @param string $type_key
   *   The entity type name.
   * @param string $entity_bundle_name
   *   The entity bundle name.
   * @param array $context
   *   The batch context.
   */
  public static function batch($type_key, $entity_bundle_name, array &$context) {
    $message = 'Checking ' . $type_key . '.' . $entity_bundle_name . '...';
    $results = [];
    if (isset($context['results'])) {
      $results = $context['results'];
    }

    _cms_content_sync_display_entity_type_differences_recursively($results, $type_key, $entity_bundle_name);

    $context['message'] = $message;
    $context['results'] = $results;
  }

}
