<?php

namespace Drupal\Tests\cognito\Functional;

use Drupal\user\Entity\User;

/**
 * Test editing a users account.
 *
 * @group cognito
 */
class EmailFlowUserEditTest extends CognitoTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'cognito',
    'cognito_tests',
  ];

  /**
   * The user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $user;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->user = $this->createExternalUser([
      'administer permissions',
      'administer users',
    ], ['name' => 'Ben']);
    $this->drupalGet('/user/login');

    $this->submitForm([
      'mail' => $this->user->getEmail(),
      'pass' => 'letmein',
    ], 'Log in');
  }

  /**
   * Test editing a users account.
   */
  public function testEditUserAccount() {
    $this->drupalGet($this->user->toUrl('edit-form'));
    $this->assertSession()->fieldValueEquals('mail', $this->user->getEmail());
    $this->assertSession()->fieldNotExists('name');
    $this->drupalGet($this->user->toUrl('edit-form'));

    // Change the users password.
    $this->submitForm([
      'current_pass' => 'letmein',
      'pass[pass1]' => 'letmein-new',
      'pass[pass2]' => 'letmein-new',
    ], 'Save');

    $this->assertSession()->pageTextContains('Your account has been updated');
    $this->drupalGet($this->user->toUrl('edit-form'));

    // If they save the form without providing a new password then nothing
    // changes.
    $this->submitForm([
      'pass[pass1]' => 'letmein-new',
      'pass[pass2]' => 'letmein-new',
    ], 'Save');
  }

  /**
   * Test updating the users email from the UI.
   */
  public function testUpdateUserEmail() {
    $this->drupalGet($this->user->toUrl('edit-form'));
    // Update the users email address.
    $this->submitForm([
      'current_pass' => 'letmein',
      'mail' => $new_email = 'new-email@example.com',
    ], 'Save');

    // Re-load the user and ensure the email was updated.
    $account = User::load($this->user->id());
    $this->assertEquals($new_email, $account->getEmail());

    // Ensure the username was kept in sync.
    $this->assertEquals($new_email, $account->getDisplayName());

    // Ensure that externalauth also has the updated username.
    $this->assertEquals($account->id(), \Drupal::service('externalauth.externalauth')->load($new_email, 'cognito')->id());
  }

}
