<?php

namespace Drupal\combined_image_style_formatters\Plugin\Field\FieldFormatter;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityRepository;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceEntityFormatter;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\ProxyClass\File\MimeType\ExtensionMimeTypeGuesser;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\TranslatableInterface;
use Drupal\breakpoint\BreakpointManagerInterface;
use Drupal\combined_image_style_formatters\CombinedImageStyleFormatterTrait;
use Drupal\file\FileInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin for responsive image formatter.
 */
#[FieldFormatter(
  id: 'responsive_media_combined_image_style',
  label: new TranslatableMarkup('Responsive media (Combined image style)'),
  field_types: [
    'entity_reference',
  ],
)]
class ResponsiveMediaFormatter extends EntityReferenceEntityFormatter {

  use CombinedImageStyleFormatterTrait;

  /**
   * Constructs an EntityReferenceEntityFormatter instance.
   *
   * @param string $plugin_id
   *   The plugin ID for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository.
   * @param \Drupal\Core\Breakpoint\BreakpointManagerInterface $breakpointManager
   *   The breakpoint manager service.
   * @param \Drupal\Core\ProxyClass\File\MimeType\ExtensionMimeTypeGuesser $mimeTypeGuesser
   *   The extension MIME type guesser service.
   * @param \Drupal\Core\Entity\EntityRepository $entityRepository
   *   The entity repository service.
   */
  public function __construct(
    string $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    string $label,
    string $view_mode,
    array $third_party_settings,
    LoggerChannelFactoryInterface $logger_factory,
    EntityTypeManagerInterface $entity_type_manager,
    EntityDisplayRepositoryInterface $entity_display_repository,
    protected BreakpointManagerInterface $breakpointManager,
    protected ExtensionMimeTypeGuesser $mimeTypeGuesser,
    protected EntityRepository $entityRepository,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings, $logger_factory, $entity_type_manager, $entity_display_repository);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('logger.factory'),
      $container->get('entity_type.manager'),
      $container->get('entity_display.repository'),
      $container->get('breakpoint.manager'),
      $container->get('file.mime_type.guesser.extension'),
      $container->get('entity.repository'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $elements = [];
    $files = $this->getEntitiesToView($items, $langcode);

    foreach ($files as $delta => $file) {
      $item = $items[$delta];

      if ($file instanceof FileInterface) {
        $elements[$delta] = $this->viewElement($item, $file);
      }
      else {
        $view_builder = $this->entityTypeManager->getViewBuilder('media');
        $elements[$delta] = $view_builder->view($file, 'default');
      }
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEntitiesToView(EntityReferenceFieldItemListInterface $items, $langcode): array {
    $entities = [];

    /**
     * @var \Drupal\media\MediaInterface $item
     */
    foreach ($items->referencedEntities() as $delta => $item) {
      if ($entity = $item->get($item->getSource()->configuration['source_field'])->entity) {
        // Set the entity in the correct language for display.
        if ($entity instanceof TranslatableInterface) {
          $entity = $this->entityRepository->getTranslationFromContext($entity, $langcode);
        }

        $access = $this->checkAccess($entity);
        // Add the access result's cacheability, ::view() needs it.
        $item->_accessCacheability = CacheableMetadata::createFromObject($access);
        if ($access->isAllowed()) {
          // Get the field definitions of the entity.
          $fieldDefinitions = $item->getFieldDefinitions();
          // Iterate through the field definitions.
          foreach ($fieldDefinitions as $fieldDefinition) {
            // Check if the field definition is of type 'image'
            // and get the field name.
            if ($fieldDefinition->getType() === 'image') {
              $fieldName = $fieldDefinition->getName();
              if ($item->hasField($fieldName) && !$item->get($fieldName)->isEmpty()) {
                // Get the field's value.
                $fieldValueArray = $item->get($fieldName)->getValue();
                $fieldValue = reset($fieldValueArray);
              }
            }
          }
          // Add the attributes to the item.
          $items[$delta]->alt = $fieldValue['alt'] ?? '';
          $items[$delta]->width = $fieldValue['width'] ?? '';
          $items[$delta]->height = $fieldValue['height'] ?? '';
          $items[$delta]->title = $fieldValue['title'] ?? '';
          $items[$delta]->_attributes = $fieldValue['_attributes'] ?? '';
          // Add the referring item, in case the formatter needs it.
          $entity->_referringItem = $items[$delta];
          $entities[$delta] = $entity;
        }
      }
      else {
        $entities[$delta] = $item;
      }
    }

    return $entities;
  }

}
