<?php

namespace Drupal\combined_image_style\Entity;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\image\Entity\ImageStyle as ImageStyleBase;

/**
 * Provides an extended ImageStyle class for combined image styles.
 *
 * This class extends the core ImageStyle class to provide additional
 * functionality specific to combined image styles.
 */
class ImageStyle extends ImageStyleBase {

  /**
   * {@inheritdoc}
   *
   * Implements postSave() for the responsive image field item.
   *
   * Overrides the parent method to prevent the execution of parent::postSave().
   * Invalidates cache tags after saving the entity to ensure accurate caching.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage handler.
   * @param bool $update
   *   (Optional) TRUE if the entity is being updated,
   *   FALSE if it is being inserted.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   Thrown if there was an error saving the entity.
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE) {
    // To prevent the parent::postSave().
    $this->invalidateTagsOnSave($update);

    if ($update) {
      if (!empty($this->original) && $this->id() !== $this->original->id()) {
        // The old image style name needs flushing after a rename.
        $this->original->flush();
        // Update field settings if necessary.
        if (!$this->isSyncing()) {
          static::replaceImageStyle($this);
        }
      }
      elseif ($this->getEffects()->getConfiguration() !== $this->original->getEffects()->getConfiguration()) {
        // Flush image style only when effects configuration changed.
        $this->flush();
      }
    }
  }

  /**
   * {@inheritdoc}
   *
   * Flushes the cached data for this responsive image field item.
   *
   * If a specific path is provided, only the cached data
   * associated with that path
   * will be cleared. Otherwise, all cached data for
   * this responsive image field item
   * and its corresponding combined image styles will be cleared.
   *
   * @param string|null $path
   *   (Optional) The specific path for which to clear the cached data.
   *   If not provided,
   *   all cached data for this responsive image field item and
   *   its combined image styles will be cleared.
   *
   * @return $this
   *   The called entity.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   Thrown if there was an error flushing the cached data.
   */
  public function flush($path = NULL) {
    foreach (CombinedImageStyle::loadCombinedBySingle($this) as $combinedImageStyle) {
      $combinedImageStyle->flush($path);
    }

    return parent::flush($path);
  }

}
