<?php

namespace Drupal\combined_image_style\Entity;

use Drupal\Component\Utility\Crypt;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Routing\RequestHelper;
use Drupal\Core\StreamWrapper\StreamWrapperInterface;
use Drupal\Core\Url;
use Drupal\image\Entity\ImageStyle;
use Drupal\image\ImageEffectInterface;
use Drupal\image\ImageEffectPluginCollection;
use Drupal\image\ImageStyleInterface;
use Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException;

/**
 * Defines a combined image style entity.
 *
 * Represents a combination of individual image styles for responsive images.
 */
class CombinedImageStyle extends ImageStyle {

  // Constant for separator.
  public const SEPARATOR = '-';

  /**
   * The source URI of the combined image style.
   *
   * @var string
   */
  private string $sourceUri;

  /**
   * An array of image styles associated with the combined image style.
   *
   * @var array
   */
  private array $imageStyles = [];

  /**
   * {@inheritdoc}
   */
  // phpcs:disable
  public function __construct(array $values = [], string $entity_type = '') {
    parent::__construct($values, $entity_type);
  }
  // phpcs:enable

  /**
   * Create a CombinedImageStyle entity from a combined style name.
   *
   * @param string $name
   *   The combined image style name in the format 'style1__style2'.
   *
   * @return \Drupal\combined_image_style\Entity\CombinedImageStyle
   *   A CombinedImageStyle entity.
   */
  public static function fromName(string $name): CombinedImageStyle {
    $imageStyleIds = explode(self::SEPARATOR, $name);
    $imageStyles = ImageStyle::loadMultiple($imageStyleIds);

    return (new static())
      ->setImageStyles($imageStyles);
  }

  /**
   * Load combined image styles that use a single image style.
   *
   * @param \Drupal\image\ImageStyleInterface $imageStyle
   *   The image style to search for in combined image styles.
   *
   * @return array
   *   An array of CombinedImageStyle entities that use the given image style.
   */
  public static function loadCombinedBySingle(ImageStyleInterface $imageStyle): array {
    $matches = [];

    $wrappers = \Drupal::service('stream_wrapper_manager')->getWrappers(StreamWrapperInterface::WRITE_VISIBLE);
    foreach ($wrappers as $wrapper => $wrapper_data) {
      if (!file_exists($wrapper . '://styles')) {
        continue;
      }

      // Find image style folders matching the mask.
      $mask = '/.*' . $imageStyle->id() . '.*/';
      $matches += \Drupal::service('file_system')->scanDirectory($wrapper . '://styles', $mask, ['recurse' => FALSE]);
    }

    // Load combined image styles.
    return array_values(array_map(static function (\stdClass $match) {
      return self::fromName($match->name);
    }, $matches));
  }

  /**
   * Get the unique identifier for this combined image style.
   *
   * @return string
   *   The unique identifier for the combined image style.
   */
  public function id(): string {
    return implode(self::SEPARATOR, array_map(static function (ImageStyleInterface $imageStyle) {
      return $imageStyle->id();
    }, $this->getImageStyles()));
  }

  /**
   * Get the source URI for the combined image style.
   *
   * @return string
   *   The source URI for the combined image style.
   */
  public function getSourceUri(): string {
    return $this->sourceUri;
  }

  /**
   * Set the source URI for the combined image style.
   *
   * @param string $sourceUri
   *   The source URI to set.
   *
   * @return \Drupal\combined_image_style\Entity\CombinedImageStyle
   *   The calling entity.
   */
  public function setSourceUri(string $sourceUri): CombinedImageStyle {
    $this->sourceUri = $sourceUri;
    return $this;
  }

  /**
   * Get the array of image styles used in this combined image style.
   *
   * @return \Drupal\image\ImageStyleInterface[]
   *   An array of image styles used in the combined image style.
   */
  public function getImageStyles(): array {
    return $this->imageStyles;
  }

  /**
   * Set the image styles for the combined image style.
   *
   * @param array $imageStyles
   *   An array of image styles to set.
   *
   * @return \Drupal\combined_image_style\Entity\CombinedImageStyle
   *   The calling entity.
   */
  public function setImageStyles(array $imageStyles): CombinedImageStyle {
    $this->imageStyles = array_filter(array_map(static function ($imageStyle) {
      return ($imageStyle instanceof ImageStyleInterface ? $imageStyle : ImageStyle::load($imageStyle));
    }, $imageStyles));
    return $this;
  }

  /**
   * Add an image style to the combined image style.
   *
   * @param mixed $imageStyle
   *   The image style to add.
   *
   * @return \Drupal\combined_image_style\Entity\CombinedImageStyle
   *   The calling entity.
   */
  public function addImageStyle($imageStyle): CombinedImageStyle {
    $this->imageStyles[] = ($imageStyle instanceof ImageStyleInterface ? $imageStyle : ImageStyle::load($imageStyle));
    return $this;
  }

  /**
   * Check if the combined image style has any image styles.
   *
   * @return bool
   *   True if the combined image style has image styles, false otherwise.
   */
  public function hasImageStyles(): bool {
    return !empty($this->imageStyles);
  }

  /**
   * Gets the collection of image effects used by the combined image styles.
   *
   * @return \Drupal\image\ImageEffectPluginCollection
   *   The collection of image effects.
   *
   * @throws \Exception
   */
  public function getEffects(): ImageEffectPluginCollection {
    if (!$this->effectsCollection) {
      $effects = array_map(static function (ImageStyle $imageStyle) {
        return array_map(static function (ImageEffectInterface $effect) {
          return $effect->getConfiguration();
        }, iterator_to_array($imageStyle->getEffects()->getIterator()));
      }, $this->getImageStyles());

      $effects = array_merge(...array_values($effects));
      $this->effectsCollection = new ImageEffectPluginCollection($this->getImageEffectPluginManager(), $effects);
    }

    return $this->effectsCollection;
  }

  /**
   * {@inheritdoc}
   */
  public function getPathToken($uri): string {
    $uriWithExtension = $this->addExtension($uri);

    return implode(self::SEPARATOR, array_map(static function (ImageStyleInterface $imageStyle) use ($uriWithExtension) {
      return substr(Crypt::hmacBase64($imageStyle->id() . ':' . $uriWithExtension, $imageStyle->getPrivateKey() . $imageStyle->getHashSalt()), 0, 8);
    }, $this->getImageStyles()));
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTagsToInvalidate(): array {
    return array_merge(...array_values(array_map(static function (ImageStyleInterface $imageStyle) {
      return ['config:' . $imageStyle->getConfigDependencyName()];
    }, $this->getImageStyles())));
  }

  /**
   * Retrieves the dimensions (width and height) of an image.
   *
   * @param string $uri
   *   The URI of the image.
   *
   * @return array
   *   An array containing 'width' and 'height' keys representing the dimensions
   *   of the image.
   */
  public function getDimensions(string $uri): array {
    $cache = \Drupal::cache('image_dimensions');
    $derivativeUri = $this->buildUri($this->sourceUri);

    // Load from cache.
    if ($cacheEntry = $cache->get($derivativeUri)) {
      return $cacheEntry->data;
    }

    // Calculate dimensions after effects.
    $resource = \Drupal::service('image.factory')->get($uri);
    $dimensions = [
      'width' => $resource->getWidth(),
      'height' => $resource->getHeight(),
    ];
    $this->transformDimensions($dimensions, $uri);

    // Set cache and return.
    $cache->set($derivativeUri, $dimensions);
    return $dimensions;
  }

  /**
   * Gets the width of the image represented by this CombinedImageStyle object.
   *
   * @return int
   *   The width of the image in pixels. Returns 0 if the width cannot be
   *   determined or if the source image URI is invalid.
   */
  public function getWidth(): int {
    return $this->getDimensions($this->sourceUri)['width'] ?? 0;
  }

  /**
   * Gets the height of the image.
   *
   * @return int
   *   The height of the image.
   */
  public function getHeight(): int {
    return $this->getDimensions($this->sourceUri)['height'] ?? 0;
  }

  /**
   * Builds the combined URI for the image.
   *
   * @return string
   *   The combined URI for the image.
   */
  public function buildCombinedUri(): string {
    return $this->buildUri($this->sourceUri);
  }

  /**
   * Builds the combined URL for the image.
   *
   * @param bool|null $clean_urls
   *   Whether to use clean URLs.
   */
  public function buildCombinedUrl(?bool $clean_urls = NULL): string {
    $uri = $this->buildCombinedUri();

    /** @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager */
    $stream_wrapper_manager = \Drupal::service('stream_wrapper_manager');

    // The token query is added even if the
    // 'image.settings:allow_insecure_derivatives' configuration is TRUE, so
    // that the emitted links remain valid if it is changed back to the default
    // FALSE. However, sites which need to prevent the token query from being
    // emitted at all can additionally set the
    // 'image.settings:suppress_itok_output' configuration to TRUE to achieve
    // that (if both are set, the security token will neither be emitted in the
    // image derivative URL nor checked for in
    // \Drupal\image\ImageStyleInterface::deliver()).
    $token_query = [];
    if (!\Drupal::config('image.settings')->get('suppress_itok_output')) {
      $path = $this->getSourceUri();

      // The passed $path variable can be either a relative path or a full URI.
      if (!$stream_wrapper_manager::getScheme($path)) {
        $path = \Drupal::config('system.file')->get('default_scheme') . '://' . $path;
      }
      $original_uri = $stream_wrapper_manager->normalizeUri($path);
      $token_query = [IMAGE_DERIVATIVE_TOKEN => $this->getPathToken($original_uri)];
    }

    if ($clean_urls === NULL) {
      // Assume clean URLs unless the request tells us otherwise.
      $clean_urls = TRUE;
      try {
        $request = \Drupal::request();
        $clean_urls = RequestHelper::isCleanUrl($request);
      }
      catch (ServiceNotFoundException $e) {
      }
    }

    // If not using clean URLs, the image derivative callback is only available
    // with the script path. If the file does not exist, use Url::fromUri() to
    // ensure that it is included. Once the file exists it's fine to fall back
    // to the actual file path, this avoids bootstrapping PHP once the files are
    // built.
    if ($clean_urls === FALSE && $stream_wrapper_manager::getScheme($uri) == 'public' && !file_exists($uri)) {
      $directory_path = $stream_wrapper_manager->getViaUri($uri)->getDirectoryPath();
      return Url::fromUri('base:' . $directory_path . '/' . $stream_wrapper_manager::getTarget($uri), [
        'absolute' => TRUE,
        'query' => $token_query,
      ])->toString();
    }

    /** @var \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator */
    $file_url_generator = \Drupal::service('file_url_generator');
    $file_url = $file_url_generator->generateAbsoluteString($uri);
    // Append the query string with the token, if necessary.
    if ($token_query) {
      $file_url .= (strpos($file_url, '?') !== FALSE ? '&' : '?') . UrlHelper::buildQuery($token_query);
    }

    return $file_url;
  }

  /**
   * {@inheritdoc}
   */
  public function flush($path = NULL) {
    if (isset($this->sourceUri)) {
      // Invalidate image dimensions cache.
      $derivativeUri = $this->buildUri($this->sourceUri);
      \Drupal::cache('image_dimensions')->delete($derivativeUri);
    }
    return parent::flush($path);
  }

  /**
   * {@inheritdoc}
   *
   * Creates a derivative image file based on the original and stores it at the
   * specified derivative URI.
   *
   * @param string $original_uri
   *   The URI of the original image file.
   * @param string $derivative_uri
   *   The URI where the derivative image file will be stored.
   * @param bool $overwrite
   *   (Optional) If TRUE, the derivative will overwrite an existing file at
   *   the derivative URI. Defaults to FALSE.
   *
   * @return bool
   *   TRUE if the derivative image file was successfully created and stored,
   *   FALSE otherwise.
   *
   * @throws \Exception
   *   If an error occurs during derivative creation.
   */
  public function createDerivative($original_uri, $derivative_uri, $overwrite = FALSE): bool {
    $created = ($overwrite || !file_exists($derivative_uri)) && parent::createDerivative($original_uri, $derivative_uri);

    if ($created) {
      // Invalidate image dimensions cache when new file gets written.
      \Drupal::cache('image_dimensions')->delete($derivative_uri);
    }

    return file_exists($derivative_uri);
  }

  /**
   * Converts the combined image data into a renderable array for theming.
   *
   * @return array
   *   A renderable array for the combined image.
   */
  public function toImage(): array {
    return [
      '#theme' => 'image',
      '#width' => $this->getWidth(),
      '#height' => $this->getHeight(),
      '#uri' => $this->buildCombinedUrl(),
    ];
  }

}
