<?php

/**
 * @file
 * Post update functions for ComfyUI module.
 */

/**
 * Install media entity types and fields for ComfyUI images.
 */
function comfyui_post_update_install_media_entities(&$sandbox) {
  // Step 1: Import configs
  \Drupal::service('config.installer')->installDefaultConfig('module', 'comfyui');
  
  // Step 2: Clear caches (this is needed for Drupal to recognize new field storages)
  drupal_flush_all_caches();
  
  return t('Installed ComfyUI media configurations.');
}

/**
 * Create ComfyUI file directories.
 */
function comfyui_post_update_create_file_directories(&$sandbox) {
  $helper = \Drupal::service('comfyui.file_scheme_helper');
  
  try {
    // Create input directory
    $input_dir = $helper->getInputDirectory();
    \Drupal::service('file_system')->prepareDirectory(
      $input_dir,
      \Drupal\Core\File\FileSystemInterface::CREATE_DIRECTORY | 
      \Drupal\Core\File\FileSystemInterface::MODIFY_PERMISSIONS
    );
    
    // Create generated directory
    $generated_dir = $helper->getGeneratedDirectory();
    \Drupal::service('file_system')->prepareDirectory(
      $generated_dir,
      \Drupal\Core\File\FileSystemInterface::CREATE_DIRECTORY | 
      \Drupal\Core\File\FileSystemInterface::MODIFY_PERMISSIONS
    );
    
    return t('Created ComfyUI file directories: @input and @generated', [
      '@input' => $input_dir,
      '@generated' => $generated_dir,
    ]);
  } catch (\Exception $e) {
    return t('Warning: Could not create directories. Manual creation may be needed: @error', [
      '@error' => $e->getMessage(),
    ]);
  }
}

/**
 * Install bypass mapping service and enable node/group bypass functionality.
 * 
 * This post-update function:
 * 1. Clears caches to register the new bypass mapping service
 * 2. Logs the addition of bypass functionality
 */
function comfyui_post_update_install_bypass_mapping_service(&$sandbox) {
  // Clear caches to ensure the new service is registered
  drupal_flush_all_caches();
  
  \Drupal::logger('comfyui')->info('Bypass mapping service installed. Nodes and groups can now be bypassed via Boolean field configuration.');
  
  return t('Installed ComfyUI bypass mapping service. Nodes and groups can now be conditionally bypassed during workflow execution.');
}

/**
 * Ensure bypass_mappings field is properly initialized on all workflows.
 */
function comfyui_post_update_initialize_bypass_mappings(&$sandbox) {
  // CRITICAL: Rebuild entity and field definitions FIRST
  // Otherwise Drupal won't know about the new bypass_mappings field
  \Drupal::service('entity_type.manager')->clearCachedDefinitions();
  \Drupal::service('entity_field.manager')->clearCachedFieldDefinitions();
  
  $entity_storage = \Drupal::entityTypeManager()->getStorage('comfyui_workflow');
  
  // Load all workflows - MUST explicitly set accessCheck
  $workflow_ids = $entity_storage->getQuery()
    ->accessCheck(FALSE)
    ->execute();
  
  $workflows = $entity_storage->loadMultiple($workflow_ids);
  
  $count = 0;
  foreach ($workflows as $workflow) {
    // Ensure bypass_mappings field is set (even if empty)
    if (empty($workflow->getBypassMappings())) {
      $workflow->setBypassMappings([
        'group_bypasses' => [],
        'node_bypasses' => [],
      ]);
      $workflow->save();
      $count++;
    }
  }
  
  return t('Initialized bypass_mappings field on @count workflows.', ['@count' => $count]);
}
