/**
 * ComfyUI Media Selector Integration
 */
(function ($, Drupal, once) {
  'use strict';

  Drupal.behaviors.comfyUIMediaSelector = {
    attach: function (context, settings) {
      // Handle "Select/Change Media" button - opens media library browser
      $(once('comfyui-media-select', '.comfyui-media-select-btn', context)).on('click', function(e) {
        e.preventDefault();
        
        const button = $(this);
        const fieldName = button.data('field-name');
        
        // Show media library browser
        showMediaLibraryBrowser(fieldName);
      });

      // Handle "Upload New" button - opens upload dialog
      $(once('comfyui-media-upload', '.comfyui-media-upload-btn', context)).on('click', function(e) {
        e.preventDefault();
        
        const button = $(this);
        const fieldName = button.data('field-name');
        
        // Show upload dialog for new file
        showUploadDialog(fieldName);
      });
    }
  };

  /**
   * Show upload dialog for new media.
   */
  function showUploadDialog(fieldName) {
    // Create simple upload form in modal
    const fieldSettings = drupalSettings.comfyui?.comfyui_media_fields?.[fieldName] || {};
    const maxFilesize = fieldSettings.max_filesize || 10; // MB

    const uploadHtml = `
      <div class="comfyui-upload-modal" data-field="${fieldName}">
        <div class="modal-content">
          <button type="button" class="close-modal" style="float:right; font-size:24px; border:none; background:none; cursor:pointer;">×</button>
          <h3>Upload New Input Image</h3>
          <p style="font-size: 12px; color: #666;">Max file size: ${maxFilesize}MB</p>
          <form id="comfyui-upload-form-${fieldName}" enctype="multipart/form-data">
            <input type="file" name="image" accept="image/*" required />
            <input type="text" name="name" placeholder="Image name (optional)" />
            <button type="submit" class="button button--primary">Upload</button>
            <button type="button" class="button button--secondary close-modal">Cancel</button>
          </form>
        </div>
      </div>
    `;
    
    $('body').append(uploadHtml);
    
    // Handle upload submission
    $(`#comfyui-upload-form-${fieldName}`).on('submit', function(e) {
      e.preventDefault();
      handleMediaUpload(this, fieldName);
    });
    
    // Handle cancel
    $(`.comfyui-upload-modal[data-field="${fieldName}"] .close-modal`).on('click', function() {
      $(`.comfyui-upload-modal[data-field="${fieldName}"]`).remove();
    });
  }

  /**
   * Show media library browser to select existing media.
   */
  function showMediaLibraryBrowser(fieldName) {
    // Use Drupal's media library widget dialog
    const allowedTypes = $('[data-field-name="' + fieldName + '"]').data('allowed-types');
    const mediaTypesJson = $('[data-field-name="' + fieldName + '"]').data('allowed-types');
    
    // Parse allowed media types
    let allowedBundles = ['comfyui_input', 'comfyui_generated'];
    if (mediaTypesJson) {
      try {
        allowedBundles = JSON.parse(mediaTypesJson);
      } catch (e) {
        debugWarn('Could not parse allowed media types:', mediaTypesJson);
      }
    }
    
    // Build media library dialog
    const dialogHtml = `
      <div class="comfyui-media-library-modal" data-field="${fieldName}">
        <div class="modal-content">
          <button type="button" class="close-modal" style="float:right; font-size:24px; border:none; background:none; cursor:pointer;">×</button>
          <h3>Select Existing Media</h3>
          <div id="comfyui-media-browser-${fieldName}" class="media-browser-container"></div>
        </div>
      </div>
    `;
    
    $('body').append(dialogHtml);
    
    // Load media list via AJAX
    $.ajax({
      url: '/comfyui/media-list',
      method: 'GET',
      data: {
        bundles: allowedBundles.join(','),
        owner_only: 1
      },
      dataType: 'json',
      success: function(response) {
        if (response.success && response.media_items) {
          renderMediaBrowser(fieldName, response.media_items);
        } else {
          $(`#comfyui-media-browser-${fieldName}`).html(
            '<div class="messages messages--warning">No media found</div>'
          );
        }
      },
      error: function() {
        $(`#comfyui-media-browser-${fieldName}`).html(
          '<div class="messages messages--error">Failed to load media library</div>'
        );
      }
    });
    
    // Handle close button
    $(`.comfyui-media-library-modal[data-field="${fieldName}"] .close-modal`).on('click', function() {
      $(`.comfyui-media-library-modal[data-field="${fieldName}"]`).remove();
    });
    
    // Close modal when clicking outside
    $(`.comfyui-media-library-modal[data-field="${fieldName}"]`).on('click', function(e) {
      if (e.target === this) {
        $(this).remove();
      }
    });
  }

  /**
   * Render the media browser with thumbnail grid.
   */
  function renderMediaBrowser(fieldName, mediaItems) {
    let html = '<div class="media-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(120px, 1fr)); gap: 15px; max-height: 500px; overflow-y: auto; padding: 15px 0;">';
    
    mediaItems.forEach(function(item) {
      const itemClass = item.thumbnail_url ? '' : 'no-thumbnail';
      const thumbnail = item.thumbnail_url ? 
        `<img src="${item.thumbnail_url}" alt="${item.name}" style="width: 100%; height: 100px; object-fit: cover; border-radius: 4px;" />` :
        `<div style="width: 100%; height: 100px; background: #f0f0f0; display: flex; align-items: center; justify-content: center; border-radius: 4px; font-size: 12px; color: #999;">No Image</div>`;
      
      html += `
        <div class="media-item ${itemClass}" data-media-id="${item.id}" style="cursor: pointer; border: 2px solid #ddd; border-radius: 4px; padding: 5px; transition: all 0.2s;" title="${item.name}">
          ${thumbnail}
          <div style="font-size: 11px; margin-top: 5px; text-align: center; word-break: break-word;">${item.name}</div>
        </div>
      `;
    });
    
    html += '</div>';
    
    $(`#comfyui-media-browser-${fieldName}`).html(html);
    
    // Attach click handlers to media items
    $(`.comfyui-media-library-modal[data-field="${fieldName}"] .media-item`).on('click', function() {
      const mediaId = $(this).data('media-id');
      const mediaName = $(this).find('div').text();
      
      updateMediaSelection(fieldName, mediaId);
      
      // Close modal
      $(`.comfyui-media-library-modal[data-field="${fieldName}"]`).remove();
    });
    
    // Highlight on hover
    $(`.comfyui-media-library-modal[data-field="${fieldName}"] .media-item`).on('mouseenter', function() {
      $(this).css('border-color', '#0066cc').css('box-shadow', '0 0 4px rgba(0, 102, 204, 0.3)');
    }).on('mouseleave', function() {
      $(this).css('border-color', '#ddd').css('box-shadow', 'none');
    });
  }

  /**
   * Handle media file upload.
   */
  function handleMediaUpload(form, fieldName) {
    const formData = new FormData(form);
    const fileInput = $(form).find('input[type="file"]')[0];
    const file = fileInput.files[0];
    
    // Get max from field settings
    const fieldSettings = drupalSettings.comfyui?.comfyui_media_fields?.[fieldName] || {};
    const maxFilesize = (fieldSettings.max_filesize || 10) * 1024 * 1024; // Convert to bytes
    
    // Validate file size
    if (file.size > maxFilesize) {
      const maxMB = maxFilesize / 1024 / 1024;
      const fileMB = (file.size / 1024 / 1024).toFixed(2);
      alert('File too large. Max ' + maxMB + 'MB, got ' + fileMB + 'MB');
      return;
    }
    
    const submitBtn = $(form).find('button[type="submit"]');
    
    submitBtn.prop('disabled', true).text('Uploading...');
    
    $.ajax({
      url: '/comfyui/upload-media',
      method: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function(response) {
        console.log('Upload response:', response);
        
        if (response.success && response.media_id) {
          updateMediaSelection(fieldName, response.media_id);
          $(`.comfyui-upload-modal[data-field="${fieldName}"]`).remove();
        } else {
          alert('Upload failed: ' + (response.message || 'Unknown error'));
          submitBtn.prop('disabled', false).text('Upload');
        }
      },
      error: function(xhr, status, error) {
        console.error('Upload failed:', error);
        console.error('Response:', xhr.responseText);
        alert('Upload failed: ' + error);
        submitBtn.prop('disabled', false).text('Upload');
      }
    });
  }

  function updateMediaSelection(fieldName, mediaId) {
    // Find the hidden input for this field
    const hiddenInput = $('[data-comfyui-field="' + fieldName + '"].comfyui-media-id-input');
    
    if (hiddenInput.length > 0) {
      hiddenInput.val(mediaId).trigger('change');
      
      // Get container and clear ALL old preview content
      const container = hiddenInput.closest('.comfyui-media-selector');
      
      // Remove all previous displays (both old and new format)
      container.find('.no-media-selected, .media-selected-message, .current-media-display').remove();
      
      // Fetch media thumbnail via custom endpoint
      $.ajax({
        url: '/comfyui/media-preview/' + mediaId,
        method: 'GET',
        dataType: 'json',
        success: function(response) {
          if (response.success && response.thumbnail_url) {
                    const mediaName = response.media_name;
                    const imageWidth = response.image_width;
                    const imageHeight = response.image_height;
                    
                    // Build resolution display if available
                    let resolutionHtml = '';
                    if (imageWidth && imageHeight) {
                      resolutionHtml = `<small style="color: #0066cc; font-weight: 500;">📏 ${imageWidth}×${imageHeight}px</small><br>`;
                    }
                    
                    // Create fresh preview element
                    const previewHtml = $(`
                      <div class="current-media-display">
                        <div class="media-thumbnail-preview">
                          <img src="${response.thumbnail_url}" alt="${mediaName}" style="max-width: 150px; max-height: 150px; border-radius: 4px; display: block; margin-bottom: 10px;" />
                        </div>
                        <div class="media-info-display">
                          <strong>${mediaName}</strong><br>
                          ${resolutionHtml}
                          <small>ID: ${mediaId}</small><br>
                          <small style="color: #666; font-size: 12px;">Selected - save workflow to apply</small>
                        </div>
                      </div>
                    `);
            
            // Append fresh preview to container
            container.append(previewHtml);
            
            debugLog('Updated media preview for field:', fieldName, 'media ID:', mediaId);
          } else {
            // Fallback if response doesn't have thumbnail
            fallbackMediaDisplay(fieldName, mediaId);
          }
        },
        error: function(xhr, status, error) {
          debugWarn('Failed to fetch media preview:', error);
          // Fallback if AJAX fails
          fallbackMediaDisplay(fieldName, mediaId);
        }
      });
    }
  }

  /**
   * Fallback display when preview fetch fails.
   */
  function fallbackMediaDisplay(fieldName, mediaId) {
    const hiddenInput = $('[data-comfyui-field="' + fieldName + '"].comfyui-media-id-input');
    const container = hiddenInput.closest('.comfyui-media-selector');
    
    // Clear ALL old preview content
    container.find('.no-media-selected, .media-selected-message, .current-media-display').remove();
    
    // Append fresh fallback display
    const fallbackHtml = $(`
      <div class="media-selected-message">
        <span style="font-size: 20px; margin-right: 8px;">📎</span>
        <strong>Media Selected</strong><br>
        <small>ID: ${mediaId}</small><br>
        <small style="color: #666; font-size: 12px;">Save workflow to apply</small>
      </div>
    `);
    
    container.append(fallbackHtml);
  }

})(jQuery, Drupal, once);
