/**
 * ComfyUI Seed Management System
 * Only loaded when RandomSeedWidget is used
 */
(function ($, Drupal, once) {
  'use strict';

  console.log('ComfyUI seed-manager.js loaded');

  /**
   * Seed Management Behavior - handles both widget inputs and block wrappers
   */
  Drupal.behaviors.comfyUISeedManager = {
    attach: function (context, settings) {
      // Method 1: Direct input elements with the class
      $(once('comfyui-seed-direct', 'input.comfyui-seed-field', context)).each(function() {
        const seedInput = $(this);
        const fieldName = seedInput.attr('data-comfyui-field') || 
                          seedInput.attr('name') || 
                          'seed';
        
        console.log('Found direct seed input:', fieldName);
        initializeSeedControls(seedInput, fieldName);
      });

      // Method 2: Block wrappers with the class containing inputs
      $(once('comfyui-seed-wrapper', '.comfyui-seed-field', context)).each(function() {
        const wrapper = $(this);
        const seedInput = wrapper.find('input[type="number"]');
        
        if (seedInput.length > 0) {
          const fieldName = wrapper.attr('data-comfyui-field') || 
                            seedInput.attr('name') || 
                            'seed';
          
          console.log('Found seed input in wrapper:', fieldName);
          initializeSeedControls(seedInput, fieldName);
        }
      });
    }
  };

  /**
   * Initialize seed control buttons for a field.
   */
  function initializeSeedControls(seedField, fieldName) {
    // Check if controls already exist
    const existingControls = seedField.closest('.js-form-item').find('.comfyui-seed-controls');
    if (existingControls.length > 0) {
      return;
    }
    
    // Create control container
    const controlsHtml = `
      <div class="comfyui-seed-controls" data-field="${fieldName}">
        <button type="button" class="comfyui-seed-btn comfyui-seed-random" data-mode="random" title="Random seed after each execution">
          🎲 Random
        </button>
        <button type="button" class="comfyui-seed-btn comfyui-seed-fixed" data-mode="fixed" title="Keep current seed fixed">
          🔒 Fixed
        </button>
        <button type="button" class="comfyui-seed-btn comfyui-seed-last" data-mode="last" title="Use seed from last saved result">
          🔄 Use Last
        </button>
      </div>
    `;
    
    // Add controls after the form item
    seedField.closest('.js-form-item').after(controlsHtml);
    const controls = seedField.closest('.js-form-item').next('.comfyui-seed-controls');
    
    // Initialize field if empty
    if (!seedField.val()) {
      generateRandomSeed(seedField);
    }
    
    // Load saved mode preference
    const savedMode = localStorage.getItem(`comfyui_seed_mode_${fieldName}`) || 'random';
    setSeedMode(controls, savedMode);
    
    // Attach button events
    controls.find('.comfyui-seed-btn').on('click', function(e) {
      e.preventDefault();
      const mode = $(this).data('mode');
      setSeedMode(controls, mode);
      handleSeedModeChange(seedField, fieldName, mode);
    });
  }

  /**
   * Set active seed mode and update UI.
   */
  function setSeedMode(controls, mode) {
    const fieldName = controls.data('field');
    
    controls.find('.comfyui-seed-btn').removeClass('active');
    controls.find(`[data-mode="${mode}"]`).addClass('active');
    
    localStorage.setItem(`comfyui_seed_mode_${fieldName}`, mode);
  }

  /**
   * Handle seed mode change actions.
   */
  function handleSeedModeChange(seedField, fieldName, mode) {
    switch(mode) {
      case 'random':
        generateRandomSeed(seedField);
        break;
        
      case 'fixed':
        // Keep current value
        break;
        
      case 'last':
        loadLastUsedSeed(seedField, fieldName);
        break;
    }
  }

  /**
   * Generate a random seed value.
   */
  function generateRandomSeed(seedField) {
    const randomSeed = Math.floor(Math.random() * 900000) + 100000;
    seedField.val(randomSeed).trigger('change');
    return randomSeed;
  }

  /**
   * Load the last used seed from storage.
   */
  function loadLastUsedSeed(seedField, fieldName) {
    const lastSeed = localStorage.getItem(`comfyui_last_seed_${fieldName}`);
    
    if (lastSeed) {
      seedField.val(lastSeed).trigger('change');
    } else {
      generateRandomSeed(seedField);
    }
  }

  /**
   * Save seed as "last used" when workflow executes successfully.
   */
  function saveLastUsedSeed(fieldName, seedValue) {
    localStorage.setItem(`comfyui_last_seed_${fieldName}`, seedValue);
  }

  /**
   * Handle pre-execution seed management - called BEFORE workflow execution.
   */
  function handlePreExecutionSeeds() {
    $('input.comfyui-seed-field, .comfyui-seed-field input[type="number"]').each(function() {
      const seedField = $(this);
      const fieldName = seedField.attr('data-comfyui-field') || 
                        seedField.closest('[data-comfyui-field]').attr('data-comfyui-field') || 
                        'seed';
      const currentSeed = seedField.val();
      const mode = localStorage.getItem(`comfyui_seed_mode_${fieldName}`) || 'random';
      
      console.log(`Pre-execution seed handling for ${fieldName}: mode=${mode}, current=${currentSeed}`);
      
      // Save current seed as "last used" BEFORE changing it
      if (currentSeed) {
        saveLastUsedSeed(fieldName, currentSeed);
      }
      
      // Handle mode-specific pre-execution behavior
      if (mode === 'random') {
        // Generate new seed for next time (user will see this before execution)
        generateRandomSeed(seedField);
        console.log(`Generated new seed for next execution: ${seedField.val()}`);
      }
      // For 'fixed' and 'last' modes, keep the current seed
    });
  }

  /**
   * Save seed as "last used" when workflow executes.
   */
  function saveLastUsedSeed(fieldName, seedValue) {
    localStorage.setItem(`comfyui_last_seed_${fieldName}`, seedValue);
    console.log(`Saved last used seed: ${seedValue} for field: ${fieldName}`);
  }

  // Expose for use by workflow form
  window.ComfyUISeedManager = {
    handlePreExecutionSeeds: handlePreExecutionSeeds
  };

})(jQuery, Drupal, once);
