// Prevent duplicate submissions
let isSubmitting = false;

/**
 * Debug logging with levels.
 * Levels: 0 = errors only, 1 = + warnings, 2 = + info/logs (verbose)
 */
const DEBUG_LEVEL = 0; // Set to 0 for production, 1 for warnings, 2 for full debug

const debugLog = (...args) => DEBUG_LEVEL >= 2 && console.log(...args);
const debugWarn = (...args) => DEBUG_LEVEL >= 1 && console.warn(...args);
const debugError = (...args) => console.error(...args);

debugLog('ComfyUI workflow-form.js loaded');

(function ($, Drupal, once) {
  'use strict';
  
  debugLog('ComfyUI Drupal behavior initializing');

  /**
   * Collect form values from all ComfyUI field blocks and widgets on the page.
   * Includes both regular mapped fields AND bypass control fields (Boolean checkboxes).
   */
  function collectFieldValues() {
    const values = {};
    
    // Find elements with data-comfyui-field attribute (blocks and widgets)
    const fieldElements = document.querySelectorAll('[data-comfyui-field]');
    
    fieldElements.forEach(element => {
      const fieldName = element.getAttribute('data-comfyui-field');
      const isBypassControl = element.getAttribute('data-comfyui-bypass-control') === 'true';
      
      let value;
      
      // Check if this element itself is an input
      if (element.tagName === 'INPUT' || element.tagName === 'TEXTAREA' || element.tagName === 'SELECT') {
        // This is the input element
        if (element.type === 'checkbox') {
          value = element.checked;
          
          // Debug bypass control fields
          if (isBypassControl) {
            debugLog('Bypass control field:', fieldName, '-> checked:', value, '(1=enabled, 0=bypass enabled)');
          }
        } else if (element.type === 'number') {
          value = parseFloat(element.value) || 0;
        } else {
          value = element.value;
        }
      } else {
        // This might be a wrapper, find the actual input inside
        const input = element.querySelector('input, textarea, select');
        if (input) {
          if (input.type === 'checkbox') {
            value = input.checked;
            
            // Debug bypass control fields
            if (isBypassControl) {
              debugLog('Bypass control field:', fieldName, '-> checked:', value, '(1=enabled, 0=bypass enabled)');
            }
          } else if (input.type === 'number') {
            value = parseFloat(input.value) || 0;
          } else {
            value = input.value;
          }
        } else {
          debugWarn('No input found for field:', fieldName, element);
          value = '';
        }
      }
      
      values[fieldName] = value;
      debugLog('Collected value for', fieldName, ':', value);
    });
    
    debugLog('All collected field values:', values);
    return values;
  }

  /**
   * Validate that all required fields have values.
   */
  function validateRequiredFields() {
    const fieldElements = document.querySelectorAll('[data-comfyui-field]');
    const errors = [];
    
    fieldElements.forEach(element => {
      const fieldName = element.getAttribute('data-comfyui-field');
      const isRequired = element.querySelector('[required]') !== null;
      
      if (!isRequired) return; // Skip non-required fields
      
      let value = null;
      
      // Handle different input types
      const input = element.querySelector('input, textarea, select');
      if (input) {
        if (input.type === 'checkbox') {
          value = input.checked ? 'checked' : null;
        } else if (input.type === 'hidden' && element.classList.contains('comfyui-media-selector')) {
          // Media selector - check the hidden input for media ID
          value = input.value;
        } else {
          value = input.value?.trim();
        }
      }
      
      // Check if value is empty
      if (!value) {
        const label = element.querySelector('label') || element.querySelector('[data-comfyui-field]');
        const fieldLabel = label ? label.textContent : fieldName;
        errors.push(fieldLabel + ' is required');
      }
    });
    
    return errors;
  }

  /**
   * Submit workflow via AJAX with polling for results.
   */
  function submitWorkflow(workflowId, fieldValues) {
    debugLog('submitWorkflow called with:', {workflowId, fieldValues});
    
    // VALIDATE REQUIRED FIELDS FIRST
    const validationErrors = validateRequiredFields();
    if (validationErrors.length > 0) {
      alert('Please fill in all required fields:\n\n' + validationErrors.join('\n'));
      isSubmitting = false;
      return;
    }
    
    const submitButton = document.querySelector('[data-comfyui-submit]');
    const resultsContainer = document.getElementById('comfyui-results-container');
    
    if (submitButton) {
      submitButton.disabled = true;
      submitButton.textContent = 'Executing...';
    }
    
    if (resultsContainer) {
      resultsContainer.innerHTML = '<div class="comfyui-progress">Executing workflow...</div>';
    }

    // Handle pre-execution seed management
    if (window.ComfyUISeedManager) {
      window.ComfyUISeedManager.handlePreExecutionSeeds();
    }

    // Make AJAX request
    $.ajax({
      url: '/comfyui/execute-workflow',
      method: 'POST',
      data: {
        workflow_id: workflowId,
        field_values: fieldValues,
        view_mode: 'form'
      },
      dataType: 'json',
      success: function(response) {
        debugLog('AJAX success:', response);
        
        if (response.success && response.polling_required) {
          // Start polling for results
          if (resultsContainer) {
            resultsContainer.innerHTML = '<div class="comfyui-progress">Workflow submitted. Waiting for results...</div>';
          }
          
          pollForResults(response.prompt_id, response.workflow_id, response.view_mode, submitButton, resultsContainer);
        } else if (response.success && response.results) {
          // Immediate results available
          if (resultsContainer) {
            displayUniversalResults(response.results, resultsContainer);
          }
          if (submitButton) {
            submitButton.disabled = false;
            submitButton.textContent = 'Execute Workflow';
            isSubmitting = false;
          }
          // RESET FLAG
          isSubmitting = false;
        } else {
          if (resultsContainer) {
            resultsContainer.innerHTML = '<div class="comfyui-error">Error: ' + (response.message || 'Unknown error') + '</div>';
          }
          if (submitButton) {
            submitButton.disabled = false;
            submitButton.textContent = 'Execute Workflow';
            isSubmitting = false;
          }
          // RESET FLAG
          isSubmitting = false;
        }
      },
      error: function(xhr, status, error) {
        debugLog('AJAX error:', {xhr, status, error});
        
        if (resultsContainer) {
          resultsContainer.innerHTML = '<div class="comfyui-error">Execution failed: ' + error + '</div>';
        }
        if (submitButton) {
          submitButton.disabled = false;
          submitButton.textContent = 'Execute Workflow';
          isSubmitting = false;
        }
        // RESET FLAG
        isSubmitting = false;
      }
    });
  }

  /**
   * Smart polling for workflow results with dynamic intervals and status tracking.
   */
  function pollForResults(promptId, workflowId, viewMode, submitButton, resultsContainer, attemptCount) {
    attemptCount = attemptCount || 0;

    // const maxAttempts = 120;
    // if (attemptCount >= maxAttempts) { return; }
    
    // Show elapsed time after 5 minutes so user has feedback
    if (attemptCount > 300) {
      if (attemptCount % 60 === 0) {
        const elapsedMinutes = Math.round(attemptCount / 30);
        if (resultsContainer) {
          resultsContainer.innerHTML = '<div class="comfyui-progress"><strong>Still processing...</strong><br><small>Time elapsed: ~' + elapsedMinutes + ' minutes</small></div>';
        }
      }
    }
    
    debugLog('Polling for results, attempt:', attemptCount + 1);
    
    $.ajax({
      url: '/comfyui/check-status/' + promptId,
      method: 'GET',
      data: {
        workflow_id: workflowId,
        view_mode: viewMode
      },
      dataType: 'json',
      success: function(statusResponse) {
        debugLog('Status check response:', statusResponse);
        
        // Handle different statuses
        switch (statusResponse.status) {
          case 'completed':
            // Results are ready!
            debugLog('Results ready:', statusResponse.results);
            if (resultsContainer) {
              displayUniversalResults(statusResponse.results, resultsContainer);
            }
            if (submitButton) {
              submitButton.disabled = false;
              submitButton.textContent = 'Execute Workflow';
              isSubmitting = false;
            }
            break;
            
          case 'processing':
            // Workflow is actively processing (same as 'running')
            if (resultsContainer) {
              resultsContainer.innerHTML = '<div class="comfyui-progress"><strong>Processing workflow...</strong><br><small>Your workflow is currently executing. This may take a few moments.</small></div>';
            }
            // Poll again with 2-second interval (processing can take time)
            setTimeout(function() {
              pollForResults(promptId, workflowId, viewMode, submitButton, resultsContainer, attemptCount + 1);
            }, 2000);
            break;
            
          case 'queued':
            // Workflow is in queue waiting
            var queuePosition = statusResponse.details && statusResponse.details.queue_position ? statusResponse.details.queue_position : '?';
            if (resultsContainer) {
              resultsContainer.innerHTML = '<div class="comfyui-progress"><strong>Queued for execution</strong><br><small>Position in queue: ' + queuePosition + '</small></div>';
            }
            // Poll again with 1-second interval (queue moves fast)
            setTimeout(function() {
              pollForResults(promptId, workflowId, viewMode, submitButton, resultsContainer, attemptCount + 1);
            }, 1000);
            break;
            
          case 'pending':
            // Just submitted, not yet in queue or history
            if (resultsContainer && attemptCount > 0) {
              resultsContainer.innerHTML = '<div class="comfyui-progress"><strong>Submitting workflow...</strong><br><small>Waiting for server response...</small></div>';
            }
            // Poll again quickly (should move to queue soon)
            setTimeout(function() {
              pollForResults(promptId, workflowId, viewMode, submitButton, resultsContainer, attemptCount + 1);
            }, 500);
            break;
            
          case 'failed':
            // Execution failed
            if (resultsContainer) {
              resultsContainer.innerHTML = '<div class="comfyui-error"><strong>Workflow execution failed</strong><br><small>' + (statusResponse.message || 'Unknown error occurred') + '</small></div>';
            }
            if (submitButton) {
              submitButton.disabled = false;
              submitButton.textContent = 'Execute Workflow';
              isSubmitting = false;
            }
            break;
            
          case 'error':
            // Error checking status
            debugError('Status check error:', statusResponse.message);
            // Retry a few times before giving up
            if (attemptCount < 10) {
              setTimeout(function() {
                pollForResults(promptId, workflowId, viewMode, submitButton, resultsContainer, attemptCount + 1);
              }, 1000);
            } else {
              if (resultsContainer) {
                resultsContainer.innerHTML = '<div class="comfyui-error">Error checking workflow status: ' + (statusResponse.message || 'Unknown error') + '</div>';
              }
              if (submitButton) {
                submitButton.disabled = false;
                submitButton.textContent = 'Execute Workflow';
                isSubmitting = false;
              }
            }
            break;
            
          default:
            // Unknown status, retry
            debugWarn('Unknown status:', statusResponse.status);
            setTimeout(function() {
              pollForResults(promptId, workflowId, viewMode, submitButton, resultsContainer, attemptCount + 1);
            }, 1000);
            break;
        }
      },
      error: function(xhr, status, error) {
        debugError('Status check request failed:', error);
        
        // Retry on network error
        if (attemptCount < maxAttempts) {
          if (resultsContainer) {
            resultsContainer.innerHTML = '<div class="comfyui-progress"><strong>Checking status...</strong><br><small>Retrying connection... (attempt ' + (attemptCount + 1) + ')</small></div>';
          }
          setTimeout(function() {
            pollForResults(promptId, workflowId, viewMode, submitButton, resultsContainer, attemptCount + 1);
          }, 2000);
        } else {
          if (resultsContainer) {
            resultsContainer.innerHTML = '<div class="comfyui-error">Failed to check status: ' + error + '</div>';
          }
          if (submitButton) {
            submitButton.disabled = false;
            submitButton.textContent = 'Execute Workflow';
            isSubmitting = false;
          }
        }
      }
    });
  }

  /**
   * Display workflow results (legacy function, kept for compatibility).
   */
  function displayResults(results, container) {
    let html = '<div class="comfyui-results-content">';
    
    if (results.images && results.images.length > 0) {
      html += '<div class="comfyui-images">';
      html += '<h4>Generated Images:</h4>';
      results.images.forEach(function(imageUrl) {
        html += '<img src="' + imageUrl + '" alt="Generated Image" style="max-width: 300px; margin: 10px;" />';
      });
      html += '</div>';
    }
    
    Object.keys(results).forEach(function(key) {
      if (key !== 'images') {
        html += '<div class="comfyui-output-item">';
        html += '<strong>' + key + ':</strong> ' + results[key];
        html += '</div>';
      }
    });
    
    html += '</div>';
    container.innerHTML = html;
  }

  /**
   * Display universal results with save/discard actions.
   */
  function displayUniversalResults(results, container) {
    if (!results || Object.keys(results).length === 0) {
      container.innerHTML = '<div class="comfyui-info">No results to display.</div>';
      return;
    }

    let html = '<div class="comfyui-results-content">';
    
    Object.keys(results).forEach(function(fieldName) {
      const resultData = results[fieldName];
      html += buildResultItem(fieldName, resultData);
    });
    
    html += '</div>';
    container.innerHTML = html;
    
    // Attach event listeners to save/discard buttons
    attachResultActionListeners(container);
  }

  /**
   * Build individual result item HTML.
   */
  function buildResultItem(fieldName, resultData) {
    // Convert field name to readable title (e.g., field_image_output -> Image Output)
    const fieldTitle = formatFieldNameAsTitle(fieldName);
    
    // Check if result data is empty or invalid
    if (!resultData || (typeof resultData === 'object' && Object.keys(resultData).length === 0)) {
      return '<div class="comfyui-result-item comfyui-error" data-field-name="' + fieldName + '">' +
        '<h4>' + fieldTitle + '</h4>' +
        '<div class="comfyui-preview">No output generated for this field.</div>' +
        '</div>';
    }
    
    let html = '<div class="comfyui-result-item" data-field-name="' + fieldName + '">';
    html += '<h4>' + fieldTitle + '</h4>';
    
    // Determine result type and build preview
    if (resultData.filename) {
      const extension = resultData.filename.split('.').pop().toLowerCase();
      
      if (['png', 'jpg', 'jpeg', 'gif', 'webp'].includes(extension)) {
        // Image preview
        html += '<div class="comfyui-preview">';
        
        // If media was auto-saved, fetch the Drupal file URL via AJAX
        if (resultData.media_id) {
          html += '<img src="" alt="Generated Image" class="comfyui-preview-image" data-media-id="' + resultData.media_id + '" />';
        } else {
          // Not yet saved, use ComfyUI proxy
          html += '<img src="/comfyui/image-proxy/' + resultData.filename + '" alt="Generated Image" class="comfyui-preview-image" />';
        }
        html += '</div>';
      } else if (['mp4', 'mov', 'avi', 'mkv', 'webm'].includes(extension)) {
        // Video preview
        html += '<div class="comfyui-preview">';
        html += '<video controls style="max-width: 400px;"><source src="/comfyui/video-proxy/' + displayFilename + '" type="video/mp4">Your browser does not support video.</video>';
        html += '</div>';
      } else {
        // Generic file
        html += '<div class="comfyui-preview">File: ' + displayFilename + '</div>';
      }
    } else if (typeof resultData === 'string' || typeof resultData === 'number') {
      // Text/data preview
      html += '<div class="comfyui-preview comfyui-text-preview">' + escapeHtml(resultData) + '</div>';
    } else {
      // Unknown type
      html += '<div class="comfyui-preview">Unknown result type</div>';
    }
    
    // ADD FEEDBACK FORM SECTION
    html += '<div class="comfyui-feedback-section">';
    html += '<h5>Your Feedback</h5>';
    html += '<div class="feedback-form">';
    
    // Thumbs up/down rating
    html += '<div class="feedback-rating">';
    html += '<label>Rate this result:</label>';
    html += '<div class="rating-buttons">';
    html += '<button type="button" class="feedback-thumbs-up" data-field-name="' + fieldName + '" data-rating="positive" title="Thumbs Up">👍</button>';
    html += '<button type="button" class="feedback-thumbs-down" data-field-name="' + fieldName + '" data-rating="negative" title="Thumbs Down">👎</button>';
    html += '</div>';
    html += '<input type="hidden" class="feedback-rating-value" data-field-name="' + fieldName + '" value="" />';
    html += '</div>';
    
    // Feedback textama
    html += '<div class="feedback-text">';
    html += '<label for="feedback-comments-' + fieldName + '">Comments (optional):</label>';
    html += '<textarea id="feedback-comments-' + fieldName + '" class="feedback-text-value" data-field-name="' + fieldName + '" placeholder="What could be improved? Any issues?" rows="3"></textarea>';
    html += '</div>';
    
    html += '</div>'; // end feedback-section
    
    // SMART ACTION BUTTONS - check if media already exists
    html += '<div class="comfyui-result-actions">';

    if (resultData.filename) {
      if (resultData.media_id) {
        // Media already exists (auto-saved or user saved via form)
        html += '<span class="auto-saved-message">✓ Saved to Media Library</span>';
        html += '<button class="comfyui-save-feedback btn btn-success" data-field-name="' + fieldName + '" data-media-id="' + resultData.media_id + '">Save Feedback</button>';
      } else {
        // Media not yet saved → show button to save both
        html += '<button class="comfyui-save-to-media btn btn-success" data-field-name="' + fieldName + '" data-result-data="' + escapeHtml(JSON.stringify(resultData)) + '">Save Image and Feedback</button>';
      }
    }

    html += '</div>';
    return html;
  }

  /**
   * Attach event listeners to result action buttons.
   */
  function attachResultActionListeners(container) {
    // Load Drupal file URLs for auto-saved media
    $(container).find('img[data-media-id]').each(function() {
      const img = $(this);
      const mediaId = img.data('media-id');
      
      $.ajax({
        url: '/comfyui/media-preview/' + mediaId,
        method: 'GET',
        dataType: 'json',
        success: function(response) {
          if (response.success && response.thumbnail_url) {
            img.attr('src', response.thumbnail_url);
          }
        },
        error: function() {
          img.attr('src', '/images/broken-image.png'); // Fallback
        }
      });
    });

    // Feedback rating buttons
    $(container).find('.feedback-thumbs-up').on('click', function() {
      const button = $(this);
      const fieldName = button.data('field-name');
      const rating = button.data('rating');
      
      // Update hidden input
      $(container).find('.feedback-rating-value[data-field-name="' + fieldName + '"]').val(rating);
      
      // Update button states
      $(container).find('.feedback-thumbs-up[data-field-name="' + fieldName + '"]').addClass('active');
      $(container).find('.feedback-thumbs-down[data-field-name="' + fieldName + '"]').removeClass('active');
      
      debugLog('Set rating for ' + fieldName + ': ' + rating);
    });
    
    $(container).find('.feedback-thumbs-down').on('click', function() {
      const button = $(this);
      const fieldName = button.data('field-name');
      const rating = button.data('rating');
      
      // Update hidden input
      $(container).find('.feedback-rating-value[data-field-name="' + fieldName + '"]').val(rating);
      
      // Update button states
      $(container).find('.feedback-thumbs-down[data-field-name="' + fieldName + '"]').addClass('active');
      $(container).find('.feedback-thumbs-up[data-field-name="' + fieldName + '"]').removeClass('active');
      
      debugLog('Set rating for ' + fieldName + ': ' + rating);
    });
    
    // Save feedback to existing media (auto-save mode)
    $(container).find('.comfyui-save-feedback').on('click', function() {
      const button = $(this);
      const fieldName = button.data('field-name');
      const mediaId = button.data('media-id');
      
      // Collect feedback data
      const feedbackRating = $(container).find('.feedback-rating-value[data-field-name="' + fieldName + '"]').val();
      const feedbackText = $(container).find('.feedback-text-value[data-field-name="' + fieldName + '"]').val();
      
      saveFeedbackToMedia(mediaId, feedbackRating, feedbackText, button);
    });
    
    // Save to media library buttons (only shown in manual mode)
    $(container).find('.comfyui-save-to-media').on('click', function() {
      const button = $(this);
      const fieldName = button.data('field-name');
      const resultData = button.data('result-data');
      
      // Collect feedback data
      const feedbackRating = $(container).find('.feedback-rating-value[data-field-name="' + fieldName + '"]').val();
      const feedbackText = $(container).find('.feedback-text-value[data-field-name="' + fieldName + '"]').val();
      
      // Add feedback to result data
      if (typeof resultData === 'string') {
        resultData = JSON.parse(resultData);
      }
      resultData.feedback_rating = feedbackRating;
      resultData.feedback_text = feedbackText;
      
      saveResultToMediaLibrary(fieldName, resultData, button);
    });
    
  }

  /**
   * Save result to media library.
   */
  function saveResultToMediaLibrary(fieldName, resultData, button) {
    const workflowId = $('#comfyui-workflow-form-container').data('workflow-id');
    
    // Parse result data if it's a string
    if (typeof resultData === 'string') {
      try {
        resultData = JSON.parse(resultData);
      } catch(e) {
        debugError('Failed to parse result data:', e);
      }
    }
    
    button.prop('disabled', true).text('Saving to Library...');
    
    $.ajax({
      url: '/comfyui/save-to-media-library',
      method: 'POST',
      data: {
        workflow_id: workflowId,
        field_name: fieldName,
        filename: resultData.filename,
        subfolder: resultData.subfolder || '',
        type: resultData.type || 'output'
      },
      dataType: 'json',
      success: function(response) {
        if (response.success) {
          button.text('✓ Saved to Library').addClass('saved');
          debugLog('Result saved to media library, media ID:', response.media_id);
          
          // Show media ID
          button.after('<small class="media-id-display"> (Media #' + response.media_id + ')</small>');
        } else {
          button.text('Save Failed').addClass('error');
          debugError('Save failed:', response.message);
        }
      },
      error: function(xhr, status, error) {
        button.text('Save Failed').addClass('error');
        debugError('Save request failed:', error);
      },
      complete: function() {
        setTimeout(() => {
          button.prop('disabled', false);
        }, 2000);
      }
    });
  }

  /**
   * Escape HTML for safe display.
   */
  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  /**
   * Convert field name to readable title.
   * Examples: field_image_output -> Image Output, field_prompt -> Prompt
   */
  function formatFieldNameAsTitle(fieldName) {
    // Remove 'field_' prefix if present
    let title = fieldName.replace(/^field_/, '');
    
    // Split by underscore and capitalize each word
    title = title
      .split('_')
      .map(function(word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
      })
      .join(' ');
    
    return title;
  }

  /**
   * Initialize ComfyUI form handling when DOM is ready.
   */
  Drupal.behaviors.comfyUIWorkflowForm = {
    attach: function (context, settings) {
      debugLog('ComfyUI behavior attach called', context);
      
      // Handle submit button clicks
      $(once('comfyui-submit', '[data-comfyui-submit]', context)).each(function() {
        $(this).on('click', function(e) {
          debugLog('Submit button clicked!');
          e.preventDefault();
          
          // PREVENT DUPLICATE SUBMISSIONS
          if (isSubmitting) {
            debugLog('Already submitting, ignoring click');
            return;
          }
          isSubmitting = true;
          
          const container = document.getElementById('comfyui-workflow-form-container');
          const workflowId = container ? container.getAttribute('data-workflow-id') : null;
          
          if (!workflowId) {
            debugError('No workflow ID found');
            isSubmitting = false;
            return;
          }
          
          const fieldValues = collectFieldValues();
          
          if (Object.keys(fieldValues).length === 0) {
            alert('No field values found. Please ensure you have added field blocks to the layout.');
            isSubmitting = false;
            return;
          }
          
          submitWorkflow(workflowId, fieldValues);
        });
      });
      
      // Initialize fields
      $(once('comfyui-field', '[data-comfyui-field]', context)).each(function() {
        debugLog('Initialized field:', this.getAttribute('data-comfyui-field'));
      });
    }
  };

  /**
   * Save feedback to an existing media entity (for auto-save mode).
   */
  function saveFeedbackToMedia(mediaId, feedbackRating, feedbackText, button) {
    button.prop('disabled', true).text('Saving Feedback...');
    
    $.ajax({
      url: '/comfyui/save-feedback/' + mediaId,
      method: 'POST',
      data: {
        feedback_rating: feedbackRating,
        feedback_text: feedbackText,
      },
      dataType: 'json',
      success: function(response) {
        if (response.success) {
          button.text('✓ Feedback Saved').addClass('saved').prop('disabled', false);
          debugLog('Feedback saved to media:', response.media_id);
        } else {
          button.text('Save Failed').addClass('error').prop('disabled', false);
          debugError('Save feedback failed:', response.message);
        }
      },
      error: function(xhr, status, error) {
        button.text('Save Failed').addClass('error').prop('disabled', false);
        debugError('Save feedback request failed:', error);
      },
    });
  }

})(jQuery, Drupal, once);
