<?php

namespace Drupal\comfyui_login\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for managing ComfyUI authentication tokens.
 */
class ComfyUITokenService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a ComfyUITokenService.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    KeyRepositoryInterface $key_repository,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->configFactory = $config_factory;
    $this->keyRepository = $key_repository;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Check if token authentication is enabled.
   *
   * @return bool
   *   TRUE if enabled, FALSE otherwise.
   */
  public function isEnabled() {
    $config = $this->configFactory->get('comfyui.settings');
    return (bool) $config->get('enable_token_auth');
  }

  /**
   * Get the authentication token.
   *
   * @return string|null
   *   The token string, or NULL if not configured.
   */
  public function getToken() {
    if (!$this->isEnabled()) {
      return NULL;
    }

    $config = $this->configFactory->get('comfyui.settings');
    $key_id = $config->get('api_token_key');

    if (empty($key_id)) {
      return NULL;
    }

    try {
      $key = $this->keyRepository->getKey($key_id);
      
      if ($key) {
        $token = $key->getKeyValue();
        
        if (!empty($token)) {
          return $token;
        }
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('comfyui_login')
        ->error('Failed to retrieve API token: @error', [
          '@error' => $e->getMessage(),
        ]);
    }

    return NULL;
  }

  /**
   * Build request options with authentication headers.
   *
   * @param array $base_options
   *   Base request options to merge with.
   *
   * @return array
   *   Complete request options with authentication.
   */
  public function buildRequestOptions(array $base_options = []) {
    $token = $this->getToken();

    if (!empty($token)) {
      if (!isset($base_options['headers'])) {
        $base_options['headers'] = [];
      }
      
      $base_options['headers']['Authorization'] = 'Bearer ' . $token;
      
      // Log that we're using authentication (but not the token itself)
      $this->loggerFactory->get('comfyui_login')
        ->debug('Adding authentication header to request (token ending: @suffix)', [
          '@suffix' => substr($token, -8),
        ]);
    }

    return $base_options;
  }

  /**
   * Validate that the endpoint is HTTPS when token auth is enabled.
   *
   * @param string $endpoint
   *   The API endpoint URL.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  public function validateEndpoint($endpoint) {
    if (!$this->isEnabled()) {
      return TRUE; // No validation needed if auth is disabled
    }

    // Allow localhost/127.0.0.1 for testing
    if (strpos($endpoint, 'http://localhost') === 0 || strpos($endpoint, 'http://127.0.0.1') === 0) {
      return TRUE;
    }

    // Require HTTPS for remote endpoints
    return strpos($endpoint, 'https://') === 0;
  }

}
