<?php

namespace Drupal\comfyui;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a list builder for ComfyUI workflows.
 */
class ComfyUIWorkflowListBuilder extends EntityListBuilder {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a new ComfyUIWorkflowListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    DateFormatterInterface $date_formatter
  ) {
    parent::__construct($entity_type, $storage);
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['id'] = $this->t('ID');
    $header['title'] = $this->t('Title');
    $header['field_mappings'] = $this->t('Field Mappings');
    $header['node_dependencies'] = $this->t('Node Dependencies');
    $header['status'] = $this->t('Status');
    $header['created'] = $this->t('Created');
    $header['changed'] = $this->t('Updated');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\comfyui\Entity\ComfyUIWorkflow $entity */
    $row['id'] = $entity->id();
    $row['title'] = $entity->toLink();
    
    // Show field mappings count
    $field_mappings = $entity->getFieldMappings();
    $input_count = count($entity->getInputMappings());
    $output_count = count($entity->getOutputMappings());
    $total_count = count($field_mappings);
    
    if ($total_count > 0) {
      $row['field_mappings'] = $this->t('@total mappings (@input in, @output out)', [
        '@total' => $total_count,
        '@input' => $input_count,
        '@output' => $output_count,
      ]);
    } else {
      $row['field_mappings'] = $this->t('None configured');
    }
    
    // Show node dependencies
    $discovered_nodes = $entity->getDiscoveredNodes();
    $node_mappings = $entity->getNodeMappings();
    
    if (!empty($discovered_nodes)) {
      $total_nodes = count($discovered_nodes);
      $configured_nodes = count(array_filter($node_mappings, function($mapping) {
        return !empty($mapping['git_url']);
      }));
      $core_nodes = count(array_filter($node_mappings, function($mapping) {
        return isset($mapping['git_url']) && $mapping['git_url'] === '_core';
      }));
      $pending_nodes = $total_nodes - $configured_nodes;
      
      if ($pending_nodes > 0) {
        // Has pending nodes that need configuration
        $row['node_dependencies'] = [
          'data' => [
            '#markup' => $this->t('@configured/@total configured (@core core) <strong style="color: #dc3545;">⚠ @pending pending</strong>', [
              '@configured' => $configured_nodes,
              '@total' => $total_nodes,
              '@core' => $core_nodes,
              '@pending' => $pending_nodes,
            ]),
          ],
        ];
      } else {
        // All nodes configured
        $row['node_dependencies'] = $this->t('@total nodes (@custom custom, @core core)', [
          '@total' => $total_nodes,
          '@custom' => $configured_nodes - $core_nodes,
          '@core' => $core_nodes,
        ]);
      }
      
      // Add model count
      $model_mappings = $entity->getModelMappings();
      if (!empty($model_mappings)) {
        $model_count = count($model_mappings);
        $row['node_dependencies'] .= ' | ' . $this->t('@count models', ['@count' => $model_count]);
      }
      
    } else {
      $row['node_dependencies'] = [
        'data' => [
          '#markup' => '<span style="color: #6c757d;">' . $this->t('Not discovered') . '</span>',
        ],
      ];
    }
    
    $row['status'] = $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled');
    $row['created'] = $this->dateFormatter->format($entity->get('created')->value);
    $row['changed'] = $this->dateFormatter->format($entity->get('changed')->value);
    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  protected function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);

    if ($entity->access('delete') && $entity->hasLinkTemplate('delete-form')) {
      $operations['delete'] = [
        'title' => $this->t('Delete'),
        'weight' => 100,
        'url' => $entity->toUrl('delete-form'),
      ];
    }

    // Add execute operation for enabled workflows
    if ($entity->get('status')->value && $entity->hasLinkTemplate('execute-form')) {
      $operations['execute'] = [
        'title' => $this->t('Execute'),
        'weight' => 20,
        'url' => $entity->toUrl('execute-form'),
      ];
    }

    // Add layout builder operation
    $operations['layout'] = [
      'title' => $this->t('Manage Layout'),
      'weight' => 15,
      'url' => \Drupal\Core\Url::fromRoute('layout_builder.overrides.comfyui_workflow.view', [
        'comfyui_workflow' => $entity->id()
      ]),
    ];

    // Add user view operation
    $operations['view_form'] = [
      'title' => $this->t('View Form'),
      'weight' => 10,
      'url' => \Drupal\Core\Url::fromRoute('comfyui.workflow_display', [
        'comfyui_workflow' => $entity->id()
      ]),
    ];

    return $operations;
  }
}
