<?php

namespace Drupal\comfyui\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Request;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Site\Settings;

class ComfyUIImageProxyController extends ControllerBase {

  protected $httpClient;
  protected $configFactory;
  protected $workflowExecutionService;

  public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, $workflow_execution_service = NULL) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->workflowExecutionService = $workflow_execution_service;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('http_client'),
      $container->get('config.factory'),
      $container->get('comfyui.workflow_execution')
    );
  }

  public function proxyImage($filename, Request $request) {
    $token_id = $request->query->get('token_id');
    
    // Token validation is OPTIONAL (for input images)
    if ($token_id) {
      if (!$this->validateFileToken($filename, $token_id)) {
        \Drupal::logger('comfyui')->warning('Invalid token_id for file: @file', [
          '@file' => $filename
        ]);
        return new Response('Access denied', Response::HTTP_FORBIDDEN);
      }
      
      // TOKEN PRESENT = INPUT IMAGE (serve from Drupal)
      return $this->serveFromDrupal($filename, $token_id);
    }
    
    // NO TOKEN = OUTPUT IMAGE (fetch from ComfyUI)
    return $this->serveFromComfyUI($filename, $request);
  }
  
  /**
   * Serve image from Drupal filesystem (input images).
   */
  private function serveFromDrupal($filename, $token_id) {
    try {
      $cached = \Drupal::cache()->get('comfyui_token_' . $token_id);
      if (!$cached) {
        throw new \Exception('Token expired');
      }
      
      $file_uri = $cached->data;
      $file_system = \Drupal::service('file_system');
      $real_path = $file_system->realpath($file_uri);
      
      if (!$real_path || !file_exists($real_path)) {
        throw new \Exception('File not found on disk: ' . $file_uri);
      }
      
      $image_data = file_get_contents($real_path);
      $content_type = mime_content_type($real_path) ?: 'image/png';
      
      $response = new Response($image_data);
      $response->headers->set('Content-Type', $content_type);
      $response->headers->set('Cache-Control', 'public, max-age=3600');
      
      return $response;
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Error serving from Drupal @file: @error', [
        '@file' => $filename, 
        '@error' => $e->getMessage()
      ]);
      return new Response('File not found', Response::HTTP_NOT_FOUND);
    }
  }
  
  /**
   * Serve image from ComfyUI (output images).
   */
  private function serveFromComfyUI($filename, Request $request) {
    try {
      $config = $this->configFactory->get('comfyui.settings');
      $api_endpoint = $config->get('api_endpoint');
      $subfolder = $request->query->get('subfolder', '');
      $type = $request->query->get('type', 'output');
      
      // Build ComfyUI image URL
      $image_url = $api_endpoint . '/view?' . http_build_query([
        'filename' => $filename,
        'subfolder' => $subfolder,
        'type' => $type,
      ]);
      
      $request_options = [
        'headers' => [
          'ngrok-skip-browser-warning' => 'true',
          'User-Agent' => 'Mozilla/5.0 (compatible; DrupalComfyUI/1.0)',
        ],
        'timeout' => 30,
      ];
      
      if (\Drupal::moduleHandler()->moduleExists('comfyui_login')) {
        $token_service = \Drupal::service('comfyui_login.token_service');
        $request_options = $token_service->buildRequestOptions($request_options);
      }
      
      $response = $this->httpClient->get($image_url, $request_options);
      $image_data = $response->getBody()->getContents();
      $content_type = $response->getHeader('Content-Type')[0] ?? 'image/png';
      
      $drupal_response = new Response($image_data);
      $drupal_response->headers->set('Content-Type', $content_type);
      $drupal_response->headers->set('Cache-Control', 'public, max-age=3600');
      $drupal_response->headers->set('Access-Control-Allow-Origin', '*');
      
      return $drupal_response;
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Error serving from ComfyUI @file: @error', [
        '@file' => $filename, 
        '@error' => $e->getMessage()
      ]);
      return new Response('File not found', Response::HTTP_NOT_FOUND);
    }
  }
  
  /**
   * Find file URI by searching filesystem (for output images without token).
   */
  private function findFileUri($filename) {
    $config = $this->configFactory->get('comfyui.settings');
    $file_scheme = $config->get('generated_image_uri_scheme') ?? 'public';
    $base_dir = $file_scheme . '://comfyui/';
    
    $file_system = \Drupal::service('file_system');
    $real_base = $file_system->realpath($base_dir);
    
    \Drupal::logger('comfyui')->debug('Looking for file @filename in @dir', [
      '@filename' => $filename,
      '@dir' => $real_base
    ]);
    
    if (!$real_base) {
      \Drupal::logger('comfyui')->error('Base directory not found: @dir', ['@dir' => $base_dir]);
      return NULL;
    }
    
    $files = glob($real_base . '/**/' . preg_quote($filename, '/'), 1);  // 1 = GLOB_RECURSIVE
    
    if (empty($files)) {
      return NULL;
    }
    
    return $file_scheme . '://comfyui/' . substr($files[0], strlen($real_base) + 1);
  }
  
  /**
   * Validate token for file access.
   */
  private function validateFileToken($filename, $token_id) {
    // Check token in default cache
    $cached = \Drupal::cache()->get('comfyui_token_' . $token_id);
    
    if (!$cached) {
      \Drupal::logger('comfyui')->warning('Token not found or expired: @token_id', [
        '@token_id' => $token_id
      ]);
      return FALSE;
    }
    
    $file_uri = $cached->data;
    
    // Verify filename matches
    if (basename($file_uri) !== $filename) {
      \Drupal::logger('comfyui')->warning('Filename mismatch: expected @expected, got @got', [
        '@expected' => basename($file_uri),
        '@got' => $filename
      ]);
      return FALSE;
    }
    
    return TRUE;
  }

  /**
   * Automatically save a generated image as a media entity.
   */
  protected function autoSaveGeneratedImage($filename, Request $request) {
    try {
      if (empty($filename) || $filename === 'undefined' || $filename === 'null') {
        return;
      }
      
      $subfolder = $request->query->get('subfolder', '');
      $type = $request->query->get('type', 'output');
      
      $session = $request->getSession();
      $workflow_id = $session->get('comfyui_last_workflow_id');
      $prompt_id = $session->get('comfyui_last_prompt_id');
      $source_media_id = $session->get('comfyui_last_source_media_id');
      
      if (!$workflow_id) {
        return;
      }
      
      $query = \Drupal::entityQuery('media')
        ->condition('bundle', 'comfyui_generated')
        ->accessCheck(FALSE);
      
      if ($prompt_id) {
        $query->condition('field_comfyui_prompt_id', $prompt_id);
      }
      
      $existing_ids = $query->execute();
      
      if (!empty($existing_ids)) {
        return;
      }
      
      $media = $this->workflowExecutionService->saveGeneratedImageAsMedia(
        $filename,
        $subfolder,
        $type,
        $workflow_id,
        $source_media_id,
        NULL,
        [],
        $prompt_id
      );
      
      if ($media) {
        \Drupal::logger('comfyui')->info(
          'Auto-saved generated image as media entity @id',
          ['@id' => $media->id()]
        );
      }
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error(
        'Failed to auto-save generated image: @error',
        ['@error' => $e->getMessage()]
      );
    }
  }

}
