<?php

namespace Drupal\comfyui\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\comfyui\Entity\ComfyUIWorkflow;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;

/**
 * Controller for ComfyUI workflow operations.
 */
class ComfyUIWorkflowController extends ControllerBase {

  /**
   * The entity type manager.
   */
  protected $entityTypeManager;

  /**
   * Constructs a ComfyUIWorkflowController object.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Displays a workflow for end users.
   */
  public function viewWorkflow(ComfyUIWorkflow $comfyui_workflow, $view_mode = 'form') {
    // Use entity view builder - the pseudo-field will render the form
    $view_builder = $this->entityTypeManager()->getViewBuilder('comfyui_workflow');
    $build = $view_builder->view($comfyui_workflow, 'full');
    
    return $build;
  }

  /**
   * Returns the title for the workflow page.
   */
  public function getWorkflowTitle(ComfyUIWorkflow $comfyui_workflow) {
    return $comfyui_workflow->label() ?: $this->t('ComfyUI Workflow');
  }

  /**
   * Check workflow execution status.
   */
  public function checkStatus($prompt_id) {
    $execution_service = \Drupal::service('comfyui.workflow_execution');
    $status_result = $execution_service->checkExecutionStatus($prompt_id);
    
    // If completed, try to get formatted results
    if ($status_result['status'] === 'completed' && isset($status_result['data'])) {
      try {
        // Extract workflow_id from the request if available
        $request = \Drupal::request();
        $workflow_id = $request->query->get('workflow_id');
        $view_mode = $request->query->get('view_mode', 'form');
        
        if ($workflow_id) {
          $results = $execution_service->getFormattedResultsWithViewMode(
            $prompt_id,
            $workflow_id,
            $view_mode
          );

          // **TRIGGER AUTO-SAVE ON COMPLETION**
          $this->triggerAutoSaveIfEnabled($results, $workflow_id, $prompt_id, $request);

          // Set results AFTER auto-save (so media_id is included)
          $status_result['results'] = $results;
        }
      } catch (\Exception $e) {
        \Drupal::logger('comfyui')->warning('Could not format results: @error', [
          '@error' => $e->getMessage()
        ]);
      }
    }
    
    return new JsonResponse($status_result);
  }

  /**
   * Execute workflow with field values from the blocks.
   */
  protected function executeWorkflowWithValues($workflow, $field_values, $view_mode = 'form') {
    if (!$workflow) {
      throw new \Exception('No workflow provided');
    }
    
    // Transform field values to the format expected by the execution service
    $input_values = $this->transformFieldValuesForExecution($field_values, $workflow, $view_mode);
    
    \Drupal::logger('comfyui')->info('Executing workflow with input values: @values', ['@values' => print_r($input_values, TRUE)]);
    
    // Use the enhanced execution service
    $execution_service = \Drupal::service('comfyui.workflow_execution');
    
    // Execute the workflow with configuration
    $execution_result = $execution_service->executeWorkflowWithViewMode(
      $workflow->id(),
      $input_values,
      $view_mode
    );
    
    if (!$execution_result['success']) {
      throw new \Exception($execution_result['message'] ?? 'Workflow execution failed');
    }
    
    $prompt_id = $execution_result['data']['prompt_id'];
    
    \Drupal::logger('comfyui')->info('Workflow execution started with prompt_id: @prompt_id', ['@prompt_id' => $prompt_id]);

    // Get results with retry logic
    $results = [];
    $max_retries = 3;
    $retry_count = 0;

    while ($retry_count < $max_retries) {
      try {
        $results = $execution_service->getFormattedResultsWithViewMode(
          $prompt_id,
          $workflow->id(),
          $view_mode
        );
        
        // Filter out system fields from results
        $filtered_results = [];
        foreach ($results as $field_name => &$result_data) {
          // Only include actual output fields, not status/message
          if (!in_array($field_name, ['status', 'message', 'prompt_id'])) {
            $filtered_results[$field_name] = $result_data;
          }
        }
        
        if (!empty($filtered_results)) {
          $results = $filtered_results;
          break; // Success, exit loop
        }
        
      } catch (\Exception $e) {
        $retry_count++;
        \Drupal::logger('comfyui')->warning('Results not ready yet, retry @retry of @max: @error', [
          '@retry' => $retry_count,
          '@max' => $max_retries,
          '@error' => $e->getMessage()
        ]);
        
        if ($retry_count < $max_retries) {
          sleep(1); // Wait 1 second before retry
        }
      }
    }
    
    // If no results after retries, return processing status
    if (empty($results)) {
      return [
        'success' => true,
        'prompt_id' => $prompt_id,
        'results' => ['processing' => 'Workflow executed successfully. Results processing...'],
        'status' => 'processing'
      ];
    }
    
    return [
      'success' => true,
      'prompt_id' => $prompt_id,
      'results' => $results,
      'status' => 'completed'
    ];
  }

  /**
   * Transform field values to the format expected by the execution service.
   */
  protected function transformFieldValuesForExecution($field_values, $workflow, $view_mode) {
    $input_values = [];
    $field_mappings = $workflow->getFieldMappings();
    $bypass_mappings = $workflow->getBypassMappings();
    
    // Include regular mapped input fields
    foreach ($field_mappings as $field_name => $mapping) {
      if ($mapping['direction'] !== 'input') {
        continue;
      }
      
      if (isset($field_values[$field_name])) {
        $field_value = $field_values[$field_name];
        $transformed_value = $this->transformFieldValue($field_value, $mapping);
        
        if ($transformed_value !== NULL) {
          $input_values[$field_name] = $transformed_value;
        }
      }
    }
    
    // Include bypass control fields
    $bypass_field_names = [];
    if (!empty($bypass_mappings['group_bypasses'])) {
      $bypass_field_names = array_merge($bypass_field_names, array_keys($bypass_mappings['group_bypasses']));
    }
    if (!empty($bypass_mappings['node_bypasses'])) {
      $bypass_field_names = array_merge($bypass_field_names, array_keys($bypass_mappings['node_bypasses']));
    }
    
    foreach ($bypass_field_names as $field_name) {
      if (isset($field_values[$field_name])) {
        // Boolean fields come as string "false" or "true" from form data
        $value = $field_values[$field_name];
        if ($value === "false" || $value === "0") {
          $input_values[$field_name] = false;
        } else {
          $input_values[$field_name] = (bool) $value;
        }
      }
    }
    
    return $input_values;
  }

  /**
   * Transform Drupal field values to ComfyUI parameter format.
   */
  protected function transformFieldValue($field_value, $mapping) {
    // Handle different field types and extract the actual value
    if (is_array($field_value)) {
      // Multi-value field or structured field
      if (isset($field_value[0]['value'])) {
        // Text field
        return $field_value[0]['value'];
      } elseif (isset($field_value[0]['target_id'])) {
        // Entity reference field
        return $field_value[0]['target_id'];
      } elseif (isset($field_value[0])) {
        // Other structured field
        return $field_value[0];
      }
    }
    
    return $field_value;
  }

  /**
   * AJAX endpoint for workflow execution from blocks.
   */
  public function executeWorkflowAjax(Request $request) {
    $data = $request->request->all();
    $workflow_id = $data['workflow_id'] ?? null;
    $field_values = $data['field_values'] ?? [];
    $view_mode = $data['view_mode'] ?? 'form';
    
    // Ensure session is started
    if (!$request->hasSession()) {
      $request->setSession(\Drupal::service('session'));
    }
    
    \Drupal::logger('comfyui')->info('Received AJAX request - workflow_id: @id, field_values: @values', [
      '@id' => $workflow_id,
      '@values' => print_r($field_values, TRUE),
    ]);
    
    try {
      $workflow = $this->entityTypeManager()
        ->getStorage('comfyui_workflow')
        ->load($workflow_id);
      
      if (!$workflow) {
        return new JsonResponse(['success' => false, 'message' => 'Workflow not found']);
      }
      
      // Transform field values
      $input_values = $this->transformFieldValuesForExecution($field_values, $workflow, $view_mode);
      
      // Store workflow context in session for auto-save
      $session = $request->getSession();
      $session->set('comfyui_last_workflow_id', $workflow_id);
      
      // Extract source media IDs from input values (if any)
      $source_media_ids = $this->extractMediaIdsFromInputs($input_values, $workflow);
      if (!empty($source_media_ids)) {
        $session->set('comfyui_last_source_media_id', reset($source_media_ids)); // Use first one
      }
      
      // Execute workflow using the service
      $execution_service = \Drupal::service('comfyui.workflow_execution');
      $execution_result = $execution_service->executeWorkflowWithViewMode(
        $workflow->id(),
        $input_values,
        $view_mode
      );
      
      // Store prompt ID in session after successful execution
      if ($execution_result['success']) {
        $session->set('comfyui_last_prompt_id', $execution_result['data']['prompt_id']);
      }
      
      if (!$execution_result['success']) {
        return new JsonResponse([
          'success' => false,
          'message' => $execution_result['message'] ?? 'Execution failed'
        ]);
      }
      
      $prompt_id = $execution_result['data']['prompt_id'];
      
      // Return immediately with prompt_id for polling
      return new JsonResponse([
        'success' => true,
        'prompt_id' => $prompt_id,
        'workflow_id' => $workflow_id,
        'view_mode' => $view_mode,
        'message' => 'Workflow queued successfully. Checking for results...',
        'polling_required' => true
      ]);
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Workflow execution failed: @error', ['@error' => $e->getMessage()]);
      return new JsonResponse([
        'success' => false,
        'message' => $e->getMessage()
      ]);
    }
  }

  /**
   * AJAX endpoint for saving individual results to fields.
   */
  public function saveResultAjax(Request $request) {
    $data = $request->request->all();
    $workflow_id = $data['workflow_id'] ?? null;
    $field_name = $data['field_name'] ?? null;
    $result_data = $data['result_data'] ?? null;
    
    try {
      $workflow = $this->entityTypeManager()
        ->getStorage('comfyui_workflow')
        ->load($workflow_id);
      
      if (!$workflow) {
        return new JsonResponse(['success' => false, 'message' => 'Workflow not found']);
      }
      
      if (!$field_name || !$result_data) {
        return new JsonResponse(['success' => false, 'message' => 'Missing field name or result data']);
      }
      
      // Decode result data
      $result_data_decoded = json_decode($result_data, TRUE);
      if (json_last_error() !== JSON_ERROR_NONE) {
        return new JsonResponse(['success' => false, 'message' => 'Invalid result data JSON']);
      }
      
      // Get field definition
      $field_definitions = \Drupal::service('entity_field.manager')
        ->getFieldDefinitions($workflow->getEntityTypeId(), $workflow->bundle());
      
      $field_definition = $field_definitions[$field_name] ?? null;
      if (!$field_definition) {
        return new JsonResponse(['success' => false, 'message' => 'Field not found: ' . $field_name]);
      }
      
      // Use output type manager to process and save
      $output_type_manager = \Drupal::service('comfyui.output_type_manager');
      $output_type = $output_type_manager->detectOutputType($result_data_decoded, $field_definition);
      $handler = $output_type_manager->getOutputHandler($output_type);
      
      if (!$handler) {
        return new JsonResponse(['success' => false, 'message' => 'No handler found for output type: ' . $output_type]);
      }
      
      // Process for saving
      $field_value = $handler->processForSave($result_data_decoded, $field_definition);
      
      if ($field_value === null) {
        return new JsonResponse(['success' => false, 'message' => 'Cannot save this output type to this field type']);
      }
      
      // Save to entity field
      $workflow->set($field_name, $field_value);
      $workflow->save();
      
      return new JsonResponse([
        'success' => true,
        'message' => 'Result saved successfully',
        'field_name' => $field_name
      ]);
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Save result failed: @error', ['@error' => $e->getMessage()]);
      return new JsonResponse([
        'success' => false,
        'message' => $e->getMessage()
      ]);
    }
  }

  /**
   * Access callback for workflow operations.
   */
  public function access(AccountInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'execute comfyui workflows');
  }

  /**
   * Extract media entity IDs from input values.
   *
   * @param array $input_values
   *   The transformed input values.
   * @param \Drupal\comfyui\Entity\ComfyUIWorkflow $workflow
   *   The workflow entity.
   *
   * @return array
   *   Array of media entity IDs found in inputs.
   */
  protected function extractMediaIdsFromInputs(array $input_values, $workflow) {
    $media_ids = [];
    $field_mappings = $workflow->getFieldMappings();
    
    foreach ($input_values as $field_name => $value) {
      if (!isset($field_mappings[$field_name])) {
        continue;
      }
      
      // Check if this is an entity reference field
      $field_definitions = \Drupal::service('entity_field.manager')
        ->getFieldDefinitions('comfyui_workflow', 'comfyui_workflow');
      
      if (isset($field_definitions[$field_name])) {
        $field_definition = $field_definitions[$field_name];
        $field_type = $field_definition->getType();
        
        // If it's an entity reference, check if it references media
        if ($field_type === 'entity_reference') {
          $target_type = $field_definition->getSetting('target_type');
          
          if ($target_type === 'media') {
            // This is a media reference - extract the ID
            if (is_array($value) && isset($value[0]['target_id'])) {
              $media_ids[] = $value[0]['target_id'];
            } elseif (is_numeric($value)) {
              $media_ids[] = $value;
            }
          }
        }
      }
    }
    
    return array_filter($media_ids);
  }

  /**
   * Trigger auto-save of generated images if enabled.
   */
  protected function triggerAutoSaveIfEnabled(&$results, $workflow_id, $prompt_id, Request $request) {
    $config = \Drupal::config('comfyui.settings');
    $auto_save_enabled = $config->get('auto_save_generated_images') ?? TRUE;
    
    if (!$auto_save_enabled) {
      \Drupal::logger('comfyui')->debug('Auto-save disabled for prompt @id', ['@id' => $prompt_id]);
      return;
    }
    
    // Get session data for context
    $session = $request->getSession();
    $source_media_id = $session->get('comfyui_last_source_media_id');
    
    // Save each result as media entity
    foreach ($results as $field_name => &$result_data) {
      if (empty($result_data) || isset($result_data['error'])) {
        continue;
      }
      
      // Only auto-save file-based outputs (images/videos)
      if (!isset($result_data['filename'])) {
        continue;
      }
      
      // Check if already saved for this prompt
      $query = \Drupal::entityQuery('media')
        ->condition('bundle', 'comfyui_generated')
        ->condition('field_comfyui_prompt_id', $prompt_id)
        ->accessCheck(FALSE);
      
      $existing_ids = $query->execute();
      
      if (!empty($existing_ids)) {
        \Drupal::logger('comfyui')->debug('Media already saved for prompt @id', ['@id' => $prompt_id]);
        continue;
      }
      
      // Save the image as media entity
      $execution_service = \Drupal::service('comfyui.workflow_execution');
      $saved_media = $execution_service->saveGeneratedImageAsMedia(
        $result_data['filename'],
        $result_data['subfolder'] ?? '',
        $result_data['type'] ?? 'output',
        $workflow_id,
        $source_media_id,
        NULL, // prompt text
        [], // workflow params
        $prompt_id
      );

      if ($saved_media) {
        // Get the hashed filename from the saved media entity
        $hashed_filename = $this->getHashedFilenameFromMedia($saved_media);
        
        // Update result data with hashed filename for frontend display
        $result_data['hashed_filename'] = $hashed_filename;
        $result_data['media_id'] = $saved_media->id();
        
        \Drupal::logger('comfyui')->info(
          'Auto-saved result for field @field as media @id (hash: @hash)',
          ['@field' => $field_name, '@id' => $saved_media->id(), '@hash' => $hashed_filename]
        );
      }
    }
    
    // Unset the reference
    unset($result_data);
  }

  /**
   * AJAX endpoint for saving results to media library.
   */
  public function saveToMediaLibraryAjax(Request $request) {
    $data = $request->request->all();
    $workflow_id = $data['workflow_id'] ?? null;
    $field_name = $data['field_name'] ?? null;
    $filename = $data['filename'] ?? null;
    $subfolder = $data['subfolder'] ?? '';
    $type = $data['type'] ?? 'output';
    $feedback_rating = $data['feedback_rating'] ?? null;
    $feedback_text = $data['feedback_text'] ?? null;

    try {
      if (!$workflow_id || !$filename) {
        return new JsonResponse([
          'success' => false,
          'message' => 'Missing required parameters'
        ]);
      }

      // Get workflow
      $workflow = $this->entityTypeManager()
        ->getStorage('comfyui_workflow')
        ->load($workflow_id);

      if (!$workflow) {
        return new JsonResponse([
          'success' => false,
          'message' => 'Workflow not found'
        ]);
      }

      // Get session data for context
      $session = $request->getSession();
      $source_media_id = $session->get('comfyui_last_source_media_id');
      $prompt_id = $session->get('comfyui_last_prompt_id');

      // Use the execution service to save the media
      $execution_service = \Drupal::service('comfyui.workflow_execution');

      $media = $execution_service->saveGeneratedImageAsMedia(
        $filename,
        $subfolder,
        $type,
        $workflow_id,
        $source_media_id,
        null, // prompt - could be extracted from field mappings
        [], // workflow params - could be extracted from session
        $prompt_id
      );

      if ($media) {
        // Save feedback if provided
        if ($feedback_rating || $feedback_text) {
          try {
            if ($feedback_rating && $media->hasField('field_media_feedback_rating')) {
              $media->set('field_media_feedback_rating', $feedback_rating);
            }
            if ($feedback_text && $media->hasField('field_media_feedback_text')) {
              $media->set('field_media_feedback_text', $feedback_text);
            }
            $media->save();
            \Drupal::logger('comfyui')->info(
              'Saved feedback for media @id: rating=@rating',
              ['@id' => $media->id(), '@rating' => $feedback_rating]
            );
          } catch (\Exception $e) {
            \Drupal::logger('comfyui')->warning('Could not save feedback: @error', [
              '@error' => $e->getMessage()
            ]);
          }
        }

        return new JsonResponse([
          'success' => true,
          'message' => 'Saved to media library',
          'media_id' => $media->id(),
          'media_label' => $media->label(),
        ]);
      } else {
        return new JsonResponse([
          'success' => false,
          'message' => 'Failed to create media entity'
        ]);
      }

    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Save to media library failed: @error', [
        '@error' => $e->getMessage()
      ]);
      return new JsonResponse([
        'success' => false,
        'message' => $e->getMessage()
      ]);
    }
  }

  /**
   * AJAX endpoint for getting media preview thumbnail.
   */
  public function mediaPreview($media_id) {
    try {
      $media = $this->entityTypeManager()
        ->getStorage('media')
        ->load($media_id);
      
      if (!$media) {
        return new JsonResponse([
          'success' => FALSE,
          'message' => 'Media not found'
        ]);
      }
      
      // Check access
      if (!$media->access('view')) {
        return new JsonResponse([
          'success' => FALSE,
          'message' => 'Access denied'
        ]);
      }
      
      $media_name = $media->label();
      $thumbnail_url = '';
      $image_width = NULL;
      $image_height = NULL;
      
      // Try to get image URL and dimensions from field_media_image
      if ($media->hasField('field_media_image')) {
        $image_field = $media->get('field_media_image');
        
        if (!$image_field->isEmpty()) {
          $file = $image_field->entity;
          
          if ($file) {
            $file_uri = $file->getFileUri();
            $thumbnail_url = \Drupal::service('file_url_generator')
              ->generateAbsoluteString($file_uri);
            
            // Get image dimensions using Drupal's image utility
            try {
              $real_path = \Drupal::service('file_system')->realpath($file_uri);
              if ($real_path && file_exists($real_path)) {
                $image_info = getimagesize($real_path);
                if ($image_info) {
                  $image_width = $image_info[0];
                  $image_height = $image_info[1];
                }
              }
            } catch (\Exception $e) {
              \Drupal::logger('comfyui')->debug('Could not get image dimensions: @error', [
                '@error' => $e->getMessage()
              ]);
            }
          }
        }
      }
      
      return new JsonResponse([
        'success' => TRUE,
        'media_id' => $media_id,
        'media_name' => $media_name,
        'thumbnail_url' => $thumbnail_url,
        'image_width' => $image_width,
        'image_height' => $image_height,
      ]);
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Media preview error: @error', [
        '@error' => $e->getMessage()
      ]);
      
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Error fetching preview'
      ]);
    }
  }

  /**
   * AJAX endpoint for getting list of media for browsing.
   */
  public function mediaList(Request $request) {
    try {
      $bundles = $request->query->get('bundles', 'comfyui_input,comfyui_generated');
      $bundles = array_filter(explode(',', $bundles));
      
      $owner_only = $request->query->get('owner_only', 0);
      
      // Build query for media entities
      $query = $this->entityTypeManager()
        ->getStorage('media')
        ->getQuery()
        ->condition('bundle', $bundles, 'IN')
        ->accessCheck(TRUE)
        ->sort('changed', 'DESC');
      
      // Filter to current user's media if requested
      if ($owner_only) {
        $query->condition('uid', \Drupal::currentUser()->id());
      }
      
      $media_ids = $query->execute();
      
      $media_items = [];
      
      foreach ($media_ids as $media_id) {
        $media = $this->entityTypeManager()
          ->getStorage('media')
          ->load($media_id);
        
        if (!$media || !$media->access('view')) {
          continue;
        }
        
        $thumbnail_url = '';
        
        // Try to get image URL from field_media_image
        if ($media->hasField('field_media_image')) {
          $image_field = $media->get('field_media_image');
          
          if (!$image_field->isEmpty()) {
            $file = $image_field->entity;
            
            if ($file) {
              $file_uri = $file->getFileUri();
              $thumbnail_url = \Drupal::service('file_url_generator')
                ->generateAbsoluteString($file_uri);
            }
          }
        }
        
        $media_items[] = [
          'id' => $media->id(),
          'name' => $media->label(),
          'bundle' => $media->bundle(),
          'thumbnail_url' => $thumbnail_url,
        ];
      }
      
      return new JsonResponse([
        'success' => TRUE,
        'media_items' => $media_items,
        'count' => count($media_items),
      ]);
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Media list error: @error', [
        '@error' => $e->getMessage()
      ]);
      
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Error fetching media list'
      ]);
    }
  }

  /**
   * AJAX endpoint for uploading new media.
   */
  public function uploadMediaAjax(Request $request) {
    try {
      $files = $request->files->get('image');
      $name = $request->request->get('name');
      
      if (!$files) {
        return new JsonResponse([
          'success' => false,
          'message' => 'No file uploaded'
        ]);
      }
      
      // Get upload directory from helper
      $helper = \Drupal::service('comfyui.file_scheme_helper');
      $upload_dir = $helper->getInputDirectory();
      
      // Ensure directory exists with proper permissions
      try {
        \Drupal::service('file_system')->prepareDirectory(
          $upload_dir,
          \Drupal\Core\File\FileSystemInterface::CREATE_DIRECTORY | 
          \Drupal\Core\File\FileSystemInterface::MODIFY_PERMISSIONS
        );
      } catch (\Exception $e) {
        return new JsonResponse([
          'success' => false,
          'message' => 'Directory error: ' . $e->getMessage(),
        ]);
      }
      
      // Validate MIME type and extension
      $allowed_extensions = ['png', 'jpg', 'jpeg', 'gif', 'webp'];
      $file_extension = strtolower(pathinfo($files->getClientOriginalName(), PATHINFO_EXTENSION));
      
      if (!in_array($file_extension, $allowed_extensions)) {
        return new JsonResponse([
          'success' => false,
          'message' => 'Invalid file type: ' . $file_extension . '. Allowed: ' . implode(', ', $allowed_extensions),
        ]);
      }
      
      // Check file size (10MB max)
      $max_size = 10 * 1024 * 1024;
      if ($files->getSize() > $max_size) {
        return new JsonResponse([
          'success' => false,
          'message' => 'File too large. Max 10MB, got ' . round($files->getSize() / 1024 / 1024, 2) . 'MB',
        ]);
      }
      
      // Modern way to save uploaded file
      try {
        $file_contents = file_get_contents($files->getRealPath());
        $filename = $files->getClientOriginalName();
        $file_uri = $upload_dir . '/' . $filename;
        
        // Save using file system service (Drupal 11 way)
        $file_system = \Drupal::service('file_system');
        $real_path = $file_system->saveData(
          $file_contents,
          $file_uri,
          \Drupal\Core\File\FileSystemInterface::EXISTS_RENAME
        );
        
        if (!$real_path) {
          return new JsonResponse([
            'success' => false,
            'message' => 'Failed to save file',
          ]);
        }
        
        // Create file entity
        $file = \Drupal::entityTypeManager()
          ->getStorage('file')
          ->create([
            'uri' => $real_path,
            'uid' => \Drupal::currentUser()->id(),
          ]);
        $file->setPermanent();
        $file->save();
        
        if (!$file) {
          return new JsonResponse([
            'success' => false,
            'message' => 'Failed to save file to disk',
          ]);
        }
        
        // Make file permanent
        $file->setPermanent();
        $file->save();
        
      } catch (\Exception $e) {
        \Drupal::logger('comfyui')->error('File save error: @error', [
          '@error' => $e->getMessage(),
        ]);
        
        return new JsonResponse([
          'success' => false,
          'message' => 'File save error: ' . $e->getMessage(),
        ]);
      }
      
      // Create media entity
      try {
        $current_user = \Drupal::currentUser();
        $media_name = $name ?: $files->getClientOriginalName();
        
        $media = $this->entityTypeManager()
          ->getStorage('media')
          ->create([
            'bundle' => 'comfyui_input',
            'uid' => $current_user->id(),
            'name' => $media_name,
            'field_media_image' => [
              'target_id' => $file->id(),
              'alt' => $media_name,
            ],
          ]);
        
        $media->save();
        
        \Drupal::logger('comfyui')->info('User uploaded media: @id (@name)', [
          '@id' => $media->id(),
          '@name' => $media_name,
        ]);
        
        return new JsonResponse([
          'success' => true,
          'message' => 'Media uploaded successfully',
          'media_id' => $media->id(),
          'media_name' => $media->label(),
        ]);
        
      } catch (\Exception $e) {
        \Drupal::logger('comfyui')->error('Media entity creation error: @error', [
          '@error' => $e->getMessage(),
        ]);
        
        return new JsonResponse([
          'success' => false,
          'message' => 'Media creation error: ' . $e->getMessage(),
        ]);
      }
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Media upload exception: @error', [
        '@error' => $e->getMessage(),
      ]);
      
      return new JsonResponse([
        'success' => false,
        'message' => 'Upload error: ' . $e->getMessage(),
      ]);
    }
  }

  /**
   * Get the hashed filename from a media entity.
   */
  protected function getHashedFilenameFromMedia($media) {
    try {
      if ($media->hasField('field_media_image')) {
        $image_field = $media->get('field_media_image');
        
        if (!$image_field->isEmpty()) {
          $file = $image_field->entity;
          
          if ($file) {
            // Return just the filename (with hash)
            return basename($file->getFileUri());
          }
        }
      }
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->warning('Could not get hashed filename: @error', [
        '@error' => $e->getMessage()
      ]);
    }
    
    return NULL;
  }

  /**
   * AJAX endpoint for saving feedback to existing media.
   */
  public function saveFeedbackAjax($media_id, Request $request) {
    try {
      $media = $this->entityTypeManager()
        ->getStorage('media')
        ->load($media_id);

      if (!$media) {
        return new JsonResponse([
          'success' => false,
          'message' => 'Media not found'
        ]);
      }

      // Check access
      if (!$media->access('update')) {
        return new JsonResponse([
          'success' => false,
          'message' => 'Access denied'
        ]);
      }

      $feedback_rating = $request->request->get('feedback_rating');
      $feedback_text = $request->request->get('feedback_text');

      // Update feedback fields if they exist
      if ($feedback_rating && $media->hasField('field_media_feedback_rating')) {
        $media->set('field_media_feedback_rating', $feedback_rating);
      }

      if ($feedback_text && $media->hasField('field_media_feedback_text')) {
        $media->set('field_media_feedback_text', $feedback_text);
      }

      $media->save();

      \Drupal::logger('comfyui')->info(
        'Updated feedback for media @id: rating=@rating',
        ['@id' => $media_id, '@rating' => $feedback_rating]
      );

      return new JsonResponse([
        'success' => true,
        'message' => 'Feedback saved successfully',
        'media_id' => $media_id,
      ]);

    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Save feedback failed: @error', [
        '@error' => $e->getMessage()
      ]);
      return new JsonResponse([
        'success' => false,
        'message' => $e->getMessage()
      ]);
    }
  }

  /**
   * Access callback for save feedback endpoint.
   */
  public static function saveFeedbackAccess($media_id) {
    $media = \Drupal::entityTypeManager()
      ->getStorage('media')
      ->load($media_id);

    if (!$media) {
      return \Drupal\Core\Access\AccessResult::forbidden();
    }

    return $media->access('update', NULL, TRUE);
  }

}