<?php

namespace Drupal\comfyui\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;

/**
 * Defines the ComfyUI Workflow entity type.
 *
 * @ContentEntityType(
 *   id = "comfyui_workflow",
 *   label = @Translation("ComfyUI Workflow"),
 *   label_collection = @Translation("ComfyUI Workflows"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\comfyui\ComfyUIWorkflowListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "add" = "Drupal\comfyui\Form\ComfyUIWorkflowForm",
 *       "edit" = "Drupal\comfyui\Form\ComfyUIWorkflowForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm"
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     }
 *   },
 *   base_table = "comfyui_workflow",
 *   admin_permission = "administer comfyui workflows",
 *   field_ui_base_route = "entity.comfyui_workflow.collection",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "title",
 *     "uuid" = "uuid"
 *   },
 *   links = {
 *     "canonical" = "/admin/structure/comfyui-workflow/{comfyui_workflow}",
 *     "add-form" = "/admin/structure/comfyui-workflow/add",
 *     "edit-form" = "/admin/structure/comfyui-workflow/{comfyui_workflow}/edit", 
 *     "delete-form" = "/admin/structure/comfyui-workflow/{comfyui_workflow}/delete",
 *     "collection" = "/admin/structure/comfyui-workflow"
 *   }
 * )
 */
class ComfyUIWorkflow extends ContentEntityBase {

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Title'))
      ->setRequired(true)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['description'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Description'))
      ->setDescription(t('A description of this workflow.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'text_default',
        'weight' => -3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'text_textarea',
        'weight' => -3,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['workflow_json'] = BaseFieldDefinition::create('file')
      ->setLabel(t('Workflow JSON'))
      ->setDescription(t('Upload the ComfyUI workflow JSON file.'))
      ->setRequired(TRUE)
      ->setSettings([
        'file_directory' => 'comfyui_workflows',
        'file_extensions' => 'json',
        'uri_scheme' => 'public',
        'handlers' => ['file'],
      ])
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'file_default',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'file_generic',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['workflow_api'] = BaseFieldDefinition::create('file')
      ->setLabel(t('Workflow API JSON'))
      ->setDescription(t('Upload the ComfyUI workflow API JSON file.'))
      ->setRequired(TRUE)
      ->setSettings([
        'file_directory' => 'comfyui_workflows',
        'file_extensions' => 'json',
        'uri_scheme' => 'public',
        'handlers' => ['file'],
      ])
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'file_default',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'file_generic',
        'weight' => 1,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Add this field back to baseFieldDefinitions():
    $fields['field_mappings'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Field Mappings'))
      ->setDescription(t('Drupal field to ComfyUI node mappings.'))
      ->setDefaultValue([])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['node_mappings'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Node Mappings'))
      ->setDescription(t('Custom node type to git repository mappings.'))
      ->setDefaultValue([])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['discovered_nodes'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Discovered Nodes'))
      ->setDescription(t('Auto-discovered custom node types from workflow.'))
      ->setDefaultValue([])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);


    $fields['model_mappings'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Model Mappings'))
      ->setDescription(t('Model filename to download URL and path mappings.'))
      ->setDefaultValue([])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['discovered_models'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Discovered Models'))
      ->setDescription(t('Auto-discovered model references from workflow.'))
      ->setDefaultValue([])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    // Auto-discovered fields from workflow JSON
    $fields['discovered_fields'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Discovered Fields'))
      ->setDescription(t('Auto-discovered input and output fields from workflow JSON.'))
      ->setDefaultValue([])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['bypass_mappings'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Bypass Mappings'))
      ->setDescription(t('Boolean field to node/group bypass directive mappings.'))
      ->setDefaultValue([])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Status'))
      ->setDefaultValue(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => 10,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time the workflow was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time the workflow was last edited.'));

    return $fields;
  }

  /**
   * Gets the workflow JSON content.
   */
  public function getWorkflowJson() {
    if ($this->workflow_json->target_id) {
      $file = \Drupal::entityTypeManager()->getStorage('file')->load($this->workflow_json->target_id);
      if ($file) {
        $file_uri = $file->getFileUri();
        $real_path = \Drupal::service('file_system')->realpath($file_uri);
        \Drupal::logger('comfyui')->info('Loading workflow JSON from: @path', ['@path' => $real_path]);
        
        if ($real_path && file_exists($real_path)) {
          $content = file_get_contents($real_path);
          \Drupal::logger('comfyui')->info('Workflow JSON content length: @len', ['@len' => strlen($content)]);
          return $content;
        } else {
          \Drupal::logger('comfyui')->error('Workflow JSON file not found at: @path', ['@path' => $real_path ?: 'unknown']);
        }
      } else {
        \Drupal::logger('comfyui')->error('Workflow JSON file entity not found for ID: @id', ['@id' => $this->workflow_json->target_id]);
      }
    } else {
      \Drupal::logger('comfyui')->warning('No workflow JSON file ID set');
    }
    return NULL;
  }

  /**
   * Gets the workflow API JSON content.
   */
  public function getWorkflowApi() {
    if ($this->workflow_api->target_id) {
      $file = \Drupal::entityTypeManager()->getStorage('file')->load($this->workflow_api->target_id);
      if ($file) {
        $file_uri = $file->getFileUri();
        $real_path = \Drupal::service('file_system')->realpath($file_uri);
        \Drupal::logger('comfyui')->info('Loading workflow API from: @path', ['@path' => $real_path]);
        
        if ($real_path && file_exists($real_path)) {
          $content = file_get_contents($real_path);
          \Drupal::logger('comfyui')->info('Workflow API content length: @len', ['@len' => strlen($content)]);
          return $content;
        } else {
          \Drupal::logger('comfyui')->error('Workflow API file not found at: @path', ['@path' => $real_path ?: 'unknown']);
        }
      } else {
        \Drupal::logger('comfyui')->error('Workflow API file entity not found for ID: @id', ['@id' => $this->workflow_api->target_id]);
      }
    } else {
      \Drupal::logger('comfyui')->warning('No workflow API file ID set');
    }
    return NULL;
  }

  /**
   * Gets discovered fields.
   */
  public function getDiscoveredFields() {
    $field = $this->get('discovered_fields');
    if ($field && ($item = $field->first())) {
      return $item->getValue() ?: [];
    }
    return [];
  }

  /**
   * Sets discovered fields.
   */
  public function setDiscoveredFields(array $discovered_fields) {
    $this->set('discovered_fields', $discovered_fields);
    return $this;
  }

  /**
   * Gets field mappings (shared across all view modes).
   */
  public function getFieldMappings() {
    $field = $this->get('field_mappings');
    if ($field && ($item = $field->first())) {
      return $item->getValue() ?: [];
    }
    return [];
  }

  /**
   * Sets field mappings (shared across all view modes).
   */
  public function setFieldMappings(array $field_mappings) {
    $this->set('field_mappings', $field_mappings);
    return $this;
  }

  /**
   * Gets input mappings.
   */
  public function getInputMappings() {
    $field_mappings = $this->getFieldMappings();
    return array_filter($field_mappings, function($mapping) {
      return isset($mapping['direction']) && $mapping['direction'] === 'input';
    });
  }

  /**
   * Gets output mappings.
   */
  public function getOutputMappings() {
    $field_mappings = $this->getFieldMappings();
    return array_filter($field_mappings, function($mapping) {
      return isset($mapping['direction']) && $mapping['direction'] === 'output';
    });
  }

  /**
   * Get available view modes for this entity type.
   */
  public function getAvailableViewModes() {
    $display_repository = \Drupal::service('entity_display.repository');
    return $display_repository->getViewModeOptions('comfyui_workflow');
  }


  /**
   * Gets node mappings.
   */
  public function getNodeMappings() {
    $field = $this->get('node_mappings');
    if ($field && ($item = $field->first())) {
      return $item->getValue() ?: [];
    }
    return [];
  }

  /**
   * Sets node mappings.
   */
  public function setNodeMappings(array $node_mappings) {
    $this->set('node_mappings', $node_mappings);
    return $this;
  }

  /**
   * Gets discovered nodes.
   */
  public function getDiscoveredNodes() {
    $field = $this->get('discovered_nodes');
    if ($field && ($item = $field->first())) {
      return $item->getValue() ?: [];
    }
    return [];
  }

  /**
   * Sets discovered nodes.
   */
  public function setDiscoveredNodes(array $discovered_nodes) {
    $this->set('discovered_nodes', $discovered_nodes);
    return $this;
  }

  /**
   * Gets model mappings.
   */
  public function getModelMappings() {
    $field = $this->get('model_mappings');
    if ($field && ($item = $field->first())) {
      return $item->getValue() ?: [];
    }
    return [];
  }

  /**
   * Sets model mappings.
   */
  public function setModelMappings(array $model_mappings) {
    $this->set('model_mappings', $model_mappings);
    return $this;
  }

  /**
   * Gets discovered models.
   */
  public function getDiscoveredModels() {
    $field = $this->get('discovered_models');
    if ($field && ($item = $field->first())) {
      return $item->getValue() ?: [];
    }
    return [];
  }

  /**
   * Sets discovered models.
   */
  public function setDiscoveredModels(array $discovered_models) {
    $this->set('discovered_models', $discovered_models);
    return $this;
  }

  /**
   * Gets bypass mappings (field to group/node bypass directives).
   */
  public function getBypassMappings() {
    $field = $this->get('bypass_mappings');
    if ($field && ($item = $field->first())) {
      return $item->getValue() ?: [];
    }
    return [];
  }

  /**
   * Sets bypass mappings.
   */
  public function setBypassMappings(array $bypass_mappings) {
    $this->set('bypass_mappings', $bypass_mappings);
    return $this;
  }

}