<?php

namespace Drupal\comfyui\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\comfyui\Service\ComfyUIWorkflowExecutionService;
use Drupal\comfyui\Entity\ComfyUIWorkflow;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\InvokeCommand;

/**
 * Dynamic workflow form that uses Drupal's field system.
 */
class ComfyUIDynamicWorkflowForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The workflow execution service.
   *
   * @var \Drupal\comfyui\Service\ComfyUIWorkflowExecutionService
   */
  protected $executionService;

  /**
   * Constructs a ComfyUIDynamicWorkflowForm object.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, ComfyUIWorkflowExecutionService $execution_service) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->executionService = $execution_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('comfyui.workflow_execution')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'comfyui_dynamic_workflow_form';
  }

  /**
   * Get field mappings from the workflow configuration.
   */
  protected function getFieldMappings(ComfyUIWorkflow $workflow) {
    return $workflow->getFieldMappings();
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ComfyUIWorkflow $workflow = NULL, $view_mode = 'form') {
    if (!$workflow) {
      $form['error'] = [
        '#markup' => $this->t('No workflow specified.'),
      ];
      return $form;
    }

    // Store the workflow entity and config type
    $form_state->set('workflow', $workflow);
    $form_state->set('view_mode', $view_mode);

    // Get field mappings for this configuration
    $field_mappings = $this->getFieldMappings($workflow);
    
    if (empty($field_mappings)) {
      $form['error'] = [
        '#markup' => $this->t('No field mappings found for this workflow. Please configure field mappings in the admin interface.'),
      ];
      return $form;
    }

    // Build input fields using Drupal's field system
    $this->buildInputFields($form, $form_state, $workflow, $field_mappings);

    // Add submit button
    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Execute Workflow'),
      '#ajax' => [
        'callback' => '::ajaxSubmitCallback',
        'wrapper' => 'workflow-results',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Executing workflow...'),
        ],
      ],
    ];

    // Build output fields area
    $form['results'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'workflow-results'],
      '#weight' => 100,
    ];

    // If there are execution results, display them
    if ($form_state->get('execution_results')) {
      $this->buildResultsDisplay($form, $form_state, $workflow, $field_mappings);
    }

    // Attach libraries
    $form['#attached']['library'][] = 'comfyui/workflow_form';

    return $form;
  }

  /**
   * Build input fields using Drupal's field system.
   */
  protected function buildInputFields(array &$form, FormStateInterface $form_state, ComfyUIWorkflow $workflow, $field_mappings) {
    if (empty($field_mappings)) {
      return;
    }

    $form['inputs'] = [
      '#type' => 'details',
      '#title' => $this->t('Input Parameters'),
      '#open' => TRUE,
    ];

    foreach ($field_mappings as $field_name => $mapping) {
      if ($mapping['direction'] === 'input') {
        // Get the field definition to determine field type
        $field_definitions = $this->entityFieldManager->getFieldDefinitions('comfyui_workflow', 'comfyui_workflow');
        
        if (isset($field_definitions[$field_name])) {
          $field_definition = $field_definitions[$field_name];
          $field_type = $field_definition->getType();
          
          // Build form element based on field type (not using widgets)
          $form['inputs'][$field_name] = $this->buildFormElementForFieldType(
            $field_name, 
            $field_type, 
            $field_definition,
            $mapping
          );
        }
      }
    }
  }

  /**
   * Build a form element based on field type without using widgets.
   */
  protected function buildFormElementForFieldType($field_name, $field_type, $field_definition, $mapping) {
    $element = [
      '#title' => $field_definition->getLabel(),
      '#description' => $field_definition->getDescription(),
      '#required' => $field_definition->isRequired(),
    ];

    // Map field types to form element types
    switch ($field_type) {
      case 'string':
      case 'string_long':
        $element['#type'] = 'textfield';
        break;
        
      case 'text':
      case 'text_long':
        $element['#type'] = 'textarea';
        break;
        
      case 'integer':
        $element['#type'] = 'number';
        $element['#step'] = 1;
        break;
        
      case 'decimal':
      case 'float':
        $element['#type'] = 'number';
        $element['#step'] = 0.01;
        break;
        
      case 'boolean':
        $element['#type'] = 'checkbox';
        break;
        
      case 'email':
        $element['#type'] = 'email';
        break;
        
      case 'file':
      case 'image':
        $element['#type'] = 'file';
        break;
        
      default:
        $element['#type'] = 'textfield';
    }

    return $element;
  }

  /**
   * Build results display using field formatters.
   */
  protected function buildResultsDisplay(array &$form, FormStateInterface $form_state, ComfyUIWorkflow $workflow, $field_mappings) {
    $execution_results = $form_state->get('execution_results');
    $field_definitions = $this->entityFieldManager->getFieldDefinitions('comfyui_workflow', 'comfyui_workflow');

    $form['results']['title'] = [
      '#type' => 'html_tag',
      '#tag' => 'h3',
      '#value' => $this->t('Results'),
      '#attributes' => ['class' => ['results-title']],
    ];

    foreach ($field_mappings as $field_name => $mapping) {
      if ($mapping['direction'] !== 'output') {
        continue;
      }

      if (!isset($execution_results[$field_name]) || !isset($field_definitions[$field_name])) {
        continue;
      }

      $field_definition = $field_definitions[$field_name];
      $result_data = $execution_results[$field_name];

      $form['results'][$field_name] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['field-result', 'field-result--' . str_replace('_', '-', $field_name)]],
      ];

      $form['results'][$field_name]['label'] = [
        '#type' => 'html_tag',
        '#tag' => 'h4',
        '#value' => $field_definition->getLabel(),
        '#attributes' => ['class' => ['field-label']],
      ];

      // Format the result based on type
      $form['results'][$field_name]['content'] = $this->formatResult($result_data, $field_definition);
    }
  }

  /**
   * Format result data for display.
   */
  protected function formatResult($result_data, $field_definition) {
    if (empty($result_data)) {
      return [
        '#markup' => '<p>' . $this->t('No result generated.') . '</p>',
      ];
    }

    $field_type = $field_definition->getType();

    switch ($field_type) {
      case 'image':
        return $this->formatImageResult($result_data);

      case 'text':
      case 'text_long':
      case 'string':
        return $this->formatTextResult($result_data);

      case 'file':
        return $this->formatFileResult($result_data);

      default:
        return $this->formatGenericResult($result_data);
    }
  }

  /**
   * Format image results.
   */
  protected function formatImageResult($result_data) {
    if (!is_array($result_data)) {
      $result_data = [$result_data];
    }

    $images = [];
    foreach ($result_data as $image_data) {
      if (isset($image_data['url'])) {
        $images[] = [
          '#theme' => 'image',
          '#uri' => $image_data['url'],
          '#alt' => $image_data['filename'] ?? 'Generated image',
          '#attributes' => ['class' => ['workflow-result-image']],
        ];
      }
    }

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['image-results']],
      'images' => $images,
    ];
  }

  /**
   * Format text results.
   */
  protected function formatTextResult($result_data) {
    $content = is_array($result_data) ? $result_data['content'] ?? '' : $result_data;
    
    return [
      '#type' => 'processed_text',
      '#text' => $content,
      '#format' => 'plain_text',
      '#attributes' => ['class' => ['text-result']],
    ];
  }

  /**
   * Format file results.
   */
  protected function formatFileResult($result_data) {
    return [
      '#markup' => '<p>' . $this->t('File result: @data', ['@data' => json_encode($result_data)]) . '</p>',
    ];
  }

  /**
   * Format generic results.
   */
  protected function formatGenericResult($result_data) {
    return [
      '#markup' => '<pre>' . json_encode($result_data, JSON_PRETTY_PRINT) . '</pre>',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This method is required but AJAX submission is handled separately
  }

  /**
   * AJAX submit callback.
   */
  public function ajaxSubmitCallback(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    // Get workflow and config type
    $workflow = $form_state->get('workflow');
    $view_mode = $form_state->get('view_mode');

    if (!$workflow) {
      $response->addCommand(new HtmlCommand('#workflow-results', '<div class="messages messages--error">' . $this->t('Form configuration error.') . '</div>'));
      return $response;
    }

    try {
      // Extract and transform input values
      $input_values = $this->extractInputValues($form_state, $workflow, $view_mode);

      // Execute the workflow using new service method
      $execution_result = $this->executionService->executeWorkflowWithViewMode(
        $workflow->id(),
        $input_values,
        $view_mode
      );

      if ($execution_result['success']) {
        $prompt_id = $execution_result['data']['prompt_id'];
        
        // Get formatted results using new service method
        $results = $this->executionService->getFormattedResultsWithViewMode(
          $prompt_id,
          $workflow->id(),
          $view_mode
        );

        $form_state->set('execution_results', $results);

        // Rebuild the form to show results
        $form_state->setRebuild(TRUE);
        
        $response->addCommand(new HtmlCommand('#workflow-results', $form['results']));
        $response->addCommand(new InvokeCommand('#workflow-results', 'addClass', ['results-loaded']));
        
      } else {
        $error_message = $execution_result['message'] ?? 'Unknown error occurred.';
        $response->addCommand(new HtmlCommand('#workflow-results', 
          '<div class="messages messages--error">' . $this->t('Execution failed: @error', ['@error' => $error_message]) . '</div>'
        ));
      }

    } catch (\Exception $e) {
      $this->getLogger('comfyui')->error('Form submission error: @error', ['@error' => $e->getMessage()]);
      $response->addCommand(new HtmlCommand('#workflow-results', 
        '<div class="messages messages--error">' . $this->t('An error occurred during execution.') . '</div>'
      ));
    }

    return $response;
  }

  /**
   * Extract input values from form state and transform them.
   */
  protected function extractInputValues(FormStateInterface $form_state, ComfyUIWorkflow $workflow, $view_mode) {
    $input_values = [];
    $field_mappings = $workflow->getFieldMappings();
    $all_values = $form_state->getValues();

    foreach ($field_mappings as $field_name => $mapping) {
      if ($mapping['direction'] !== 'input') {
        continue;
      }

      if (isset($all_values[$field_name])) {
        $field_value = $all_values[$field_name];
        
        // Transform the field value to ComfyUI format
        $transformed_value = $this->transformFieldValue($field_value, $mapping);
        
        if ($transformed_value !== NULL) {
          $input_values[$field_name] = $transformed_value;
        }
      }
    }

    return $input_values;
  }

  /**
   * Transform Drupal field values to ComfyUI parameter format.
   */
  protected function transformFieldValue($field_value, $mapping) {
    // Handle different field types and extract the actual value
    if (is_array($field_value)) {
      // Multi-value field or structured field
      if (isset($field_value[0]['value'])) {
        // Text field
        return $field_value[0]['value'];
      } elseif (isset($field_value[0]['target_id'])) {
        // Entity reference field
        return $field_value[0]['target_id'];
      } elseif (isset($field_value[0])) {
        // Other structured field
        return $field_value[0];
      }
    }

    return $field_value;
  }
}
