<?php

namespace Drupal\comfyui\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\AddClassCommand;
use Drupal\Core\Ajax\RemoveClassCommand;
use GuzzleHttp\Exception\RequestException;
use Drupal\Core\Config\ConfigManagerInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Provides a form for ComfyUI integration settings.
 */
class ComfyUISettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'comfyui_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['comfyui.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('comfyui.settings');

    $form['connection_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Connection Settings'),
      '#open' => TRUE,
    ];

    $form['connection_settings']['api_endpoint'] = [
      '#type' => 'url',
      '#title' => $this->t('ComfyUI API Endpoint'),
      '#default_value' => $config->get('api_endpoint') ?: 'http://localhost:8188',
      '#description' => $this->t('The URL of your ComfyUI API endpoint.'),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateTestButton',
        'event' => 'change',
        'wrapper' => 'test-connection-wrapper',
        'progress' => ['type' => 'throbber'],
      ],
    ];

    $form['connection_settings']['test_connection_wrapper'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'test-connection-wrapper',
        'class' => ['test-connection-wrapper'],
      ],
    ];

    $form['connection_settings']['test_connection_wrapper']['test_connection'] = [
      '#type' => 'button',
      '#value' => $this->t('Test Connection'),
      '#ajax' => [
        'callback' => '::testConnectionAjax',
        'progress' => ['type' => 'throbber'],
      ],
      '#attributes' => [
        'class' => ['button--primary'],
      ],
      '#prefix' => '<div class="test-connection-button">',
      '#suffix' => '</div>',
    ];

    $form['connection_settings']['test_connection_wrapper']['connection_status'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['connection-status-wrapper'],
      ],
      'message' => [
        '#type' => 'markup',
        '#markup' => '',
        '#prefix' => '<div class="connection-status-message">',
        '#suffix' => '</div>',
      ],
    ];

    $form['queue_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Queue Settings'),
      '#open' => TRUE,
    ];

    $form['queue_settings']['queue_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Queue Size'),
      '#default_value' => $config->get('queue_size') ?: 10,
      '#description' => $this->t('Maximum number of workflow executions that can be queued.'),
      '#min' => 1,
      '#required' => TRUE,
    ];

    $form['queue_settings']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Request Timeout'),
      '#default_value' => $config->get('timeout') ?: 30,
      '#description' => $this->t('Timeout in seconds for API requests.'),
      '#min' => 1,
      '#required' => TRUE,
    ];

    $form['manager_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('ComfyUI Manager Integration'),
      '#open' => FALSE,
      '#description' => $this->t('Configure URLs for fetching ComfyUI Manager node database and extension mappings.'),
    ];

    $form['manager_settings']['manager_node_map_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Extension Node Map URL'),
      '#default_value' => $config->get('manager_node_map_url') ?: 
        'https://raw.githubusercontent.com/Comfy-Org/ComfyUI-Manager/main/extension-node-map.json',
      '#description' => $this->t('URL to the extension-node-map.json file that maps node types to repositories.'),
      '#required' => TRUE,
    ];

    $form['manager_settings']['manager_custom_node_list_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Custom Node List URL'),
      '#default_value' => $config->get('manager_custom_node_list_url') ?: 
        'https://raw.githubusercontent.com/Comfy-Org/ComfyUI-Manager/main/custom-node-list.json',
      '#description' => $this->t('URL to the custom-node-list.json file that contains repository installation details.'),
      '#required' => TRUE,
    ];

    $form['manager_settings']['manager_model_list_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Model List URL'),
      '#default_value' => $config->get('manager_model_list_url') ?: 
        'https://raw.githubusercontent.com/Comfy-Org/ComfyUI-Manager/main/model-list.json',
      '#description' => $this->t('URL to the model-list.json file that contains model download information.'),
      '#required' => TRUE,
    ];

    $form['manager_settings']['clear_manager_cache'] = [
      '#type' => 'submit',
      '#value' => $this->t('Clear Manager Cache'),
      '#submit' => ['::clearManagerCache'],
      '#limit_validation_errors' => [],
      '#attributes' => [
        'class' => ['button--secondary'],
      ],
    ];

    $form['manager_settings']['cache_info'] = [
      '#type' => 'markup',
      '#markup' => '<p class="description">' . 
        $this->t('Node mapping data is cached for 1 hour. Clear the cache if you need to fetch fresh data from ComfyUI Manager immediately.') . 
        '</p>',
    ];

    $form['media_feedback_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Media Feedback Settings'),
      '#open' => TRUE,
      '#description' => $this->t('Configure feedback collection for generated media.'),
    ];

    $form['media_feedback_settings']['enable_media_feedback'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Media Feedback'),
      '#description' => $this->t('Allow authors to provide feedback (thumbs up/down and comments) on generated media. Feedback fields will appear in the media edit form.'),
      '#default_value' => $config->get('enable_media_feedback') ?? TRUE,
    ];

    $form['media_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Media Entity Settings'),
      '#open' => FALSE,
      '#description' => $this->t('Configure how ComfyUI images (input and generated) are stored and secured.'),
    ];

    $form['media_settings']['comfyui_file_scheme'] = [
      '#type' => 'radios',
      '#title' => $this->t('File Storage Scheme'),
      '#options' => [
        'public' => $this->t('Public (Docker-friendly) - Files stored in web-accessible directory'),
        'private' => $this->t('Private (Enhanced Security) - Files stored outside webroot'),
      ],
      '#default_value' => $config->get('comfyui_file_scheme') ?: 'public',
      '#description' => $this->t('Applies to both input images and generated images. All file access is protected by time-limited security tokens. Choose based on your deployment and security requirements.'),
    ];

    $form['media_settings']['auto_save_generated_images'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-save Generated Images'),
      '#description' => $this->t('Automatically save all generated images as media entities. When disabled, images are only available via temporary proxy URLs.'),
      '#default_value' => $config->get('auto_save_generated_images') ?: TRUE,
    ];

    $form['media_settings']['token_expiration'] = [
      '#type' => 'number',
      '#title' => $this->t('Token Expiration Time'),
      '#description' => $this->t('Time in seconds before temporary access tokens expire. Applies to all media file access.'),
      '#default_value' => $config->get('token_expiration') ?: 3600,
      '#min' => 300,
      '#max' => 86400,
      '#step' => 300,
      '#field_suffix' => $this->t('seconds'),
    ];

    $form['media_settings']['info'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--info">' . 
        $this->t('<strong>Security:</strong> All ComfyUI media (input and generated) require valid, time-limited security tokens for access. Tokens expire after the configured duration. Storage location choice affects only physical file location, not security.') . 
        '</div>',
    ];

    $form['field_validation'] = [
      '#type' => 'details',
      '#title' => $this->t('Field Validation Settings'),
      '#open' => FALSE,
    ];

    $form['field_validation']['allowed_field_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed Field Types'),
      '#options' => [
        'text' => $this->t('Text'),
        'text_long' => $this->t('Long text'),
        'text_with_summary' => $this->t('Text with summary'),
        'string' => $this->t('String'),
        'string_long' => $this->t('Long string'),
        'integer' => $this->t('Integer'),
        'float' => $this->t('Float'),
        'decimal' => $this->t('Decimal'),
        'boolean' => $this->t('Boolean'),
        'email' => $this->t('Email'),
        'telephone' => $this->t('Telephone'),
        'uri' => $this->t('URL'),
        'image' => $this->t('Image'),
        'file' => $this->t('File'),
        'entity_reference' => $this->t('Entity Reference'),
      ],
      '#default_value' => $config->get('allowed_field_types') ?: [],
      '#description' => $this->t('Select which field types can be used for mapping workflow parameters.'),
    ];

    $form['field_validation']['validate_field_types'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Validate Field Types'),
      '#description' => $this->t('Ensure field types match parameter types when mapping.'),
      '#default_value' => $config->get('validate_field_types') ?: TRUE,
    ];

    $form['field_validation']['strict_validation'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Strict Validation'),
      '#description' => $this->t('Prevent mapping fields with incompatible types.'),
      '#default_value' => $config->get('strict_validation') ?: FALSE,
      '#states' => [
        'visible' => [
          ':input[name="validate_field_types"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Add CSS for styling the connection status
    $form['#attached']['library'][] = 'comfyui/admin_styles';

    return parent::buildForm($form, $form_state);
  }

public function testConnectionAjax(array &$form, FormStateInterface $form_state) {
  $response = new AjaxResponse();
  $endpoint = $form_state->getValue('api_endpoint') ?: $form['connection_settings']['api_endpoint']['#default_value'];

  try {
    // Build request options
    $request_options = [
      'timeout' => 5,
      'connect_timeout' => 5,
    ];
    
    // Add authentication if login module is enabled
    if (\Drupal::moduleHandler()->moduleExists('comfyui_login')) {
      $token_service = \Drupal::service('comfyui_login.token_service');
      $request_options = $token_service->buildRequestOptions($request_options);
    }
    
    // Test 1: Basic connectivity
    $api_response = \Drupal::service('http_client')->request('GET', $endpoint, $request_options);

    if ($api_response->getStatusCode() === 200) {
      $class = 'status-success';
      $message = $this->t('✅ ComfyUI API accessible.');
      
      // Test 2: object_info endpoint availability (lightweight test)
      try {
        $test_response = \Drupal::service('http_client')->request('HEAD', $endpoint . '/object_info', $request_options);
        
        if ($test_response->getStatusCode() === 200) {
          $message .= ' ' . $this->t('✅ object_info endpoint available.');
          
          // Test 3: Queue endpoint (critical for workflow status)
          try {
            $queue_response = \Drupal::service('http_client')->request('GET', $endpoint . '/queue', $request_options);
            
            if ($queue_response->getStatusCode() === 200) {
              $queue_data = json_decode($queue_response->getBody(), TRUE);
              if (isset($queue_data['queue_running']) || isset($queue_data['queue_pending'])) {
                $message .= ' ' . $this->t('✅ Queue endpoint accessible.');
              } else {
                $message .= ' ' . $this->t('⚠️ Queue endpoint returned unexpected data.');
              }
            }
          } catch (\Exception $e) {
            $message .= ' ' . $this->t('❌ Queue endpoint failed - workflow status checks will not work. Error: @error', [
              '@error' => $e->getMessage()
            ]);
            $class = 'status-error';
          }
          
          // Test 4: Sample node type fetch (if queue test passed)
          if ($class !== 'status-error') {
            try {
              $sample_response = \Drupal::service('http_client')->request('GET', $endpoint . '/object_info/CheckpointLoaderSimple', $request_options);
              
              if ($sample_response->getStatusCode() === 200) {
                $sample_data = json_decode($sample_response->getBody(), TRUE);
                if (!empty($sample_data)) {
                  $message .= ' ' . $this->t('✅ Node definition fetching works.');
                }
              }
            } catch (\Exception $e) {
              $message .= ' ' . $this->t('⚠️ Sample node fetch failed - some dynamic options may not work.');
            }
          }

          // Test 5: Check Manager URLs accessibility
          try {
            // Check node map
            $manager_node_map_url = $form_state->getValue('manager_node_map_url') ?: 
              $config->get('manager_node_map_url') ?: 
              'https://raw.githubusercontent.com/Comfy-Org/ComfyUI-Manager/main/extension-node-map.json';
            
            $manager_response = \Drupal::service('http_client')->head($manager_node_map_url, [
              'timeout' => 5,
            ]);
            
            if ($manager_response->getStatusCode() === 200) {
              $message .= ' ' . $this->t('✅ Manager node database accessible.');
            }
            
            // Check model list
            $manager_model_list_url = $form_state->getValue('manager_model_list_url') ?: 
              $config->get('manager_model_list_url') ?: 
              'https://raw.githubusercontent.com/Comfy-Org/ComfyUI-Manager/main/model-list.json';
            
            $model_response = \Drupal::service('http_client')->head($manager_model_list_url, [
              'timeout' => 5,
            ]);
            
            if ($model_response->getStatusCode() === 200) {
              $message .= ' ' . $this->t('✅ Manager model database accessible.');
            }
          } catch (\Exception $e) {
            $message .= ' ' . $this->t('⚠️ Manager databases not fully accessible - mapping may be limited.');
          }
          
        } else {
          $message .= ' ' . $this->t('⚠️ object_info endpoint returned @code - dynamic options unavailable.', [
            '@code' => $test_response->getStatusCode()
          ]);
        }
      } catch (\Exception $e) {
        $message .= ' ' . $this->t('⚠️ object_info endpoint not accessible - dynamic options will be limited.');
      }
      
    } else {
      throw new RequestException('Unexpected response code: ' . $api_response->getStatusCode());
    }
  }
  catch (\Exception $e) {
    $class = 'status-error';
    $message = $this->t('❌ Failed to connect to ComfyUI API: @error', [
      '@error' => $e->getMessage(),
    ]);
  }

  $response->addCommand(new HtmlCommand(
    '.connection-status-message',
    [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['connection-status-message', $class],
      ],
      '#markup' => $message,
    ]
  ));

  return $response;
}

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate output directory
    $output_path = $form_state->getValue('default_output_path');
    $directory = \Drupal::service('file_system')->realpath($output_path);
    
    try {
      \Drupal::service('file_system')->prepareDirectory($output_path, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
    }
    catch (\Exception $e) {
      $form_state->setErrorByName('default_output_path', $this->t('The output directory could not be created or is not writable.'));
    }
  }

  /**
   * Clear manager cache submit handler.
   */
  public function clearManagerCache(array &$form, FormStateInterface $form_state) {
    try {
      $node_mapping_service = \Drupal::service('comfyui.node_mapping');
      $node_mapping_service->clearCache();
      
      $model_mapping_service = \Drupal::service('comfyui.model_mapping');
      $model_mapping_service->clearCache();
      
      $this->messenger()->addStatus($this->t('ComfyUI Manager cache (nodes & models) has been cleared. Fresh data will be fetched on the next discovery.'));

    } catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to clear cache: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
    
    // Don't redirect, stay on settings page
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('comfyui.settings')
      ->set('enable_media_feedback', $form_state->getValue('enable_media_feedback'))
      ->set('api_endpoint', rtrim($form_state->getValue('api_endpoint'), '/'))
      ->set('queue_size', $form_state->getValue('queue_size'))
      ->set('timeout', $form_state->getValue('timeout'))
      ->set('default_output_path', $form_state->getValue('default_output_path'))
      ->set('manager_node_map_url', $form_state->getValue('manager_node_map_url'))
      ->set('manager_custom_node_list_url', $form_state->getValue('manager_custom_node_list_url'))
      ->set('manager_model_list_url', $form_state->getValue('manager_model_list_url'))
      ->set('auto_save_generated_images', $form_state->getValue('auto_save_generated_images'))
      ->set('generated_image_uri_scheme', $form_state->getValue('generated_image_uri_scheme'))
      ->set('token_expiration', $form_state->getValue('token_expiration'))
      ->set('allowed_field_types', array_filter($form_state->getValue('allowed_field_types')))
      ->set('validate_field_types', $form_state->getValue('validate_field_types'))
      ->set('strict_validation', $form_state->getValue('strict_validation'))
      ->save();

    parent::submitForm($form, $form_state);
  }
}
