<?php

namespace Drupal\comfyui\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Provides a form for executing a ComfyUI workflow.
 */
class ComfyUIWorkflowExecuteForm extends ConfirmFormBase {

  /**
   * The workflow entity.
   *
   * @var \Drupal\comfyui\Entity\ComfyUIWorkflow
   */
  protected $workflow;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'comfyui_workflow_execute_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $comfyui_workflow = NULL) {
    $form = parent::buildForm($form, $form_state);
    $this->workflow = $comfyui_workflow;
    $form_state->set('workflow', $comfyui_workflow);

    if (!$this->workflow) {
      \Drupal::messenger()()->addError($this->t('No workflow specified.'));
      return $this->redirect('entity.comfyui_workflow.collection');
    }

    // Load the workflow's API JSON
    $workflow_api = json_decode($this->workflow->getWorkflowApi(), TRUE);
    if (empty($workflow_api)) {
      \Drupal::messenger()()->addError($this->t('Invalid workflow API data.'));
      return $form;
    }

    // Get field mappings from the workflow entity
    $field_mappings = $this->workflow->getFieldMappings();

    // Add editible fields container
    $form['workflow_inputs'] = [
      '#type' => 'details',
      '#title' => $this->t('Workflow Parameters'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    // Add form fields for each mapped input
    foreach ($workflow_api as $node_id => $node) {
      if (empty($node['inputs'])) {
        continue; 
      }

      $editable_inputs = 0;
      $title = $node['_meta']['title'] ?? $node['class_type'];
      
      // Count editable inputs
      foreach ($node['inputs'] as $input_name => $value) {
        if (!is_array($value) || !isset($value[0]) || !isset($value[1])) {
          $editable_inputs++;
        }
      }

      // Only create fieldset if there are editable inputs
      if ($editable_inputs > 0) {
        $form['workflow_inputs'][$node_id] = [
          '#type' => 'details',
          '#title' => $this->t('@title (Node @id)', ['@title' => $title, '@id' => $node_id]),
          '#open' => TRUE,
        ];

        // Add input fields
        foreach ($node['inputs'] as $input_name => $value) {
          if (!is_array($value) || !isset($value[0]) || !isset($value[1])) {
            $form['workflow_inputs'][$node_id][$input_name] = $this->createInputElement(
              $input_name,
              $value,
              $node['class_type']
            );
          }
        }
      }
    }

    return $form;
  }

  /**
   * Creates a form element for a workflow input.
   */
  protected function createInputElement($input_name, $value, $node_type) {
    $element = [];

    // Determine input type and create appropriate form element
    switch ($this->determineInputType($input_name, $value, $node_type)) {
      case 'integer':
        $element = [
          '#type' => 'number',
          '#step' => 1,
        ];
        break;

      case 'float':
        $element = [
          '#type' => 'number',
          '#step' => 'any',
        ];
        break;

      case 'boolean':
        $element = [
          '#type' => 'checkbox',
        ];
        break;

      default:
        $element = [
          '#type' => 'textfield',
        ];
    }

    // Common element properties
    $element['#title'] = $this->t('@name', ['@name' => $input_name]);
    $element['#default_value'] = $value;
    $element['#required'] = FALSE;

    return $element;
  }

  /**
   * Determines the type of an input.
   */
  protected function determineInputType($input_name, $value, $node_type) {
    // Common input types based on name patterns
    $type_patterns = [
      'seed|steps|width|height|batch_size' => 'integer',
      'cfg|denoise' => 'float',
      'text|filename_prefix|sampler_name|scheduler' => 'string',
      'boolean' => 'boolean',
    ];

    foreach ($type_patterns as $pattern => $type) {
      if (preg_match("/{$pattern}/i", $input_name)) {
        return $type;
      }
    }

    // Determine by value type
    if (is_int($value)) {
      return 'integer';
    }
    elseif (is_float($value)) {
      return 'float';
    }
    elseif (is_bool($value)) {
      return 'boolean';
    }

    return 'string';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Execute workflow %title?', [
      '%title' => $this->workflow ? $this->workflow->label() : $this->t('Unknown'),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.comfyui_workflow.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if (!$this->workflow) {
      $this->workflow = $form_state->get('workflow');
    }

    if (!$this->workflow) {
      \Drupal::messenger()()->addError($this->t('No workflow specified.'));
      $form_state->setRedirect('entity.comfyui_workflow.collection');
      return;
    }

    try {
      // Get the user-submitted input values
      $input_values = $form_state->getValue('workflow_inputs');

      // Execute the workflow with the input values
      $result = \Drupal::service('comfyui.workflow_execution')->executeWorkflow($this->workflow->id(), $input_values);

      if ($result['success']) {
        $prompt_id = $result['data']['prompt_id'];
        
        \Drupal::messenger()->addStatus($this->t('Workflow execution started. Prompt ID: @prompt_id', [
          '@prompt_id' => $prompt_id,
        ]));
        
        // Wait for execution to complete, then discover outputs
        // Note: In production, this should be done via queue/cron
        sleep(15);
        
        $discovery_result = \Drupal::service('comfyui.workflow_execution')
          ->discoverAndStoreOutputs($prompt_id, $this->workflow->id());
        
        if ($discovery_result) {
          \Drupal::messenger()->addStatus($this->t('Output discovery completed! You can now configure result mappings in the node mapping form.'));
        } else {
          \Drupal::messenger()->addWarning($this->t('Output discovery failed. You may need to run the workflow again or check the logs.'));
        }
      }
      else {
        \Drupal::messenger()()->addError($this->t('Workflow execution failed: @error', [
          '@error' => $result['error'],
        ]));
      }
    }
    catch (\Exception $e) {
      \Drupal::messenger()()->addError($this->t('An error occurred during workflow execution: @error', [
        '@error' => $e->getMessage(),
      ]));
    }

    $form_state->setRedirect('entity.comfyui_workflow.collection');
  }

}
