<?php

namespace Drupal\comfyui\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;

/**
 * Form for ComfyUI Workflow entities with integrated field mapping.
 */
class ComfyUIWorkflowForm extends EntityForm {

  /**
   * The entity field manager.
   */
  protected $entityFieldManager;

  /**
   * The bypass mapping service.
   */
  protected $bypassMappingService;

  /**
   * Constructs a new ComfyUIWorkflowForm object.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    $bypass_mapping_service = NULL
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->bypassMappingService = $bypass_mapping_service ?: \Drupal::service('comfyui.bypass_mapping');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('comfyui.bypass_mapping')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\comfyui\Entity\ComfyUIWorkflow $workflow */
    $workflow = $this->entity;
    $is_new = $workflow->isNew();

    // Add form mode indicator
    if ($is_new) {
      $form['form_mode_info'] = [
        '#markup' => '<div class="messages messages--status">' . 
          $this->t('Create a new ComfyUI workflow by uploading JSON files and configuring field mappings.') . 
          '</div>',
        '#weight' => -10,
      ];
    }

    // Basic workflow fields
    $form['basic_info'] = [
      '#type' => 'details',
      '#title' => $this->t('Basic Information'),
      '#open' => TRUE,
      '#weight' => -5,
    ];

    $form['basic_info']['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title'),
      '#default_value' => $workflow->get('title')->value ?? '',
      '#required' => TRUE,
      '#description' => $this->t('A descriptive name for this workflow.'),
    ];

    $form['basic_info']['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $workflow->get('description')->value ?? '',
      '#description' => $this->t('A description of what this workflow does.'),
    ];

    $form['basic_info']['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $workflow->get('status')->value ?? TRUE,
      '#description' => $this->t('Whether this workflow is enabled for use.'),
    ];

    // JSON Upload Section
    $form['json_files'] = [
      '#type' => 'details',
      '#title' => $this->t('Workflow Files'),
      '#open' => $is_new ? TRUE : FALSE, // Open for new workflows
      '#description' => $is_new ? 
        $this->t('Upload both JSON files from ComfyUI. After uploading, you can discover fields and create mappings below.') :
        $this->t('Update the workflow JSON files if needed.'),
      '#weight' => 0,
    ];

    $form['json_files']['workflow_json'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Workflow JSON'),
      '#default_value' => $workflow->get('workflow_json')->target_id ? [$workflow->get('workflow_json')->target_id] : [],
      '#upload_location' => 'public://comfyui_workflows',
      '#upload_validators' => [
        'FileExtension' => ['extensions' => 'json'],
      ],
      '#required' => $is_new,
      '#description' => $this->t('The main workflow JSON file from ComfyUI (Save → Export Workflow).'),
    ];

    $form['json_files']['workflow_api'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Workflow API JSON'),
      '#default_value' => $workflow->get('workflow_api')->target_id ? [$workflow->get('workflow_api')->target_id] : [],
      '#upload_location' => 'public://comfyui_workflows',
      '#upload_validators' => [
        'FileExtension' => ['extensions' => 'json'],
      ],
      '#required' => $is_new,
      '#description' => $this->t('The API format JSON from ComfyUI (Save → Export API Format).'),
    ];

    // Progress indicator for new workflows
    if ($is_new) {
      $form['progress'] = [
        '#type' => 'container',
        '#weight' => 5,
        '#attributes' => ['class' => ['workflow-creation-progress']],
      ];

      $has_files = !empty($workflow->get('workflow_json')->target_id) && !empty($workflow->get('workflow_api')->target_id);
      $has_discovered = !empty($workflow->getDiscoveredFields()['inputs']);

      $form['progress']['steps'] = [
        '#markup' => $this->buildProgressIndicator($has_files, $has_discovered),
      ];
    }

    // Configuration sections (only show if files are uploaded or it's an existing workflow)
    $show_config = !$is_new || (!empty($workflow->get('workflow_json')->target_id) && !empty($workflow->get('workflow_api')->target_id));


    if ($show_config) {
      // Node mapping section
      $form['node_mapping'] = [
        '#type' => 'details',
        '#title' => $this->t('Custom Node Dependencies'),
        '#open' => TRUE,
        '#description' => $this->t('Identify and map custom ComfyUI nodes to their repository sources for installation.'),
        '#weight' => 9,
        '#tree' => TRUE,
      ];

      $this->buildNodeMappingSection($form['node_mapping'], $form_state, $workflow);

      // Model mapping section
      $form['model_mapping'] = [
        '#type' => 'details',
        '#title' => $this->t('Model Dependencies'),
        '#open' => TRUE,
        '#description' => $this->t('Identify and map model files to their download sources.'),
        '#weight' => 9.5,
        '#tree' => TRUE,
      ];

      $this->buildModelMappingSection($form['model_mapping'], $form_state, $workflow);

      // Field mapping section
      $form['field_mapping'] = [
        '#type' => 'details',
        '#title' => $this->t('Field Mapping Configuration'),
        '#open' => FALSE,
        '#description' => $this->t('Configure how Drupal fields map to ComfyUI workflow inputs and outputs. These mappings are shared across all display modes.'),
        '#weight' => 10,
        '#tree' => TRUE,
      ];

      $this->buildFieldMappingSection($form['field_mapping'], $form_state, $workflow);

      // Bypass mapping section
      $form['bypass_mapping'] = [
        '#type' => 'details',
        '#title' => $this->t('Node & Group Bypass Configuration'),
        '#open' => FALSE,
        '#description' => $this->t('Configure Boolean fields to bypass workflow sections. When a Boolean field is unchecked, the mapped nodes or groups will be skipped during execution.'),
        '#weight' => 11,
        '#tree' => TRUE,
      ];

      $this->buildBypassMappingSection($form['bypass_mapping'], $form_state, $workflow);
    } else {
      $form['config_placeholder'] = [
        '#markup' => '<div class="messages messages--warning">' . 
          $this->t('Upload both JSON files above, then save to proceed with field mapping configuration.') . 
          '</div>',
        '#weight' => 10,
      ];
    }

    return $form;
  }

  /**
   * Build a progress indicator for workflow creation.
   */
  protected function buildProgressIndicator($has_files, $has_discovered) {
    $steps = [
      [
        'title' => $this->t('Upload JSON Files'),
        'completed' => $has_files,
        'current' => !$has_files,
      ],
      [
        'title' => $this->t('Discover Fields'),
        'completed' => $has_discovered,
        'current' => $has_files && !$has_discovered,
      ],
      [
        'title' => $this->t('Create Mappings'),
        'completed' => FALSE, // We could check if mappings exist
        'current' => $has_discovered,
      ],
    ];

    $output = '<div class="workflow-progress">';
    $output .= '<h4>' . $this->t('Workflow Creation Progress') . '</h4>';
    $output .= '<ol class="progress-steps">';

    foreach ($steps as $step) {
      $classes = ['progress-step'];
      if ($step['completed']) {
        $classes[] = 'completed';
      } elseif ($step['current']) {
        $classes[] = 'current';
      }

      $icon = $step['completed'] ? '✅' : ($step['current'] ? '🔄' : '⭕');
      
      $output .= '<li class="' . implode(' ', $classes) . '">';
      $output .= '<span class="step-icon">' . $icon . '</span>';
      $output .= '<span class="step-title">' . $step['title'] . '</span>';
      $output .= '</li>';
    }

    $output .= '</ol></div>';

    return $output;
  }

  /**
   * Build node mapping section.
   */
  protected function buildNodeMappingSection(array &$form, FormStateInterface $form_state, $workflow) {
    // Node discovery section
    $form['discovery'] = [
      '#type' => 'details',
      '#title' => $this->t('Node Discovery'),
      '#open' => TRUE,
    ];

    $form['discovery']['discover_nodes'] = [
      '#type' => 'submit',
      '#value' => $this->t('Discover Custom Nodes'),
      '#submit' => ['::discoverNodesSubmit'],
      '#ajax' => [
        'callback' => '::discoverNodesAjax',
        'wrapper' => 'node-mappings-wrapper',
      ],
      '#name' => 'discover_nodes',
      '#limit_validation_errors' => [],
    ];

    // Show discovered nodes summary
    $discovered_nodes = $workflow->getDiscoveredNodes();
    if (!empty($discovered_nodes)) {
      $custom_count = count($discovered_nodes);
      $configured_count = count(array_filter($workflow->getNodeMappings()));
      $pending_count = $custom_count - $configured_count;

      $form['discovery']['summary'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['discovered-nodes-summary']],
        'content' => [
          '#markup' => '<h4>' . $this->t('Discovery Results') . '</h4>' .
            '<div class="stat"><span class="stat-value">' . $custom_count . '</span> ' . $this->t('Custom Nodes') . '</div>' .
            '<div class="stat"><span class="stat-value">' . $configured_count . '</span> ' . $this->t('Configured') . '</div>' .
            '<div class="stat"><span class="stat-value">' . $pending_count . '</span> ' . $this->t('Pending') . '</div>',
        ],
      ];
    }

    // Node mappings section
    $form['node_mappings'] = [
      '#type' => 'container',
      '#prefix' => '<div id="node-mappings-wrapper">',
      '#suffix' => '</div>',
    ];

    $this->buildNodeMappingsTable($form['node_mappings'], $form_state, $workflow);
  }

  /**
   * Build model mapping section.
   */
  protected function buildModelMappingSection(array &$form, FormStateInterface $form_state, $workflow) {
    // Model discovery section
    $form['discovery'] = [
      '#type' => 'details',
      '#title' => $this->t('Model Discovery'),
      '#open' => TRUE,
    ];

    $form['discovery']['discover_models'] = [
      '#type' => 'submit',
      '#value' => $this->t('Discover Models'),
      '#submit' => ['::discoverModelsSubmit'],
      '#ajax' => [
        'callback' => '::discoverModelsAjax',
        'wrapper' => 'model-mappings-wrapper',
      ],
      '#name' => 'discover_models',
      '#limit_validation_errors' => [],
    ];

    // Show discovered models summary
    $discovered_models = $workflow->getDiscoveredModels();
    if (!empty($discovered_models)) {
      $total_count = count($discovered_models);
      $configured_count = count(array_filter($workflow->getModelMappings(), function($mapping) {
        return !empty($mapping['urls']);
      }));
      $from_database = count(array_filter($discovered_models, function($m) {
        return $m['source'] === 'comfyui_manager';
      }));
      $pending_count = $total_count - $configured_count;

      $form['discovery']['summary'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['discovered-nodes-summary']],
        'content' => [
          '#markup' => '<h4>' . $this->t('Discovery Results') . '</h4>' .
            '<div class="stat"><span class="stat-value">' . $total_count . '</span> ' . $this->t('Models') . '</div>' .
            '<div class="stat"><span class="stat-value">' . $from_database . '</span> ' . $this->t('In Database') . '</div>' .
            '<div class="stat"><span class="stat-value">' . $configured_count . '</span> ' . $this->t('Configured') . '</div>' .
            '<div class="stat"><span class="stat-value">' . $pending_count . '</span> ' . $this->t('Pending') . '</div>',
        ],
      ];
    }

    // Model mappings section
    $form['model_mappings'] = [
      '#type' => 'container',
      '#prefix' => '<div id="model-mappings-wrapper">',
      '#suffix' => '</div>',
    ];

    $this->buildModelMappingsTable($form['model_mappings'], $form_state, $workflow);
  }

  /**
   * Build model mappings table.
   */
  protected function buildModelMappingsTable(array &$form, FormStateInterface $form_state, $workflow) {
    $discovered_models = $workflow->getDiscoveredModels();
    $model_mappings = $workflow->getModelMappings();

    if (empty($discovered_models)) {
      $form['empty'] = [
        '#markup' => '<div class="messages messages--warning">' .
          $this->t('No models discovered. Click "Discover Models" to scan your workflow.') .
          '</div>',
      ];
      return;
    }

    $form['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Model Filename'),
        $this->t('Loader'),
        $this->t('Type'),
        $this->t('Save Path'),
        $this->t('Download URL'),
      ],
      '#attributes' => ['class' => ['node-mapping-table', 'model-mapping-table']],
    ];

    // Get available model types for dropdown
    $model_mapping_service = \Drupal::service('comfyui.model_mapping');
    $available_types = $model_mapping_service->getAvailableModelTypes();
    $type_options = array_combine($available_types, $available_types);

    foreach ($discovered_models as $filename => $model_info) {
      $row_class = !empty($model_mappings[$filename]['urls']) ? 'node-mapping-configured' : 'node-mapping-pending';
      $current_mapping = $model_mappings[$filename] ?? [];
      $source = $model_info['source'] ?? 'manual';

      // Filename column - just the filename
      $source_badge = $source === 'comfyui_manager' ? 
        '<span class="node-status-badge custom">Database</span>' : 
        '<span class="node-status-badge manual">Manual</span>';
      
      $form['table'][$filename]['filename'] = [
        '#markup' => '<div class="model-filename-cell">' .
          '<div class="filename-text">' . htmlspecialchars($filename) . '</div>' .
          $source_badge .
          '</div>',
      ];
      
      // Loader column
      $form['table'][$filename]['loader'] = [
        '#markup' => '<span class="loader-type-badge">' . htmlspecialchars($model_info['loader_type'] ?? '') . '</span>',
      ];

      // Type column - read-only if from database, editable if manual
      if ($source === 'comfyui_manager') {
        $type_value = $current_mapping['type'] ?? $model_info['type'] ?? 'unknown';
        $form['table'][$filename]['type'] = [
          '#markup' => '<strong>' . htmlspecialchars($type_value) . '</strong>',
        ];
      } else {
        $form['table'][$filename]['type'] = [
          '#type' => 'select',
          '#options' => $type_options,
          '#default_value' => $current_mapping['type'] ?? $model_info['type'] ?? 'checkpoint',
          '#attributes' => ['class' => ['model-type-select']],
        ];
      }

      // Save Path column
      $form['table'][$filename]['save_path'] = [
        '#type' => 'textfield',
        '#default_value' => $current_mapping['save_path'] ?? $model_info['save_path'] ?? '',
        '#placeholder' => 'models/checkpoints/',
        '#attributes' => ['class' => ['model-save-path']],
        '#size' => 40,
      ];

      // Download URLs column - single URL input
      $default_urls = $current_mapping['urls'] ?? $model_info['urls'] ?? [];
      $default_url = !empty($default_urls) ? (is_array($default_urls) ? $default_urls[0] : $default_urls) : '';

      if ($source === 'comfyui_manager' && count($default_urls) > 1) {
        // Multiple URLs from database - show all as read-only with select
        $url_options = [];
        foreach ($default_urls as $idx => $url) {
          $url_options[$url] = substr($url, 0, 60) . (strlen($url) > 60 ? '...' : '');
        }
        
        $form['table'][$filename]['url'] = [
          '#type' => 'select',
          '#options' => $url_options,
          '#default_value' => $default_url,
          '#attributes' => ['class' => ['model-url-select']],
          '#description' => $this->t('@count URLs available', ['@count' => count($default_urls)]),
        ];
      } else {
        // Single URL input
        $form['table'][$filename]['url'] = [
          '#type' => 'url',
          '#default_value' => $default_url,
          '#placeholder' => 'https://huggingface.co/.../model.safetensors',
          '#attributes' => ['class' => ['model-url']],
          '#size' => 60,
        ];
      }

      // Add row class
      $form['table'][$filename]['#attributes']['class'][] = $row_class;
    }
  }

  /**
   * Build field mapping section for a configuration type.
   */
  protected function buildFieldMappingSection(array &$form, FormStateInterface $form_state, $workflow) {
    // Field discovery section
    $form['discovery'] = [
      '#type' => 'details',
      '#title' => $this->t('Field Discovery'),
      '#open' => TRUE,
      '#description' => $this->t('Discover available input and output fields from the workflow JSON files.'),
    ];

    $form['discovery']['discover_fields'] = [
      '#type' => 'submit',
      '#value' => $this->t('Discover Node Definitions'),
      '#submit' => ['::discoverFieldsSubmit'],
      '#ajax' => [
        'callback' => '::discoverFieldsAjax',
        'wrapper' => "field-mappings-wrapper",
      ],
      '#name' => "discover_fields",
      '#limit_validation_errors' => [], // Prevent validation of other fields
    ];

    // Show discovered fields
    $this->buildDiscoveredFieldsDisplay($form, $form_state, $workflow);

    // Field mappings section
    $form['field_mappings'] = [
      '#type' => 'details',
      '#title' => $this->t('Field Mappings'),
      '#open' => TRUE,
      '#description' => $this->t('Map Drupal fields to ComfyUI inputs and outputs. These mappings will be used across all display modes.'),
      '#prefix' => "<div id=\"field-mappings-wrapper\">",
      '#suffix' => '</div>',
    ];

    $this->buildDynamicMappingInterface($form['field_mappings'], $form_state, $workflow);
  }

  /**
   * Build discovered fields display.
   */
  protected function buildDiscoveredFieldsDisplay(array &$form, FormStateInterface $form_state, $workflow) {
    $discovered_fields = $workflow->getDiscoveredFields();
    
    if (!empty($discovered_fields['inputs'])) {
      $form['discovered_fields']['inputs'] = [
        '#type' => 'details',
        '#title' => $this->t('Discovered Inputs (@count)', ['@count' => count($discovered_fields['inputs'])]),
        '#open' => FALSE,
      ];

      $header = ['Node ID', 'Input Name', 'Type', 'Default Value'];
      $rows = [];
      foreach ($discovered_fields['inputs'] as $input) {
        $rows[] = [
          $input['node_id'] ?? '',
          $input['input_name'] ?? '',
          $input['type'] ?? '',
          is_scalar($input['default_value'] ?? '') ? (string)($input['default_value'] ?? '') : json_encode($input['default_value'] ?? ''),
        ];
      }
      
      $form['discovered_fields']['inputs']['table'] = [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No inputs discovered.'),
      ];
    }

    if (!empty($discovered_fields['outputs'])) {
      $form['discovered_fields']['outputs'] = [
        '#type' => 'details',
        '#title' => $this->t('Discovered Outputs (@count)', ['@count' => count($discovered_fields['outputs'])]),
        '#open' => FALSE,
      ];

      $header = ['Node ID', 'Output Name', 'Type', 'Source'];
      $rows = [];
      foreach ($discovered_fields['outputs'] as $output) {
        $rows[] = [
          $output['node_id'] ?? '',
          $output['output_name'] ?? '',
          $output['type'] ?? '',
          $output['source'] ?? 'node',
        ];
      }
      
      $form['discovered_fields']['outputs']['table'] = [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No outputs discovered.'),
      ];
    }
  }

  /**
   * Build dynamic mapping interface - CLEANED UP.
   */
  protected function buildDynamicMappingInterface(array &$form, FormStateInterface $form_state, $workflow) {
    // Get field options
    $drupal_field_options = $this->getDrupalFieldOptions();
    $comfyui_options = $this->getComfyUIFieldOptions($workflow);
    
    if (empty($drupal_field_options)) {
      $form['no_drupal_fields'] = [
        '#markup' => '<div class="messages messages--warning">' . 
          $this->t('No custom fields found. Please <a href="@url">add fields</a> to the ComfyUI Workflow entity first.', [
            '@url' => '/admin/structure/comfyui-workflow/fields'
          ]) . 
          '</div>',
      ];
      return;
    }
    
    if (count($comfyui_options) <= 1) {
      $form['no_comfyui_fields'] = [
        '#markup' => '<div class="messages messages--warning">' . 
          $this->t('No ComfyUI fields discovered. Upload JSON files and click "Discover Fields" first.') . 
          '</div>',
      ];
      return;
    }

    // Get existing mappings (simplified - no config type)
    $existing_mappings = $workflow->getFieldMappings();
    
    // Get number of mapping rows from form state, or default to existing + 1 empty
    $num_mappings = $form_state->get("num_mappings");
    if ($num_mappings === NULL) {
      $num_mappings = count($existing_mappings) + 1; // Existing + 1 empty row
      $form_state->set("num_mappings", $num_mappings);
    }

    // Table for mappings
    $form['mappings_table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Drupal Field'),
        $this->t('ComfyUI Field'), 
        $this->t('Actions'),
      ],
      '#empty' => $this->t('No mappings configured.'),
      '#prefix' => "<div id=\"mappings-wrapper\">",
      '#suffix' => '</div>',
    ];

    // Build mapping rows
    $mapping_values = array_values($existing_mappings);
    for ($i = 0; $i < $num_mappings; $i++) {
      $existing_mapping = $mapping_values[$i] ?? NULL;
      
      // Drupal field dropdown
      $form['mappings_table'][$i]['drupal_field'] = [
        '#type' => 'select',
        '#options' => ['' => '- Select -'] + $drupal_field_options,
        '#default_value' => $existing_mapping ? array_search($existing_mapping, $existing_mappings) : '',
      ];
      
      // ComfyUI field dropdown  
      $comfyui_default = '';
      if ($existing_mapping) {
        $direction = $existing_mapping['direction'];
        $node = $existing_mapping['comfyui_node'];
        $field = $existing_mapping['comfyui_input'] ?? $existing_mapping['comfyui_output'];
        $comfyui_default = "{$direction}:{$node}:{$field}";
      }
      
      $form['mappings_table'][$i]['comfyui_field'] = [
        '#type' => 'select',
        '#options' => $comfyui_options,
        '#default_value' => $comfyui_default,
      ];
      
      // Remove button
      if ($num_mappings > 1) {
        $form['mappings_table'][$i]['remove'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove'),
          '#name' => "remove_{$i}",
          '#submit' => ['::removeMapping'],
          '#ajax' => [
            'callback' => '::mappingAjax',
            'wrapper' => "mappings-wrapper",
          ],
          '#limit_validation_errors' => [],
        ];
      }
    }

    // Add another mapping button
    $form['add_mapping'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Another Mapping'),
      '#name' => "add_mapping",
      '#submit' => ['::addMapping'],
      '#ajax' => [
        'callback' => '::mappingAjax',
        'wrapper' => "mappings-wrapper",
      ],
      '#limit_validation_errors' => [],
    ];
  }

  /**
   * Add mapping submit handler.
   */
  public function addMapping(array &$form, FormStateInterface $form_state) {
    $num_mappings = $form_state->get("num_mappings") + 1;
    $form_state->set("num_mappings", $num_mappings);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Remove mapping submit handler.
   */
  public function removeMapping(array &$form, FormStateInterface $form_state) {
    $num_mappings = $form_state->get("num_mappings");
    if ($num_mappings > 1) {
      $form_state->set("num_mappings", $num_mappings - 1);
    }
    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback for mapping operations.
   */
  public function mappingAjax(array &$form, FormStateInterface $form_state) {
    // Return the mappings wrapper
    return $form['field_mapping']['field_mappings'];
  }

  /**
   * Submit handler for node discovery.
   */
  public function discoverNodesSubmit(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\comfyui\Entity\ComfyUIWorkflow $workflow */
    $workflow = $this->getEntity();

    try {
      // Get services
      $node_mapping_service = \Drupal::service('comfyui.node_mapping');
      
      // Get workflow data
      $api_json = $workflow->getWorkflowApi();
      $workflow_json = $workflow->getWorkflowJson();
      
      if (!$api_json || !$workflow_json) {
        throw new \Exception('Both API and workflow JSON files are required for node discovery');
      }
      
      $api_data = json_decode($api_json, TRUE);
      $workflow_data = json_decode($workflow_json, TRUE);
      
      if (!$api_data || !$workflow_data) {
        throw new \Exception('Invalid JSON format in uploaded files');
      }
      
      // Clear cache to get fresh data
      $node_mapping_service->clearCache();
      
      // Discover nodes with recommendations
      $discovered_nodes = $node_mapping_service->discoverNodesWithRecommendations(
        $workflow_data,
        $api_data
      );
      
      // Store discovered nodes
      $workflow->setDiscoveredNodes($discovered_nodes);
      
      // Count statistics
      $total = count($discovered_nodes);
      $with_repos = count(array_filter($discovered_nodes, function($n) {
        return !empty($n['repositories']);
      }));
      $conflicts = count(array_filter($discovered_nodes, function($n) {
        return !empty($n['is_core_conflict']);
      }));
      $manual = $total - $with_repos;
      
      if ($total === 0) {
        $this->messenger()->addStatus($this->t(
          '✓ No custom nodes detected. This workflow only uses core ComfyUI nodes.'
        ));
      }
      else {
        $this->messenger()->addStatus($this->t(
          'Discovered @total custom node types: @repos with repository info, @manual require manual URLs' . 
          ($conflicts > 0 ? ', @conflicts core conflicts detected' : ''),
          [
            '@total' => $total,
            '@repos' => $with_repos,
            '@manual' => $manual,
            '@conflicts' => $conflicts,
          ]
        ));
      }

    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Node discovery failed: @error', [
        '@error' => $e->getMessage()
      ]);
      $this->messenger()->addError($this->t('Node discovery failed: @msg', [
        '@msg' => $e->getMessage()
      ]));
    }

    $form_state->setRebuild(TRUE);
  }

  /**
   * Submit handler for model discovery.
   */
  public function discoverModelsSubmit(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\comfyui\Entity\ComfyUIWorkflow $workflow */
    $workflow = $this->getEntity();

    try {
      // Get services
      $model_mapping_service = \Drupal::service('comfyui.model_mapping');
      
      // Get workflow data
      $api_json = $workflow->getWorkflowApi();
      
      if (!$api_json) {
        throw new \Exception('API JSON file is required for model discovery');
      }
      
      $api_data = json_decode($api_json, TRUE);
      
      if (!$api_data) {
        throw new \Exception('Invalid JSON format in API file');
      }
      
      // Clear cache to get fresh data
      $model_mapping_service->clearCache();
      
      // Discover models with download info
      $discovered_models = $model_mapping_service->discoverModelsWithDownloads($api_data);
      
      // Store discovered models
      $workflow->setDiscoveredModels($discovered_models);
      
      // Count statistics
      $total = count($discovered_models);
      $in_database = count(array_filter($discovered_models, function($m) {
        return $m['source'] === 'comfyui_manager';
      }));
      $manual = $total - $in_database;
      
      if ($total === 0) {
        $this->messenger()->addStatus($this->t(
          '✓ No model references detected in this workflow.'
        ));
      }
      else {
        $this->messenger()->addStatus($this->t(
          'Discovered @total model references: @db found in database, @manual require manual configuration',
          [
            '@total' => $total,
            '@db' => $in_database,
            '@manual' => $manual,
          ]
        ));
      }

    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Model discovery failed: @error', [
        '@error' => $e->getMessage()
      ]);
      $this->messenger()->addError($this->t('Model discovery failed: @msg', [
        '@msg' => $e->getMessage()
      ]));
    }

    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback for model discovery.
   */
  public function discoverModelsAjax(array &$form, FormStateInterface $form_state) {
    return $form['model_mapping']['model_mappings'];
  }

  /**
   * AJAX callback for node discovery.
   */
  public function discoverNodesAjax(array &$form, FormStateInterface $form_state) {
    return $form['node_mapping']['node_mappings'];
  }

  /**
   * Submit handler for field and node definition discovery.
   */
  public function discoverFieldsSubmit(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\comfyui\Entity\ComfyUIWorkflow $workflow */
    $workflow = $this->getEntity();

    try {
      // Get services
      $parser = \Drupal::service('comfyui.workflow_parser');
      $api_service = \Drupal::service('comfyui.api_options');
      
      // Get workflow data first to identify node types
      $api_json = $workflow->getWorkflowApi();
      $workflow_json = $workflow->getWorkflowJson();
      
      if (!$api_json || !$workflow_json) {
        throw new \Exception('Both API and workflow JSON files are required for discovery');
      }
      
      $api_data = json_decode($api_json, TRUE);
      $workflow_data = json_decode($workflow_json, TRUE);
      
      if (!$api_data || !$workflow_data) {
        throw new \Exception('Invalid JSON format in uploaded files');
      }
      
      // Extract node types that will be affected by cache clearing
      $node_types = $parser->extractUniqueNodeTypes($api_data);
      
      // Clear cache
      $api_service->clearCache();
      
      // Test API connection
      if (!$api_service->testApiConnection()) {
        \Drupal::logger('comfyui')->warning('DISCOVERY: ComfyUI API not accessible, falling back to basic discovery');
        
        // Fallback to basic discovery
        $discovered_inputs = $parser->discoverInputFields($api_data);
        $discovered_outputs = $parser->discoverOutputFields($workflow_data);
        
        $workflow->setDiscoveredFields([
          'inputs' => $discovered_inputs,
          'outputs' => $discovered_outputs,
          'node_definitions' => [],
          'api_available' => FALSE,
        ]);
        
        $this->messenger()->addWarning($this->t(
          'Basic field discovery completed. ComfyUI API not accessible - dynamic options unavailable. Discovered @in inputs and @out outputs.',
          ['@in' => count($discovered_inputs), '@out' => count($discovered_outputs)]
        ));
        
      } else {
        // Discovery with API integration
        $complete_discovery = $parser->discoverCompleteWorkflowDefinitions($api_data, $workflow_data, $api_service);
        
        // Store discovered fields
        $workflow->setDiscoveredFields([
          'inputs' => $complete_discovery['inputs'],
          'outputs' => $complete_discovery['outputs'],
          'node_definitions' => $complete_discovery['node_definitions'],
          'node_types' => $complete_discovery['node_types'],
          'api_available' => TRUE,
        ]);
        
        // Count fields with dynamic options
        $dynamic_count = 0;
        foreach ($complete_discovery['inputs'] as $input) {
          if (!empty($input['has_dynamic_options'])) {
            $dynamic_count++;
          }
        }
        
        $this->messenger()->addStatus($this->t(
          'Discovery completed! Discovered @in inputs (@dynamic with dynamic options) and @out outputs from @nodes node types. (Fresh data fetched after cache clear)',
          [
            '@in' => count($complete_discovery['inputs']),
            '@dynamic' => $dynamic_count,
            '@out' => count($complete_discovery['outputs']),
            '@nodes' => count($complete_discovery['node_types']),
          ]
        ));
      }

    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('DISCOVERY: Failed - @error', ['@error' => $e->getMessage()]);
      $this->messenger()->addError($this->t('Discovery failed: @msg', ['@msg' => $e->getMessage()]));
    }

    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback for field discovery.
   */
  public function discoverFieldsAjax(array &$form, FormStateInterface $form_state) {
    return $form['field_mapping']['field_mappings'];
  }

  /**
   * Get Drupal field options.
   */
  protected function getDrupalFieldOptions() {
    $options = [];
    $skip_fields = ['id', 'uuid', 'title', 'description', 'workflow_json', 'workflow_api', 'field_mappings', 'discovered_fields', 'status', 'created', 'changed'];

    // Get field definitions for the workflow entity
    $field_definitions = $this->entityFieldManager->getFieldDefinitions('comfyui_workflow', 'comfyui_workflow');

    foreach ($field_definitions as $field_name => $field_definition) {
      if (!in_array($field_name, $skip_fields) && $field_definition instanceof \Drupal\Core\Field\FieldConfigInterface) {
        $options[$field_name] = $field_definition->getLabel() . ' (' . $field_name . ')';
      }
    }

    return $options;
  }

  /**
   * Build node mappings table.
   */
  protected function buildNodeMappingsTable(array &$form, FormStateInterface $form_state, $workflow) {
    $discovered_nodes = $workflow->getDiscoveredNodes();
    $node_mappings = $workflow->getNodeMappings();

    if (empty($discovered_nodes)) {
      $form['empty'] = [
        '#markup' => '<div class="messages messages--warning">' .
          $this->t('No custom nodes discovered. Click "Discover Custom Nodes" to scan your workflow.') .
          '</div>',
      ];
      return;
    }

    $form['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Node Type'),
        $this->t('Repository'),
        $this->t('Status'),
      ],
      '#attributes' => ['class' => ['node-mapping-table']],
    ];

    foreach ($discovered_nodes as $node_type => $node_info) {
      $row_class = !empty($node_mappings[$node_type]['git_url']) ? 'node-mapping-configured' : 'node-mapping-pending';

      // Node type column
      $form['table'][$node_type]['node_type'] = [
        '#markup' => '<span class="node-type-cell">' . $node_type . '</span>' .
          $this->buildStatusBadge($node_info),
      ];

      // Repository selection column
      $form['table'][$node_type]['repository'] = $this->buildRepositoryField(
        $node_type,
        $node_info,
        $node_mappings[$node_type] ?? []
      );

      // Status column
      $form['table'][$node_type]['status'] = [
        '#markup' => $this->buildStatusMessage($node_info, $node_mappings[$node_type] ?? []),
      ];

      // Add row class
      $form['table'][$node_type]['#attributes']['class'][] = $row_class;
    }
  }

  /**
   * Build status badge for node.
   */
  protected function buildStatusBadge($node_info) {
    $action = $node_info['recommended_action'] ?? 'manual';
    
    $badges = [
      'single' => '<span class="node-status-badge custom">' . $this->t('Custom') . '</span>',
      'multiple' => '<span class="node-status-badge custom">' . $this->t('Multiple Options') . '</span>',
      'conflict' => '<span class="node-status-badge conflict">' . $this->t('Core Conflict') . '</span>',
      'manual' => '<span class="node-status-badge manual">' . $this->t('Unknown') . '</span>',
    ];

    return $badges[$action] ?? '';
  }

  /**
   * Build repository field for node.
   */
  protected function buildRepositoryField($node_type, $node_info, $current_mapping) {
    $repositories = $node_info['repositories'] ?? [];
    $action = $node_info['recommended_action'] ?? 'manual';

    $field = [
      '#type' => 'container',
    ];

    if ($action === 'conflict') {
      // Show option to use core or select custom repo
      $options = ['_core' => $this->t('Use Core Node (No Install)')];
      foreach ($repositories as $repo) {
        $options[$repo['url']] = $repo['title'];
      }
      $options['_custom'] = $this->t('Custom URL...');

      $field['select'] = [
        '#type' => 'select',
        '#options' => $options,
        '#default_value' => $current_mapping['git_url'] ?? '_core',
        '#attributes' => ['class' => ['node-repo-select']],
      ];
    }
    elseif (!empty($repositories)) {
      // Show dropdown of available repos
      $options = [];
      foreach ($repositories as $repo) {
        $options[$repo['url']] = $repo['title'];
      }
      
      if (count($repositories) > 1) {
        $options = ['' => $this->t('- Select Repository -')] + $options;
      }
      
      $options['_custom'] = $this->t('Custom URL...');

      $field['select'] = [
        '#type' => 'select',
        '#options' => $options,
        '#default_value' => $current_mapping['git_url'] ?? (count($repositories) === 1 ? array_key_first($options) : ''),
        '#attributes' => ['class' => ['node-repo-select']],
      ];
    }
    else {
      // No repos found, show manual entry only
      $field['info'] = [
        '#markup' => '<div class="node-repo-info">' . 
          $this->t('No repository found in ComfyUI Manager database.') . 
          '</div>',
      ];
    }

    // Always show custom URL field
    $field['custom_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Git URL'),
      '#default_value' => (!empty($current_mapping['git_url']) && !isset($field['select'])) ? 
        $current_mapping['git_url'] : '',
      '#placeholder' => 'https://github.com/author/repo',
      '#attributes' => ['class' => ['node-manual-url']],
      '#states' => [
        'visible' => [
          ':input[name="node_mapping[node_mappings][table][' . $node_type . '][repository][select]"]' => ['value' => '_custom'],
        ],
      ],
    ];

    return $field;
  }

  /**
   * Build status message for node.
   */
  protected function buildStatusMessage($node_info, $current_mapping) {
    if (!empty($current_mapping['git_url'])) {
      if ($current_mapping['git_url'] === '_core') {
        return '<span style="color: #28a745;">✓ Using Core</span>';
      }
      return '<span style="color: #28a745;">✓ Configured</span>';
    }

    $action = $node_info['recommended_action'] ?? 'manual';
    
    $messages = [
      'single' => '<span style="color: #0066cc;">→ Auto-recommended</span>',
      'multiple' => '<span style="color: #856404;">⚠ Choose repository</span>',
      'conflict' => '<span style="color: #856404;">⚠ Core conflict detected</span>',
      'manual' => '<span style="color: #dc3545;">✗ Manual URL required</span>',
    ];

    return $messages[$action] ?? '';
  }

  /**
   * Get ComfyUI field options from discovered fields with additional information.
   */
  protected function getComfyUIFieldOptions($workflow) {
    $options = ['_none' => $this->t('- None -')];
    $discovered_fields = $workflow->getDiscoveredFields();
    
    // Add input options with dynamic option indicators
    foreach ($discovered_fields['inputs'] ?? [] as $input) {
      $key = 'input:' . $input['node_id'] . ':' . $input['input_name'];
      
      $label = $this->t('Input: Node @node - @input (@type)', [
        '@node' => $input['node_id'],
        '@input' => $input['input_name'], 
        '@type' => $input['type'],
      ]);
      
      // Add indicator for fields with dynamic options
      if (!empty($input['has_dynamic_options'])) {
        $count = count($input['available_options']);
        $label .= $this->t(' [🔄 @count options]', ['@count' => $count]);
      }
      
      $options[$key] = $label;
    }
    
    // Add output options  
    foreach ($discovered_fields['outputs'] ?? [] as $output) {
      $key = 'output:' . $output['node_id'] . ':' . $output['output_name'];
      $options[$key] = $this->t('Output: Node @node - @output (@type)', [
        '@node' => $output['node_id'],
        '@output' => $output['output_name'],
        '@type' => $output['type'], 
      ]);
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  protected function copyFormValuesToEntity($entity, array $form, FormStateInterface $form_state) {
    // Only copy specific entity fields, ignore form UI elements
    $entity_fields = ['title', 'description', 'workflow_json', 'workflow_api', 'status'];
    
    foreach ($entity_fields as $field_name) {
      if ($form_state->hasValue($field_name)) {
        $value = $form_state->getValue($field_name);
        
        // Special handling for file fields - they come as complex arrays
        if (in_array($field_name, ['workflow_json', 'workflow_api'])) {
          $file_id = NULL;
          
          // Handle different file field formats
          if (is_array($value)) {
            if (isset($value['fids']) && is_array($value['fids']) && !empty($value['fids'][0])) {
              // Format: ['fids' => [0 => 127], 'upload' => '', ...]
              $file_id = $value['fids'][0];
            } elseif (isset($value[0]) && is_numeric($value[0])) {
              // Format: [0 => 127]
              $file_id = $value[0];
            } elseif (is_numeric($value)) {
              // Format: 127
              $file_id = $value;
            }
          } elseif (is_numeric($value)) {
            $file_id = $value;
          }
          
          if ($file_id) {
            $entity->set($field_name, $file_id);
          }
        } else {
          $entity->set($field_name, $value);
        }
      }
    }
    
    // Process all mapping configurations
    $this->processFieldMappings($entity, $form_state);
    $this->processBypassMappings($entity, $form_state);
  }

  /**
   * Process node mappings during main form save.
   */
  protected function processNodeMappings($entity, FormStateInterface $form_state) {
    $node_mappings_data = $form_state->getValue(['node_mapping', 'node_mappings', 'table']) ?? [];
    
    if (empty($node_mappings_data)) {
      return;
    }

    $node_mappings = [];
    $discovered_nodes = $entity->getDiscoveredNodes();

    foreach ($node_mappings_data as $node_type => $row) {
      // Skip if this isn't actually a node type row
      if (!isset($discovered_nodes[$node_type])) {
        continue;
      }

      $selected_repo = $row['repository']['select'] ?? '';
      $custom_url = $row['repository']['custom_url'] ?? '';

      $git_url = '';

      // Determine the git URL based on selection
      if ($selected_repo === '_core') {
        // User chose to use core node (no installation needed)
        $git_url = '_core';
      }
      elseif ($selected_repo === '_custom' && !empty($custom_url)) {
        // User provided custom URL
        $git_url = $custom_url;
      }
      elseif (!empty($selected_repo) && $selected_repo !== '_custom') {
        // User selected from dropdown
        $git_url = $selected_repo;
      }
      elseif (!empty($custom_url)) {
        // Custom URL provided without selecting _custom option
        $git_url = $custom_url;
      }

      // Only save if we have a URL
      if (!empty($git_url)) {
        $node_mappings[$node_type] = [
          'node_type' => $node_type,
          'git_url' => $git_url,
          'recommended_action' => $discovered_nodes[$node_type]['recommended_action'] ?? 'manual',
        ];
      }
    }

    // Save node mappings
    $entity->setNodeMappings($node_mappings);

    if (!empty($node_mappings)) {
      \Drupal::logger('comfyui')->info('Saved @count node mappings', [
        '@count' => count($node_mappings),
      ]);
    }
  }

  /**
   * Process model mappings during main form save.
   */
  protected function processModelMappings($entity, FormStateInterface $form_state) {
    $model_mappings_data = $form_state->getValue(['model_mapping', 'model_mappings', 'table']) ?? [];
    
    if (empty($model_mappings_data)) {
      return;
    }

    $model_mappings = [];
    $discovered_models = $entity->getDiscoveredModels();

    foreach ($model_mappings_data as $filename => $row) {
      // Skip if this isn't actually a model
      if (!isset($discovered_models[$filename])) {
        continue;
      }

      // Type might be a string (read-only) or from select
      $type = '';
      if (isset($row['type'])) {
        $type = is_array($row['type']) ? '' : $row['type'];
      }
      
      // If type is empty, get it from discovered models
      if (empty($type) && isset($discovered_models[$filename]['type'])) {
        $type = $discovered_models[$filename]['type'];
      }
      
      $save_path = $row['save_path'] ?? '';
      $url = $row['url'] ?? '';

      // Convert single URL to array format for consistency
      $urls = [];
      if (!empty($url)) {
        $urls = [$url];
      }

      // Only save if we have required info
      if (!empty($type) && !empty($save_path)) {
        $model_mappings[$filename] = [
          'filename' => $filename,
          'type' => $type,
          'save_path' => $save_path,
          'urls' => $urls,
          'source' => $discovered_models[$filename]['source'] ?? 'manual',
          'base' => $discovered_models[$filename]['base'] ?? '',
          'loader_type' => $discovered_models[$filename]['loader_type'] ?? '',
        ];
      }
    }

    // Save model mappings
    $entity->setModelMappings($model_mappings);

    if (!empty($model_mappings)) {
      \Drupal::logger('comfyui')->info('Saved @count model mappings', [
        '@count' => count($model_mappings),
      ]);
    }
  }

  /**
   * Process all mapping configurations during main form save.
   */
  protected function processFieldMappings($entity, FormStateInterface $form_state) {
    // Process node and model mappings first
    $this->processNodeMappings($entity, $form_state);
    $this->processModelMappings($entity, $form_state);

    // Get mappings data from the simplified form structure
    $mappings_data = $form_state->getValue(['field_mapping', 'field_mappings', 'mappings_table']) ?? [];
    
    $field_mappings = [];
    
    foreach ($mappings_data as $key => $row) {
      // Skip non-numeric keys (like 'add_mapping' button)
      if (!is_numeric($key)) {
        continue;
      }
      
      $drupal_field = $row['drupal_field'] ?? '';
      $comfyui_field = $row['comfyui_field'] ?? '';
      
      if (!empty($drupal_field) && !empty($comfyui_field) && $comfyui_field !== '_none') {
        // Parse ComfyUI field
        $parts = explode(':', $comfyui_field, 3);
        if (count($parts) === 3) {
          $direction = $parts[0];
          $node_id = $parts[1]; 
          $field_name = $parts[2];
          
          if ($direction === 'input') {
            $field_mappings[$drupal_field] = [
              'direction' => 'input',
              'comfyui_node' => $node_id,
              'comfyui_input' => $field_name,
            ];
          } else {
            $field_mappings[$drupal_field] = [
              'direction' => 'output', 
              'comfyui_node' => $node_id,
              'comfyui_output' => $field_name,
            ];
          }
        }
      }
    }
    
    // Save the simplified field mappings
    $entity->setFieldMappings($field_mappings);
    
    if (!empty($field_mappings)) {
      \Drupal::logger('comfyui')->info('Saved @count field mappings', [
        '@count' => count($field_mappings),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // No special button handling needed - let normal form processing handle everything
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\comfyui\Entity\ComfyUIWorkflow $workflow */
    $workflow = $this->entity;

    // Handle file permanence
    $workflow_json_fid = $form_state->getValue('workflow_json');
    if (!empty($workflow_json_fid[0])) {
      if ($file = \Drupal::entityTypeManager()->getStorage('file')->load($workflow_json_fid[0])) {
        $file->setPermanent();
        $file->save();
      }
    }

    $workflow_api_fid = $form_state->getValue('workflow_api');
    if (!empty($workflow_api_fid[0])) {
      if ($file = \Drupal::entityTypeManager()->getStorage('file')->load($workflow_api_fid[0])) {
        $file->setPermanent();
        $file->save();
      }
    }

    // Save the workflow
    $status = $workflow->save();

    switch ($status) {
      case SAVED_NEW:
        $this->messenger()->addMessage($this->t('Created the %label workflow.', [
          '%label' => $workflow->label(),
        ]));
        break;

      default:
        $this->messenger()->addMessage($this->t('Saved the %label workflow.', [
          '%label' => $workflow->label(),
        ]));
    }

    $form_state->setRedirectUrl($workflow->toUrl('collection'));
  }

  /**
   * Build bypass mapping section.
   */
  protected function buildBypassMappingSection(array &$form, FormStateInterface $form_state, $workflow) {
    // Group discovery section
    $form['discovery'] = [
      '#type' => 'details',
      '#title' => $this->t('Group Discovery'),
      '#open' => TRUE,
      '#description' => $this->t('Discover groups from your workflow.json file.'),
    ];

    $form['discovery']['discover_groups'] = [
      '#type' => 'submit',
      '#value' => $this->t('Discover Groups'),
      '#submit' => ['::discoverGroupsSubmit'],
      '#ajax' => [
        'callback' => '::discoverGroupsAjax',
        'wrapper' => 'bypass-mappings-wrapper',
      ],
      '#name' => 'discover_groups',
      '#limit_validation_errors' => [],
    ];

    // Show discovered groups summary
    $workflow_json = $workflow->getWorkflowJson();
    if ($workflow_json) {
      try {
        $groups = $this->bypassMappingService->parseGroupsFromWorkflow($workflow_json);
        
        if (!empty($groups)) {
          $form['discovery']['summary'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['discovered-groups-summary']],
            'content' => [
              '#markup' => '<h4>' . $this->t('Discovered Groups') . '</h4>' .
                '<div class="stat"><span class="stat-value">' . count($groups) . '</span> ' . $this->t('Groups') . '</div>',
            ],
          ];
        }
      } catch (\Exception $e) {
        $form['discovery']['error'] = [
          '#markup' => '<div class="messages messages--warning">' . 
            $this->t('Could not parse workflow JSON: @error', ['@error' => $e->getMessage()]) . 
            '</div>',
        ];
      }
    }

    // Bypass mappings section
    $form['mappings'] = [
      '#type' => 'details',
      '#title' => $this->t('Bypass Mappings'),
      '#open' => TRUE,
      '#prefix' => '<div id="bypass-mappings-wrapper">',
      '#suffix' => '</div>',
    ];

    $this->buildBypassMappingsTables($form['mappings'], $form_state, $workflow);
  }

  /**
   * Build bypass mapping tables (group and node bypasses).
   */
  protected function buildBypassMappingsTables(array &$form, FormStateInterface $form_state, $workflow) {
    $boolean_fields = $this->bypassMappingService->getAvailableBooleanFields($workflow);
    $bypass_mappings = $workflow->getBypassMappings();

    if (empty($boolean_fields)) {
      $form['no_fields'] = [
        '#markup' => '<div class="messages messages--warning">' . 
          $this->t('No Boolean fields found. Please <a href="@url">add Boolean fields</a> to the ComfyUI Workflow entity first.', [
            '@url' => '/admin/structure/comfyui-workflow/fields'
          ]) . 
          '</div>',
      ];
      return;
    }

    // Get available groups
    $workflow_json = $workflow->getWorkflowJson();
    $groups = [];
    if ($workflow_json) {
      try {
        $groups = $this->bypassMappingService->parseGroupsFromWorkflow($workflow_json);
      } catch (\Exception $e) {
        $this->messenger()->addWarning($this->t('Could not parse groups: @error', ['@error' => $e->getMessage()]));
      }
    }

    // GROUP BYPASS MAPPINGS TABLE
    $form['group_bypasses_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Group Bypass Mappings'),
      '#open' => TRUE,
      '#description' => $this->t('Map Boolean fields to groups. When a field is unchecked, all nodes in that group will be bypassed.'),
    ];

    $form['group_bypasses_section']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Boolean Field'),
        $this->t('Group Name'),
        $this->t('Actions'),
      ],
      '#attributes' => ['class' => ['bypass-mapping-table']],
    ];

    $group_bypasses = $bypass_mappings['group_bypasses'] ?? [];
    $num_group_rows = $form_state->get('num_group_rows');
    if ($num_group_rows === NULL) {
      $num_group_rows = count($group_bypasses) + 1;
      $form_state->set('num_group_rows', $num_group_rows);
    }

    $group_bypass_items = array_values($group_bypasses);
    for ($i = 0; $i < $num_group_rows; $i++) {
      $field_name = '';
      $group_name = '';

      if (isset($group_bypass_items[$i])) {
        // Reconstruct from stored mapping
        foreach ($group_bypasses as $field => $group) {
          if ($group === $group_bypass_items[$i]) {
            $field_name = $field;
            $group_name = $group;
            break;
          }
        }
      }

      // Boolean field dropdown
      $form['group_bypasses_section']['table'][$i]['field'] = [
        '#type' => 'select',
        '#options' => ['' => '- Select -'] + $boolean_fields,
        '#default_value' => $field_name,
      ];

      // Group name dropdown
      $form['group_bypasses_section']['table'][$i]['group'] = [
        '#type' => 'select',
        '#options' => ['' => '- Select -'] + array_combine(array_keys($groups), array_keys($groups)),
        '#default_value' => $group_name,
      ];

      // Remove button
      if ($num_group_rows > 1) {
        $form['group_bypasses_section']['table'][$i]['remove'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove'),
          '#name' => "remove_group_{$i}",
          '#submit' => ['::removeGroupBypass'],
          '#ajax' => [
            'callback' => '::bypassMappingAjax',
            'wrapper' => 'bypass-mappings-wrapper',
          ],
          '#limit_validation_errors' => [],
        ];
      }
    }

    // Add group bypass button
    $form['group_bypasses_section']['add'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Another Group Bypass'),
      '#name' => 'add_group_bypass',
      '#submit' => ['::addGroupBypass'],
      '#ajax' => [
        'callback' => '::bypassMappingAjax',
        'wrapper' => 'bypass-mappings-wrapper',
      ],
      '#limit_validation_errors' => [],
    ];

    // NODE BYPASS MAPPINGS TABLE
    $form['node_bypasses_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Node Bypass Mappings'),
      '#open' => TRUE,
      '#description' => $this->t('Map Boolean fields to specific nodes. Enter comma-separated node IDs. When a field is unchecked, those nodes will be bypassed.'),
    ];

    $form['node_bypasses_section']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Boolean Field'),
        $this->t('Node IDs (comma-separated)'),
        $this->t('Actions'),
      ],
      '#attributes' => ['class' => ['bypass-mapping-table']],
    ];

    $node_bypasses = $bypass_mappings['node_bypasses'] ?? [];
    $num_node_rows = $form_state->get('num_node_rows');
    if ($num_node_rows === NULL) {
      $num_node_rows = count($node_bypasses) + 1;
      $form_state->set('num_node_rows', $num_node_rows);
    }

    $node_bypass_items = array_values($node_bypasses);
    for ($i = 0; $i < $num_node_rows; $i++) {
      $field_name = '';
      $node_ids = '';

      if (isset($node_bypass_items[$i])) {
        foreach ($node_bypasses as $field => $ids) {
          if ($ids === $node_bypass_items[$i]) {
            $field_name = $field;
            $node_ids = $ids;
            break;
          }
        }
      }

      // Boolean field dropdown
      $form['node_bypasses_section']['table'][$i]['field'] = [
        '#type' => 'select',
        '#options' => ['' => '- Select -'] + $boolean_fields,
        '#default_value' => $field_name,
      ];

      // Node IDs text input
      $form['node_bypasses_section']['table'][$i]['nodes'] = [
        '#type' => 'textfield',
        '#placeholder' => '42, 43, 44',
        '#default_value' => $node_ids,
      ];

      // Remove button
      if ($num_node_rows > 1) {
        $form['node_bypasses_section']['table'][$i]['remove'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove'),
          '#name' => "remove_node_{$i}",
          '#submit' => ['::removeNodeBypass'],
          '#ajax' => [
            'callback' => '::bypassMappingAjax',
            'wrapper' => 'bypass-mappings-wrapper',
          ],
          '#limit_validation_errors' => [],
        ];
      }
    }

    // Add node bypass button
    $form['node_bypasses_section']['add'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Another Node Bypass'),
      '#name' => 'add_node_bypass',
      '#submit' => ['::addNodeBypass'],
      '#ajax' => [
        'callback' => '::bypassMappingAjax',
        'wrapper' => 'bypass-mappings-wrapper',
      ],
      '#limit_validation_errors' => [],
    ];
  }

  /**
   * Submit handler for group discovery.
   */
  public function discoverGroupsSubmit(array &$form, FormStateInterface $form_state) {
    $workflow = $this->entity;

    try {
      $workflow_json = $workflow->getWorkflowJson();
      if (!$workflow_json) {
        throw new \Exception('No workflow JSON file uploaded');
      }

      $workflow_data = json_decode($workflow_json, TRUE);
      $groups = $this->bypassMappingService->parseGroupsFromWorkflow($workflow_json);

      if (empty($groups)) {
        $this->messenger()->addStatus($this->t('No groups found in this workflow.'));
      } else {
        // Pre-compute node IDs for each group
        $group_details = [];
        foreach ($groups as $group_name => $group_info) {
          $node_ids = $this->bypassMappingService->getNodesInGroup($workflow_data, $group_name);
          $group_details[$group_name] = $node_ids;
        }

        $this->messenger()->addStatus($this->t(
          'Discovered @count groups with nodes: @details',
          [
            '@count' => count($groups),
            '@details' => implode(', ', array_map(function($name, $ids) {
              return sprintf('%s (%d nodes)', $name, count($ids));
            }, array_keys($group_details), array_values($group_details))),
          ]
        ));
      }
    } catch (\Exception $e) {
      $this->messenger()->addError($this->t('Group discovery failed: @error', ['@error' => $e->getMessage()]));
    }

    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback for group discovery.
   */
  public function discoverGroupsAjax(array &$form, FormStateInterface $form_state) {
    return $form['bypass_mapping']['mappings'];
  }

  /**
   * Add group bypass row.
   */
  public function addGroupBypass(array &$form, FormStateInterface $form_state) {
    $num_rows = $form_state->get('num_group_rows') + 1;
    $form_state->set('num_group_rows', $num_rows);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Remove group bypass row.
   */
  public function removeGroupBypass(array &$form, FormStateInterface $form_state) {
    $num_rows = $form_state->get('num_group_rows');
    if ($num_rows > 1) {
      $form_state->set('num_group_rows', $num_rows - 1);
    }
    $form_state->setRebuild(TRUE);
  }

  /**
   * Add node bypass row.
   */
  public function addNodeBypass(array &$form, FormStateInterface $form_state) {
    $num_rows = $form_state->get('num_node_rows') + 1;
    $form_state->set('num_node_rows', $num_rows);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Remove node bypass row.
   */
  public function removeNodeBypass(array &$form, FormStateInterface $form_state) {
    $num_rows = $form_state->get('num_node_rows');
    if ($num_rows > 1) {
      $form_state->set('num_node_rows', $num_rows - 1);
    }
    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback for bypass mapping changes.
   */
  public function bypassMappingAjax(array &$form, FormStateInterface $form_state) {
    return $form['bypass_mapping']['mappings'];
  }

  /**
   * Process bypass mappings during form save.
   */
  protected function processBypassMappings($entity, FormStateInterface $form_state) {
    // Get workflow data for node ID pre-computation
    $workflow_json = $entity->getWorkflowJson();
    $workflow_data = $workflow_json ? json_decode($workflow_json, TRUE) : [];

    // Process group bypasses
    $group_bypass_data = $form_state->getValue(['bypass_mapping', 'mappings', 'group_bypasses_section', 'table']) ?? [];
    $group_bypasses = [];

    foreach ($group_bypass_data as $key => $row) {
      if (!is_numeric($key)) {
        continue;
      }

      $field_name = $row['field'] ?? '';
      $group_name = $row['group'] ?? '';

      if (!empty($field_name) && !empty($group_name)) {
        // Pre-compute node IDs for this group
        $node_ids = [];
        if (!empty($workflow_data)) {
          $node_ids = $this->bypassMappingService->getNodesInGroup($workflow_data, $group_name);
        }

        $group_bypasses[$field_name] = [
          'group_name' => $group_name,
          'node_ids' => $node_ids,
        ];
      }
    }

    // Process node bypasses
    $node_bypass_data = $form_state->getValue(['bypass_mapping', 'mappings', 'node_bypasses_section', 'table']) ?? [];
    $node_bypasses = [];

    foreach ($node_bypass_data as $key => $row) {
      if (!is_numeric($key)) {
        continue;
      }

      $field_name = $row['field'] ?? '';
      $node_ids = $row['nodes'] ?? '';

      if (!empty($field_name) && !empty($node_ids)) {
        $node_ids_array = array_filter(array_map('trim', explode(',', $node_ids)));
        if (!empty($node_ids_array)) {
          $node_bypasses[$field_name] = $node_ids_array;
        }
      }
    }

    // Save bypass mappings
    $bypass_mappings = [
      'group_bypasses' => $group_bypasses,
      'node_bypasses' => $node_bypasses,
    ];

    $entity->setBypassMappings($bypass_mappings);

    if (!empty($group_bypasses) || !empty($node_bypasses)) {
      $group_count = count($group_bypasses);
      $node_count = count($node_bypasses);
      $total_nodes = array_reduce($group_bypasses, function($carry, $item) {
        return $carry + count($item['node_ids']);
      }, 0);

      \Drupal::logger('comfyui')->info(
        'Saved bypass mappings: @groups group bypasses (@nodes nodes), @node_bypasses node bypasses',
        [
          '@groups' => $group_count,
          '@nodes' => $total_nodes,
          '@node_bypasses' => $node_count,
        ]
      );
    }
  }

}
