<?php

namespace Drupal\comfyui\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Render\Element;

/**
 * Provides a dynamic ComfyUI field block.
 *
 * @Block(
 *   id = "comfyui_dynamic_field",
 *   admin_label = @Translation("ComfyUI Dynamic Field"),
 *   category = @Translation("ComfyUI Integration")
 * )
 */
class ComfyUIDynamicFieldBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   */
  protected $entityFieldManager;

  /**
   * The entity display repository.
   */
  protected $entityDisplayRepository;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, $entity_display_repository) {
      parent::__construct($configuration, $plugin_id, $plugin_definition);
      $this->entityTypeManager = $entity_type_manager;
      $this->entityFieldManager = $entity_field_manager;
      $this->entityDisplayRepository = $entity_display_repository;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
      return new static(
        $configuration,
        $plugin_id,
        $plugin_definition,
        $container->get('entity_type.manager'),
        $container->get('entity_field.manager'),
        $container->get('entity_display.repository')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    // Get workflow from layout builder context or route
    $workflow = $this->getWorkflowFromContext();
    $is_in_workflow_context = !empty($workflow);

    if ($is_in_workflow_context) {
      // In workflow context - just store the workflow ID silently
      $form['workflow_id'] = [
        '#type' => 'value',
        '#value' => $workflow->id(),
      ];
    } else {
      // NOT in workflow context - show dropdown
      $workflow_options = $this->getWorkflowOptions();
      
      $form['workflow_id'] = [
        '#type' => 'select',
        '#title' => $this->t('Workflow'),
        '#options' => $workflow_options,
        '#default_value' => $this->configuration['workflow_id'] ?? '',
        '#description' => $this->t('Select the workflow for this block.'),
        '#empty_option' => $this->t('- Select a workflow -'),
        '#required' => TRUE,
      ];
    }

    // View mode selection (only 'form' supports blocks)
    $form['view_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Display Mode'),
      '#options' => [
        'form' => $this->t('Form Mode (Interactive)'),
      ],
      '#default_value' => $this->configuration['view_mode'] ?? 'form',
      '#description' => $this->t('Dynamic blocks only work in Form mode. Other modes (API, Headless, Batch) do not use blocks.'),
    ];

    $field_options = $this->getDrupalFieldOptions();

    $form['selected_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Field to Display'),
      '#description' => $this->t('Select which field this block should render.'),
      '#options' => $field_options,
      '#default_value' => $this->configuration['selected_field'] ?? '',
      '#empty_option' => $this->t('- Select a field -'),
      '#required' => TRUE,
    ];

    $form['field_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Label'),
      '#description' => $this->t('Override the default field label (optional).'),
      '#default_value' => $this->configuration['field_label'] ?? '',
    ];

    $form['field_required'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Required Field'),
      '#description' => $this->t('Mark this field as required for form submission.'),
      '#default_value' => $this->configuration['field_required'] ?? FALSE,
    ];

    $form['display_as'] = [
      '#type' => 'select',
      '#title' => $this->t('Display As'),
      '#options' => [
        'auto' => $this->t('Auto (based on field mapping)'),
        'input' => $this->t('Force Input'),
        'output' => $this->t('Force Output'),
      ],
      '#default_value' => $this->configuration['display_as'] ?? 'auto',
      '#description' => $this->t('Choose how this block behaves. Auto uses field mapping direction.'),
    ];

    $form['output_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Output Format'),
      '#options' => [
        'text' => $this->t('Text Only'),
        'image' => $this->t('Image (for image URLs)'),
        'raw' => $this->t('Raw Data'),
      ],
      '#default_value' => $this->configuration['output_format'] ?? 'text',
      '#description' => $this->t('How to format output data when displaying results.'),
      '#states' => [
        'visible' => [
          [':input[name="settings[display_as]"]' => ['value' => 'output']],
          [':input[name="settings[display_as]"]' => ['value' => 'auto']],
        ],
      ],
    ];

    return $form;
  }

  /**
   * Get workflow from layout builder context or standard route.
   */
  protected function getWorkflowFromContext() {
    $route_match = \Drupal::routeMatch();
    $route_name = $route_match->getRouteName();
    
    // Method 1: Standard route parameter (for non-layout-builder pages)
    $workflow = $route_match->getParameter('comfyui_workflow');
    if ($workflow) {
      return $workflow;
    }
    
    // Method 2: Layout Builder - extract from URL or destination
    if (strpos($route_name, 'layout_builder') !== FALSE) {
      $request = \Drupal::request();
      
      // Try to get workflow ID from destination parameter
      $destination = $request->query->get('destination');
      if ($destination && preg_match('/\/admin\/structure\/comfyui-workflow\/(\d+)\/layout/', $destination, $matches)) {
        $workflow_id = $matches[1];
        return $this->entityTypeManager->getStorage('comfyui_workflow')->load($workflow_id);
      }
      
      // Try to extract from the current URL path
      $current_path = $request->getPathInfo();
      if (preg_match('/overrides\/comfyui_workflow\.(\d+)/', $current_path, $matches)) {
        $workflow_id = $matches[1];
        return $this->entityTypeManager->getStorage('comfyui_workflow')->load($workflow_id);
      }
    }
    
    return null;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $this->configuration['workflow_id'] = $form_state->getValue('workflow_id');
    $this->configuration['view_mode'] = $form_state->getValue('view_mode');
    $this->configuration['selected_field'] = $form_state->getValue('selected_field');
    $this->configuration['field_label'] = $form_state->getValue('field_label');
    $this->configuration['field_required'] = $form_state->getValue('field_required');
    $this->configuration['display_as'] = $form_state->getValue('display_as');
    $this->configuration['output_format'] = $form_state->getValue('output_format');
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $selected_field = $this->configuration['selected_field'] ?? '';
    $view_mode = $this->configuration['view_mode'] ?? 'form';

    if (empty($selected_field)) {
      return [
        '#markup' => $this->t('No field selected. Configure this block to select a field.'),
      ];
    }

    $workflow = $this->getWorkflow();
    
    if (!$workflow) {
      return [
        '#markup' => $this->t('No workflow context available.'),
      ];
    }

    $field_mappings = $workflow->getFieldMappings();
    $mapping = $field_mappings[$selected_field] ?? null;
    $field_definition = $this->getFieldDefinition($selected_field);

    if (!$mapping) {
      if ($field_definition && $field_definition->getType() === 'boolean') {
        $bypass_mappings = $workflow->getBypassMappings();
        
        $is_bypass_field = FALSE;
        if (!empty($bypass_mappings['group_bypasses']) && isset($bypass_mappings['group_bypasses'][$selected_field])) {
          $is_bypass_field = TRUE;
        }
        if (!empty($bypass_mappings['node_bypasses']) && isset($bypass_mappings['node_bypasses'][$selected_field])) {
          $is_bypass_field = TRUE;
        }
        
        if ($is_bypass_field) {
          return $this->buildBypassControlField($selected_field, $field_definition, $workflow, $bypass_mappings);
        }
      }
      
      return [
        '#markup' => $this->t('Field "@field" is not mapped for this workflow.', ['@field' => $selected_field]),
      ];
    }

        $field_label = $this->configuration['field_label'] ?: ($field_definition ? (string) $field_definition->getLabel() : $selected_field);

    $display_as = $this->configuration['display_as'] ?? 'auto';
    $should_show_as_output = false;

    if ($display_as === 'output') {
      $should_show_as_output = true;
    } elseif ($display_as === 'input') {
      $should_show_as_output = false;
    } else {
      $should_show_as_output = ($mapping['direction'] === 'output');
    }

    if ($should_show_as_output) {
      return $this->buildOutputFieldWithFormat($selected_field, $field_label, $field_definition, $mapping);
    } else {
      return $this->buildInputField($selected_field, $field_label, $field_definition, $mapping);
    }
  }

  /**
   * Build an input field element with dynamic options support.
   */
  protected function buildInputField($field_name, $label, $field_definition, $mapping) {
    // Check if we're in layout builder mode
    $route_name = \Drupal::routeMatch()->getRouteName();
    $is_layout_builder = strpos($route_name, 'layout_builder') !== FALSE;
    
    if ($is_layout_builder) {
      return $this->buildLayoutBuilderPreview($field_name, $label, $field_definition, $mapping);
    }
    
    // Get the workflow entity
    $workflow = $this->getWorkflow();
    
    if (!$workflow || !$field_definition) {
      return $this->buildFallbackInputField($field_name, $label, $field_definition, $mapping);
    }

    // Check for dynamic options
    $dynamic_options = $this->getDynamicOptionsForField($workflow, $mapping);
    
    if (!empty($dynamic_options)) {
      return $this->buildDynamicSelectField($field_name, $label, $dynamic_options, $mapping);
    }

    // Check if this field uses Random Seed Widget by checking form display
    $is_seed_widget = $this->isRandomSeedWidget($field_definition);
    
    if ($is_seed_widget) {
      return $this->buildSeedField($field_name, $label, $workflow, $mapping);
    }

    // Build regular text input
    return $this->buildRegularInputField($field_name, $label, $field_definition, $workflow, $mapping);
  }

  /**
   * Get dynamic options for a specific field based on mapping.
   */
  protected function getDynamicOptionsForField($workflow, $mapping) {
    $discovered_fields = $workflow->getDiscoveredFields();
    
    if (empty($discovered_fields['inputs']) || $mapping['direction'] !== 'input') {
      return [];
    }
    
    $node_id = $mapping['comfyui_node'];
    $input_name = $mapping['comfyui_input'];
    
    // Find matching discovered field
    foreach ($discovered_fields['inputs'] as $input) {
      if ($input['node_id'] == $node_id && $input['input_name'] == $input_name) {
        return $input['available_options'] ?? [];
      }
    }
    
    return [];
  }

  /**
   * Build a dynamic select field with ComfyUI API options.
   */
  protected function buildDynamicSelectField($field_name, $label, $options, $mapping) {
    $workflow = $this->getWorkflow();
    
    // Prepare options for select
    $select_options = [];
    foreach ($options as $option) {
      $select_options[$option] = $option;
    }
    
    // Get current value
    $current_value = '';
    if ($workflow->hasField($field_name)) {
      $field_items = $workflow->get($field_name);
      if (!$field_items->isEmpty()) {
        $current_value = $field_items->first()->value;
      }
    }
    
    $element = [
      '#type' => 'select',
      '#title' => $label,
      '#options' => $select_options,
      '#default_value' => $current_value,
      '#required' => $this->configuration['field_required'] ?? FALSE,
      '#empty_option' => $this->t('- Select -'),
      '#attributes' => [
        'data-comfyui-field' => $field_name,
        'data-comfyui-mapping' => json_encode($mapping),
        'form' => 'comfyui-workflow-form',
        'class' => ['comfyui-dynamic-select'],
      ],
      '#wrapper_attributes' => [
        'class' => ['comfyui-field-block', 'comfyui-input-field', 'comfyui-dynamic-field'],
      ],
      '#attached' => [
        'library' => ['comfyui/workflow_form'],
      ],
      '#description' => $this->t('Options loaded from ComfyUI API (@count available)', [
        '@count' => count($options),
      ]),
    ];

    return $element;
  }

  /**
   * Build seed field (existing functionality).
   */
  protected function buildSeedField($field_name, $label, $workflow, $mapping) {
    // Get current value
    $current_value = '';
    if ($workflow->hasField($field_name)) {
      $field_items = $workflow->get($field_name);
      if (!$field_items->isEmpty()) {
        $current_value = $field_items->first()->value;
      }
    }
    
    if (empty($current_value)) {
      $current_value = mt_rand(100000, 999999);
    }

    $element = [
      '#type' => 'number',
      '#title' => $label,
      '#default_value' => $current_value,
      '#attributes' => [
        'data-comfyui-field' => $field_name,
        'data-comfyui-mapping' => json_encode($mapping),
        'form' => 'comfyui-workflow-form',
        'class' => ['comfyui-seed-field'],
      ],
      '#wrapper_attributes' => [
        'class' => ['comfyui-field-block', 'comfyui-input-field'],
      ],
      '#attached' => [
        'library' => ['comfyui/workflow_form', 'comfyui/seed_manager'],
      ],
      '#description' => $this->t('Seed value for reproducible generation. Use the controls below to manage seed behavior.'),
      '#min' => 1,
      '#max' => 999999,
      '#required' => $this->configuration['field_required'] ?? FALSE,
      '#size' => 8,
    ];

    return $element;
  }

  /**
   * Build regular input field (existing functionality enhanced).
   */
  protected function buildRegularInputField($field_name, $label, $field_definition, $workflow, $mapping) {
    $field_type = $field_definition->getType();
    
    // Special handling for media reference fields
    if ($field_type === 'entity_reference') {
      $target_type = $field_definition->getSetting('target_type');
      
      if ($target_type === 'media') {
        return $this->buildMediaSelectorField($field_name, $label, $field_definition, $workflow, $mapping);
      }
    }

    // Get current value
    $current_value = '';
    if ($workflow->hasField($field_name)) {
      $field_items = $workflow->get($field_name);
      if (!$field_items->isEmpty()) {
        $current_value = $field_items->first()->value;
      }
    }

    $element = [
      '#type' => $this->getInputTypeForFieldType($field_type),
      '#title' => $label,
      '#default_value' => $current_value,
      '#required' => $this->configuration['field_required'] ?? FALSE,
      '#attributes' => [
        'data-comfyui-field' => $field_name,
        'data-comfyui-mapping' => json_encode($mapping),
        'form' => 'comfyui-workflow-form',
      ],
      '#wrapper_attributes' => [
        'class' => ['comfyui-field-block', 'comfyui-input-field'],
      ],
      '#attached' => [
        'library' => ['comfyui/workflow_form'],
      ],
    ];

    return $element;
  }

  /**
   * Check if field uses Random Seed Widget.
   */
  protected function isRandomSeedWidget($field_definition) {
    try {
      $form_display = $this->entityDisplayRepository->getFormDisplay('comfyui_workflow', 'comfyui_workflow', 'default');
      $component = $form_display->getComponent($field_definition->getName());
      return isset($component['type']) && $component['type'] === 'random_seed';
    } catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * Check if a field uses the Random Seed Widget.
   */
  protected function fieldUsesSeedWidget($field_definition) {
    if (!$field_definition instanceof \Drupal\Core\Field\FieldConfigInterface) {
      return FALSE;
    }
    
    try {
      // Get the form display configuration
      $form_display = $this->entityDisplayRepository->getFormDisplay('comfyui_workflow', 'comfyui_workflow', 'default');
      $component = $form_display->getComponent($field_definition->getName());
      
      return isset($component['type']) && $component['type'] === 'random_seed';
    } catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * Build preview for layout builder mode.
   */
  protected function buildLayoutBuilderPreview($field_name, $label, $field_definition, $mapping) {
    $field_type = $field_definition ? $field_definition->getType() : 'string';
    $direction = $mapping['direction'] ?? 'input';
    
    return [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['comfyui-field-preview', 'layout-builder-preview'],
        'style' => 'border: 2px dashed #ccc; padding: 10px; margin: 5px 0;',
      ],
      'preview' => [
        '#markup' => sprintf(
          '<div style="color: #666; font-style: italic;">
            <strong>%s</strong><br>
            <small>Field: %s (%s)</small><br>
            <small>Type: %s | Direction: %s</small><br>
            <small>ComfyUI Node: %s</small>
          </div>',
          htmlspecialchars($label),
          htmlspecialchars($field_name),
          htmlspecialchars($field_type),
          htmlspecialchars($field_type),
          htmlspecialchars($direction),
          htmlspecialchars($mapping['comfyui_node'] ?? 'N/A')
        ),
      ],
    ];
  }

  /**
   * Add ComfyUI attributes to field element and its children.
   */
  protected function addComfyUIAttributes(&$element, $field_name, $mapping) {
    // Add to the main element
    if (!isset($element['#attributes'])) {
      $element['#attributes'] = [];
    }
    
    $element['#attributes']['data-comfyui-field'] = $field_name;
    $element['#attributes']['data-comfyui-mapping'] = json_encode($mapping);
    $element['#attributes']['form'] = 'comfyui-workflow-form';
    
    // Add wrapper classes
    if (!isset($element['#wrapper_attributes'])) {
      $element['#wrapper_attributes'] = [];
    }
    if (!isset($element['#wrapper_attributes']['class'])) {
      $element['#wrapper_attributes']['class'] = [];
    }
    $element['#wrapper_attributes']['class'][] = 'comfyui-field-block';
    $element['#wrapper_attributes']['class'][] = 'comfyui-input-field';

    // For widget elements, also add to the first child (delta 0)
    if (isset($element['widget'][0])) {
      if (!isset($element['widget'][0]['#attributes'])) {
        $element['widget'][0]['#attributes'] = [];
      }
      $element['widget'][0]['#attributes']['data-comfyui-field'] = $field_name;
      $element['widget'][0]['#attributes']['data-comfyui-mapping'] = json_encode($mapping);
    }
    
    // Also check for direct field structure (no widget wrapper)
    if (isset($element[0])) {
      if (!isset($element[0]['#attributes'])) {
        $element[0]['#attributes'] = [];
      }
      $element[0]['#attributes']['data-comfyui-field'] = $field_name;
      $element[0]['#attributes']['data-comfyui-mapping'] = json_encode($mapping);
    }
  }

  /**
   * Add ComfyUI attributes to child field elements.
   */
  protected function addComfyUIAttributesToChild(&$element, $field_name, $mapping) {
      if (!isset($element['#attributes'])) {
        $element['#attributes'] = [];
      }
      $element['#attributes']['data-comfyui-field'] = $field_name;
      $element['#attributes']['data-comfyui-mapping'] = json_encode($mapping);
  }

  /**
   * Fallback input field builder when widget rendering fails.
   */
  protected function buildFallbackInputField($field_name, $label, $field_definition, $mapping = []) {
      $field_type = $field_definition ? $field_definition->getType() : 'string';

      $element = [
        '#type' => $this->getInputTypeForFieldType($field_type),
        '#title' => $label,
        '#attributes' => [
          'data-comfyui-field' => $field_name,
          'form' => 'comfyui-workflow-form',
        ],
        '#wrapper_attributes' => [
          'class' => ['comfyui-field-block', 'comfyui-input-field'],
        ],
        '#attached' => [
          'library' => ['comfyui/workflow_form'],
        ],
      ];

      // Add mapping data if available
      if (!empty($mapping)) {
        $element['#attributes']['data-comfyui-mapping'] = json_encode($mapping);
      }

      // Add field-specific attributes for fallback
      if ($field_definition) {
        $default_value = $field_definition->getDefaultValueLiteral();
        if (!empty($default_value)) {
          $element['#default_value'] = $default_value[0]['value'] ?? '';
        }
      }

      return $element;
  }

  /**
   * Build output field with integrated formatting (no entity display dependency).
   */
  protected function buildOutputFieldWithFormat($field_name, $label, $field_definition, $mapping) {
    // Check if we're in layout builder mode
    $route_name = \Drupal::routeMatch()->getRouteName();
    $is_layout_builder = strpos($route_name, 'layout_builder') !== FALSE;
    
    if ($is_layout_builder) {
      return $this->buildLayoutBuilderPreview($field_name, $label, $field_definition, $mapping);
    }
    
    // Get the workflow entity (not the form)
    $workflow = $this->getWorkflow();
    
    if (!$workflow) {
      return [
        '#markup' => '<div class="comfyui-field-error">No workflow context available.</div>',
      ];
    }

    $output_format = $this->configuration['output_format'] ?? 'text';

    // Create container for output
    $container = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['comfyui-field-block', 'comfyui-output-field'],
        'data-comfyui-output' => $field_name,
        'data-comfyui-mapping' => json_encode($mapping),
        'data-output-format' => $output_format,
      ],
      'label' => [
        '#markup' => '<h3>' . $label . '</h3>',
      ],
      '#attached' => [
        'library' => ['comfyui/workflow_form'],
      ],
    ];

    // Check if workflow has this field and if it has data
    if ($workflow->hasField($field_name)) {
      $field_items = $workflow->get($field_name);
      
      if (!$field_items->isEmpty()) {
        $value = $field_items->first()->value;
        
        // Format based on output format setting
        switch ($output_format) {
          case 'image':
            if ($this->isImageUrl($value)) {
              $container['content'] = [
                '#theme' => 'image',
                '#uri' => $value,
                '#alt' => 'Generated Image',
                '#attributes' => [
                  'class' => ['comfyui-output-image'],
                  'style' => 'max-width: 400px; height: auto;',
                ],
              ];
            } else {
              $container['content'] = [
                '#markup' => '<div class="comfyui-no-image">No valid image URL: ' . htmlspecialchars($value) . '</div>',
              ];
            }
            break;
            
          case 'raw':
            $container['content'] = [
              '#markup' => '<pre class="comfyui-raw-output">' . htmlspecialchars($value) . '</pre>',
            ];
            break;
            
          case 'text':
          default:
            $container['content'] = [
              '#markup' => '<div class="comfyui-text-output">' . htmlspecialchars($value) . '</div>',
            ];
            break;
        }
      } else {
        // Empty field - show placeholder for results
        $container['content'] = [
          '#type' => 'container',
          '#attributes' => [
            'class' => ['comfyui-output-content'],
            'id' => 'comfyui-output-' . $field_name,
          ],
          '#markup' => '<div class="comfyui-output-placeholder">Results will appear here after workflow execution.</div>',
        ];
      }
    } else {
      // Field doesn't exist on the workflow entity
      $container['content'] = [
        '#markup' => '<div class="comfyui-field-error">Field "' . $field_name . '" not found on workflow entity. Make sure the field exists and is properly configured.</div>',
      ];
    }

    return $container;
  }

  /**
   * Check if a URL looks like an image.
   */
  protected function isImageUrl($url) {
    if (strpos($url, '/comfyui/image-proxy/') !== FALSE) {
      return TRUE;
    }
    
    $image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'];
    $extension = strtolower(pathinfo($url, PATHINFO_EXTENSION));
    
    return in_array($extension, $image_extensions);
  }

  /**
   * Add output-specific attributes to rendered field.
   */
  protected function addOutputAttributes(&$element, $field_name) {
      if (!isset($element['#attributes'])) {
        $element['#attributes'] = [];
      }
      if (!isset($element['#attributes']['class'])) {
        $element['#attributes']['class'] = [];
      }
      
      $element['#attributes']['class'][] = 'comfyui-output-content';
      $element['#attributes']['id'] = 'comfyui-output-' . $field_name;
      
      // For field items, add attributes to each item
      foreach (element_children($element) as $key) {
        if (is_numeric($key) && isset($element[$key])) {
          if (!isset($element[$key]['#attributes'])) {
            $element[$key]['#attributes'] = [];
          }
          if (!isset($element[$key]['#attributes']['class'])) {
            $element[$key]['#attributes']['class'] = [];
          }
          $element[$key]['#attributes']['class'][] = 'comfyui-output-item';
        }
      }
  }

  /**
   * Fallback output field builder when no display config available.
   */
  protected function buildFallbackOutputField($field_name, $label) {
      return [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['comfyui-field-block', 'comfyui-output-field'],
          'data-comfyui-output' => $field_name,
        ],
        'label' => [
          '#markup' => '<h3>' . $label . '</h3>',
        ],
        'content' => [
          '#type' => 'container',
          '#attributes' => [
            'class' => ['comfyui-output-content'],
            'id' => 'comfyui-output-' . $field_name,
          ],
          '#markup' => '<div class="comfyui-output-placeholder">Results will appear here after workflow execution.</div>',
        ],
        '#attached' => [
          'library' => ['comfyui/workflow_form'],
        ],
      ];
  }

  /**
   * Map Drupal field types to form input types.
   */
  protected function getInputTypeForFieldType($field_type) {
      $type_map = [
        'string' => 'textfield',
        'string_long' => 'textarea',  // FIXED: was 'textfield'
        'text' => 'textarea',
        'text_long' => 'textarea',
        'text_with_summary' => 'textarea',
        'integer' => 'number',
        'decimal' => 'number',
        'float' => 'number',
        'boolean' => 'checkbox',
        'email' => 'email',
        'file' => 'file',
        'image' => 'file',
      ];

      return $type_map[$field_type] ?? 'textfield';
  }

  /**
   * Get available Drupal field options.
   */
  protected function getDrupalFieldOptions() {
    $options = [];
    $skip_fields = ['id', 'uuid', 'title', 'description', 'workflow_json', 'workflow_api', 'field_mappings', 'bypass_mappings', 'discovered_fields', 'status', 'created', 'changed'];

    $field_definitions = $this->entityFieldManager->getFieldDefinitions('comfyui_workflow', 'comfyui_workflow');

    foreach ($field_definitions as $field_name => $field_definition) {
      if (!in_array($field_name, $skip_fields) && $field_definition instanceof \Drupal\Core\Field\FieldConfigInterface) {
        $label = $field_definition->getLabel();
        $options[$field_name] = (string) $label . ' (' . $field_name . ')';
      }
    }

    return $options;
  }

  /**
   * Get workflow from context or configuration.
   */
  protected function getWorkflow() {
    // Use the enhanced context detection
    $workflow = $this->getWorkflowFromContext();
    
    if ($workflow) {
      return $workflow;
    }
    
    // Fall back to configuration (for standalone use)
    $workflow_id = $this->configuration['workflow_id'] ?? '';
    if (!empty($workflow_id)) {
      return $this->entityTypeManager->getStorage('comfyui_workflow')->load($workflow_id);
    }

    return null;
  }

  /**
   * Get workflow options for configuration form.
   */
  protected function getWorkflowOptions() {
    $options = [];
    $workflows = $this->entityTypeManager->getStorage('comfyui_workflow')->loadMultiple();
    
    foreach ($workflows as $workflow) {
      $options[$workflow->id()] = $workflow->label();
    }
    
    return $options;
  }

  /**
   * Get field mappings for the workflow.
   */
  protected function getFieldMappings($workflow) {
    return $workflow->getFieldMappings();
  }

  /**
   * Get field definition for a field name.
   */
  protected function getFieldDefinition($field_name) {
    $field_definitions = $this->entityFieldManager->getFieldDefinitions('comfyui_workflow', 'comfyui_workflow');
    return $field_definitions[$field_name] ?? null;
  }

  /**
   * Build a media selector field using media library.
   *
   * @param string $field_name
   *   The field name.
   * @param string $label
   *   The field label.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param \Drupal\comfyui\Entity\ComfyUIWorkflow $workflow
   *   The workflow entity.
   * @param array $mapping
   *   The field mapping.
   *
   * @return array
   *   Render array for the media selector.
   */
  protected function buildMediaSelectorField($field_name, $label, $field_definition, $workflow, $mapping) {
    // Get current media entity if set
    $current_media = NULL;
    if ($workflow->hasField($field_name)) {
      $field_items = $workflow->get($field_name);
      if (!$field_items->isEmpty()) {
        $current_media = $field_items->entity;
      }
    }

    $element = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['comfyui-media-selector', 'comfyui-field-block', 'comfyui-input-field'],
        'data-comfyui-field' => $field_name,
        'data-comfyui-mapping' => json_encode($mapping),
      ],
    ];

    $element['label'] = [
      '#type' => 'html_tag',
      '#tag' => 'label',
      '#value' => $label,
      '#attributes' => ['class' => ['form-label']],
    ];

    // Show current selection with thumbnail
    if ($current_media) {
      $element['current'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['current-media-display']],
      ];

      // Try to get thumbnail
      $thumbnail_uri = NULL;
      if ($current_media->hasField('field_media_image')) {
        $image = $current_media->get('field_media_image')->entity;
        if ($image) {
          $thumbnail_uri = $image->getFileUri();
        }
      }

      if ($thumbnail_uri) {
        $element['current']['thumbnail'] = [
          '#theme' => 'image_style',
          '#style_name' => 'thumbnail',
          '#uri' => $thumbnail_uri,
          '#alt' => $current_media->label(),
          '#attributes' => ['class' => ['media-thumbnail']],
        ];
      }

      $element['current']['info'] = [
        '#markup' => '<div class="media-info"><strong>' . $current_media->label() . '</strong><br><small>ID: ' . $current_media->id() . '</small></div>',
      ];
    } else {
      $element['empty'] = [
        '#markup' => '<div class="no-media-selected">' . $this->t('No media selected') . '</div>',
      ];
    }

    // Hidden input to store media ID
    $element['media_id'] = [
      '#type' => 'hidden',
      '#value' => $current_media ? $current_media->id() : '',
      '#attributes' => [
        'data-comfyui-field' => $field_name,
        'class' => ['comfyui-media-id-input'],
      ] + ($this->configuration['field_required'] ? ['required' => 'required'] : []),
    ];

    // Media library select button
    $allowed_bundles = $field_definition->getSetting('handler_settings')['target_bundles'] ?? [];
    
    $element['select_button'] = [
      '#type' => 'button',
      '#value' => $current_media ? $this->t('Change Media') : $this->t('Select Media'),
      '#attributes' => [
        'class' => ['button', 'comfyui-media-select-btn'],
        'data-field-name' => $field_name,
        'data-allowed-types' => json_encode($allowed_bundles ?: ['comfyui_input', 'comfyui_generated']),
      ],
    ];

    // Upload new button
    $element['upload_button'] = [
      '#type' => 'button',
      '#value' => $this->t('Upload New'),
      '#attributes' => [
        'class' => ['button', 'button--secondary', 'comfyui-media-upload-btn'],
        'data-field-name' => $field_name,
      ],
    ];

    // Attach media library JavaScript and attach auto-refresh behavior
    $element['#attached']['library'][] = 'media_library/widget';
    $element['#attached']['library'][] = 'comfyui/media_selector';

    // Get max upload size from field settings
    $max_filesize = 10; // Default 10MB
    $handler_settings = $field_definition->getSetting('handler_settings') ?? [];
    $target_bundles = $handler_settings['target_bundles'] ?? ['comfyui_input'];

    // Get the first bundle's image field config
    foreach ($target_bundles as $bundle) {
      $media_field_defs = \Drupal::service('entity_field.manager')
        ->getFieldDefinitions('media', $bundle);
      
      if (isset($media_field_defs['field_media_image'])) {
        $image_config = $media_field_defs['field_media_image'];
        $image_settings = $image_config->getSettings();
        
        if (isset($image_settings['max_filesize'])) {
          // Parse "10 MB" -> 10
          if (preg_match('/(\d+)\s*MB/i', $image_settings['max_filesize'], $matches)) {
            $max_filesize = (int) $matches;
          }
        }
      }
      break; // Use first bundle only
    }

    // Add JavaScript behavior to update preview when media is selected
    $element['#attached']['drupalSettings']['comfyui_media_fields'][$field_name] = [
      'field_name' => $field_name,
      'current_media_id' => $current_media ? $current_media->id() : NULL,
      'max_filesize' => $max_filesize,
    ];

    return $element;
  }

  /**
   * Build a bypass control field (Boolean checkbox).
   *
   * @param string $field_name
   *   The field name.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param \Drupal\comfyui\Entity\ComfyUIWorkflow $workflow
   *   The workflow entity.
   * @param array $bypass_mappings
   *   The bypass mappings configuration.
   *
   * @return array
   *   Render array for the bypass control field.
   */
  protected function buildBypassControlField($field_name, $field_definition, $workflow, $bypass_mappings) {
    // Check if we're in layout builder mode
    $route_name = \Drupal::routeMatch()->getRouteName();
    $is_layout_builder = strpos($route_name, 'layout_builder') !== FALSE;
    
    if ($is_layout_builder) {
      return $this->buildLayoutBuilderPreview($field_name, $field_definition->getLabel(), $field_definition, [
        'direction' => 'control',
        'type' => 'bypass',
      ]);
    }
    
    // Get current value
    $current_value = TRUE; // Default enabled
    if ($workflow->hasField($field_name)) {
      $field_items = $workflow->get($field_name);
      if (!$field_items->isEmpty()) {
        $current_value = (bool) $field_items->first()->value;
      }
    }
    
    // Build description based on what this controls
    $description = '';
    if (!empty($bypass_mappings['group_bypasses'][$field_name])) {
      $group_name = $bypass_mappings['group_bypasses'][$field_name];
      $description = $this->t('Controls bypass for group: @group', ['@group' => $group_name]);
    }
    if (!empty($bypass_mappings['node_bypasses'][$field_name])) {
      $node_ids = $bypass_mappings['node_bypasses'][$field_name];
      $node_ids_str = is_array($node_ids) ? implode(', ', $node_ids) : (string) $node_ids;
      $description .= ($description ? ' | ' : '') . $this->t('Controls bypass for nodes: @nodes', ['@nodes' => $node_ids_str]);
    }
    
    // Use custom label if set, otherwise use field label
    $label = $this->configuration['field_label'] ?: $field_definition->getLabel();
    
    $element = [
      '#type' => 'checkbox',
      '#title' => $label,
      '#description' => $description,
      '#default_value' => $current_value,
      '#required' => $this->configuration['field_required'] ?? FALSE,
      '#attributes' => [
        'data-comfyui-field' => $field_name,
        'data-comfyui-bypass-control' => 'true',
        'form' => 'comfyui-workflow-form',
        'class' => ['comfyui-bypass-control'],
      ],
      '#wrapper_attributes' => [
        'class' => ['comfyui-field-block', 'comfyui-bypass-field'],
      ],
      '#attached' => [
        'library' => ['comfyui/workflow_form'],
      ],
    ];

    return $element;
  }

}