<?php

namespace Drupal\comfyui\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\comfyui\Service\ComfyUIOutputTypeManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the workflow form container with submit button.
 *
 * @Block(
 *   id = "comfyui_form_container",
 *   admin_label = @Translation("ComfyUI Form Container"),
 *   category = @Translation("ComfyUI Integration")
 * )
 */
class ComfyUIFormContainerBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The output type manager.
   *
   * @var \Drupal\comfyui\Service\ComfyUIOutputTypeManager
   */
  protected $outputTypeManager;

  /**
   * Constructs a new ComfyUIFormContainerBlock instance.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, FormBuilderInterface $form_builder, ComfyUIOutputTypeManager $output_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->formBuilder = $form_builder;
    $this->outputTypeManager = $output_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('form_builder'),
      $container->get('comfyui.output_type_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $route_match = \Drupal::routeMatch();
    $workflow = $route_match->getParameter('comfyui_workflow');
    
    if (!$workflow) {
      return [
        '#markup' => $this->t('No workflow context available.'),
      ];
    }

    return [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'comfyui-workflow-form-container',
        'data-workflow-id' => $workflow->id(),
        'class' => ['comfyui-form-container'],
      ],
      'submit' => [
        '#type' => 'button',
        '#value' => $this->t('Execute Workflow'),
        '#attributes' => [
          'class' => ['btn', 'btn-primary', 'comfyui-submit-btn'],
          'data-comfyui-submit' => 'true',
        ],
      ],
      'results' => [
        '#type' => 'container',
        '#attributes' => [
          'id' => 'comfyui-results-container',
          'class' => ['comfyui-results'],
        ],
        '#markup' => '<div class="comfyui-results-placeholder">Results will appear here after workflow execution.</div>',
      ],
      '#attached' => [
        'library' => ['comfyui/workflow_form'],
        'drupalSettings' => [
          'comfyui' => [
            'auto_save_enabled' => \Drupal::config('comfyui.settings')->get('auto_save_generated_images') ?? TRUE,
          ],
        ],
      ],
    ];
  }

  /**
   * Display universal results for different output types.
   * (This will be called by JavaScript AJAX to populate results)
   */
  public function displayUniversalResults($results, $workflow_form) {
    $build = [];
    
    foreach ($results as $field_name => $result_data) {
      $field_definition = $this->getFieldDefinition($field_name, $workflow_form);
      if (!$field_definition) {
        continue;
      }
      
      $output_type = $this->outputTypeManager->detectOutputType($result_data, $field_definition);
      $handler = $this->outputTypeManager->getOutputHandler($output_type);
      
      if (!$handler) {
        continue;
      }
      
      $processed = $handler->processForPreview($result_data, $field_definition);
      
      $build[$field_name] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['comfyui-result-item', "result-type-{$processed['type']}"],
          'data-field-name' => $field_name,
        ],
        'label' => [
          '#markup' => '<h4>' . $field_definition->getLabel() . '</h4>',
        ],
        'preview' => $this->buildPreviewElement($processed),
      ];
      
      if (!empty($processed['can_save'])) {
        $build[$field_name]['actions'] = $this->buildSaveActions($field_name, $result_data, $processed);
      }
    }
    
    return $build;
  }

  /**
   * Build preview element based on processed data.
   */
  protected function buildPreviewElement($processed) {
    switch ($processed['type']) {
      case 'image_preview':
        return [
          '#theme' => 'image',
          '#uri' => $processed['preview_url'],
          '#alt' => 'Generated Image Preview',
          '#attributes' => ['class' => ['comfyui-preview-image']],
        ];
        
      case 'video_preview':
        return [
          '#markup' => '<video controls style="max-width: 400px;"><source src="' . $processed['preview_url'] . '" type="video/mp4">Your browser does not support video.</video>',
        ];
        
      case 'text_preview':
        return [
          '#markup' => '<div class="comfyui-text-preview">' . htmlspecialchars($processed['preview_text']) . '</div>',
        ];
        
      case 'error':
        return [
          '#markup' => '<div class="comfyui-error">' . $processed['message'] . '</div>',
        ];
        
      default:
        return [
          '#markup' => '<div class="comfyui-unknown-preview">Unknown output type</div>',
        ];
    }
  }

  /**
   * Build save action buttons.
   */
  protected function buildSaveActions($field_name, $result_data, $processed) {
    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['comfyui-result-actions']],
      'save_button' => [
        '#type' => 'button',
        '#value' => $processed['save_label'] ?? $this->t('Save Result'),
        '#attributes' => [
          'class' => ['comfyui-save-result', 'btn', 'btn-success'],
          'data-field-name' => $field_name,
          'data-result-data' => json_encode($result_data),
        ],
      ],
      'discard_button' => [
        '#type' => 'button',
        '#value' => $this->t('Discard'),
        '#attributes' => [
          'class' => ['comfyui-discard-result', 'btn', 'btn-secondary'],
          'data-field-name' => $field_name,
        ],
      ],
    ];
  }

  /**
   * Get field definition by field name.
   */
  protected function getFieldDefinition($field_name, $workflow) {
    $field_definitions = \Drupal::service('entity_field.manager')
      ->getFieldDefinitions($workflow->getEntityTypeId(), $workflow->bundle());
    
    return $field_definitions[$field_name] ?? NULL;
  }

}
