<?php

namespace Drupal\comfyui\Plugin\ComfyUIOutputHandler;

use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Handles image file outputs from ComfyUI.
 *
 * @ComfyUIOutputHandler(
 *   id = "image_file_handler",
 *   label = @Translation("ComfyUI Image File Handler"),
 *   output_types = {"image_file"}
 * )
 */
class ComfyUIImageFileHandler extends PluginBase implements ComfyUIOutputHandlerInterface {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function processForPreview($output_data, $field_definition) {
    if (!isset($output_data['filename'])) {
      return [
        'type' => 'error',
        'message' => 'No filename in image output data',
      ];
    }

    return [
      'type' => 'image_preview',
      'preview_url' => "/comfyui/image-proxy/{$output_data['filename']}",
      'metadata' => $output_data,
      'can_save' => TRUE,
      'save_label' => $this->t('Save Image'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function processForSave($output_data, $field_definition, $save_options = []) {
      $field_type = $field_definition->getType();
      
      // Generate proxy URL for ComfyUI images
      $proxy_url = "/comfyui/image-proxy/{$output_data['filename']}";
      
      switch ($field_type) {
        case 'link':
          return [
            'uri' => $proxy_url,
            'title' => 'Generated Image',
          ];
          
        case 'string':
        case 'string_long':
        case 'text':
        case 'text_long':
          return [
            'value' => $proxy_url,
          ];
          
        default:
          return NULL;
      }
  }

  /**
   * {@inheritdoc}
   */
  public function getRecommendedFormatters() {
    return [
      'link' => ['comfyui_link_image'],
      'string' => ['basic_string'],
    ];
  }

}
